/* KinokoDataProfiler.cc */
/* Created by Enomoto Sanshiro on 23 October 1999. */
/* Last updated by Enomoto Sanshiro on 3 December 2000. */


#include <iostream>
#include <strstream>
#include <cmath>
#include <map>
#include "KinokoDataProfiler.hh"

using namespace std;

#define sqr(x) ((x) * (x))



TKinokoDataProfiler::TKinokoDataProfiler(void)
{
    _NumberOfSections = 0;
}

TKinokoDataProfiler::~TKinokoDataProfiler()
{
    map<int, TKinokoSectionDataSummary*>::iterator i;
    for (
	i = _SectionSummaryTable.begin();
	i != _SectionSummaryTable.end();
	i++
    ){
	delete (*i).second;
    }
}

int TKinokoDataProfiler::ProcessIndexedData(void* DataPacket, TKinokoIndexedDataSection* DataSection) throw(TKinokoException)
{
    string SectionName = DataSection->FullSectionName();
    int SectionId = DataSection->SectionId();

    if (_SectionSummaryTable.count(SectionId) == 0) {
	_SectionSummaryTable[SectionId] 
	    = new TKinokoSectionDataSummary(SectionName);

	_NumberOfSections++;
    }
    TKinokoSectionDataSummary* Summary = _SectionSummaryTable[SectionId];

    TKinokoIndexedDataSectionScanner* Scanner = DataSection->Scanner();
    int NumberOfElements = Scanner->NumberOfElements(DataPacket);
    int Address, Data;
    for (int Index = 0; Index < NumberOfElements; Index++) {
	Scanner->ReadFrom(DataPacket, Index, Address, Data);
	Summary->ProcessElementData(Address, Data);
    }

    return 1;
}

int TKinokoDataProfiler::ProcessTaggedData(void* DataPacket, TKinokoTaggedDataSection* DataSection) throw(TKinokoException)
{
    string SectionName = DataSection->FullSectionName();
    int SectionId = DataSection->SectionId();

    if (_SectionSummaryTable.count(SectionId) == 0) {
	_SectionSummaryTable[SectionId] 
	    = new TKinokoSectionDataSummary(SectionName);

	_NumberOfSections++;
    }
    TKinokoSectionDataSummary* Summary = _SectionSummaryTable[SectionId];

    TKinokoTaggedDataSectionScanner* Scanner = DataSection->Scanner();
    int NumberOfFields = DataSection->NumberOfFields();
    int Data;
    for (int Index = 0; Index < NumberOfFields; Index++) {
	Scanner->ReadFrom(DataPacket, Index, Data);
	Summary->ProcessElementData(
	    Index, Data, DataSection->FieldNameOf(Index)
	);
    }

    return 1;
}

int TKinokoDataProfiler::NumberOfSections(void) const
{
    return _NumberOfSections;
}

TKinokoSectionDataSummary* TKinokoDataProfiler::SectionDataSummary(int SectionIndex)
{
    if ((SectionIndex < 0) || (SectionIndex >= _NumberOfSections)) {
	return 0;
    }

    map<int, TKinokoSectionDataSummary*>::iterator SectionSummaryEntry;
    SectionSummaryEntry = _SectionSummaryTable.begin();
    for (int i = 0; i < SectionIndex; i++) {
	SectionSummaryEntry++;
    }

    return (*SectionSummaryEntry).second;
}

void TKinokoDataProfiler::WriteSummaryTo(ostream& os)
{
    TKinokoSectionDataSummary* SectionDataSummary;
    TKinokoElementDataSummary* ElementSummary;

    int NumberOfSections = this->NumberOfSections();
    for (int i = 0; i < NumberOfSections; i++) {
	SectionDataSummary = this->SectionDataSummary(i);

	os << "Section " << SectionDataSummary->SectionName() << ":" << endl;

	int NumberOfChannels = SectionDataSummary->NumberOfChannels();
	for (int Channel = 0; Channel < NumberOfChannels; Channel++) {
	    ElementSummary = SectionDataSummary->ElementDataSummary(Channel);
	    if (ElementSummary == 0) {
		continue;
	    }

	    string ElementName = ElementSummary->ElementName();

	    os << "  Channel " << ElementName << ":" << endl;
	    os << "      Number of Entries: ";
	    if (ElementSummary == 0) {
		 os << 0 << endl;
		 continue;
	    }
	    else {
		 os << ElementSummary->NumberOfEntries() << endl;
	    }

	    os << "      Total: ";
	    os << ElementSummary->Total() << endl;
	    os << "      Minimum: ";
	    os << ElementSummary->MinimumValue() << endl;
	    os << "      Maximum: ";
	    os << ElementSummary->MaximumValue() << endl;
	    os << "      Mean: ";
	    os << ElementSummary->Mean() << endl;
	    os << "      Deviation: ";
	    os << ElementSummary->Deviation() << endl;
	}
	os << endl;
    }
}



TKinokoSectionDataSummary::TKinokoSectionDataSummary(const string& SectionName)
{
    _SectionName = SectionName;
    _NumberOfChannels = 0;
}

TKinokoSectionDataSummary::~TKinokoSectionDataSummary()
{
    map<int, TKinokoElementDataSummary*>::iterator i;
    for (
	i = _ElementSummaryTable.begin();
	i != _ElementSummaryTable.end();
	i++
    ){
	delete (*i).second;
    }
}

int TKinokoSectionDataSummary::ProcessElementData(int Address, int Data, const string& FieldName)
{
    if (_ElementSummaryTable.count(Address) == 0) {
	string Name;
	if (! FieldName.empty()) {
	    Name = "\"" + FieldName + "\"";
	}
	else {
	    char NameBuffer[32];
	    ostrstream NameBufferStream(NameBuffer, sizeof(NameBuffer));
	    NameBufferStream << Address << ends;
	    Name = NameBuffer;
	}

	_ElementSummaryTable[Address] = new TKinokoElementDataSummary(Name);

	if (_NumberOfChannels < Address + 1) {
	    _NumberOfChannels = Address + 1;
	}
    }

    _ElementSummaryTable[Address]->ProcessElementData(Data);

    return 1;
}

string TKinokoSectionDataSummary::SectionName(void) const
{
    return _SectionName;
}

int TKinokoSectionDataSummary::NumberOfChannels(void) const
{
    return _NumberOfChannels;
}

TKinokoElementDataSummary* TKinokoSectionDataSummary::ElementDataSummary(int Channel)
{
    if (_ElementSummaryTable.count(Channel)) {
	return _ElementSummaryTable[Channel];
    }
    else {
	return 0;
    }
}



TKinokoElementDataSummary::TKinokoElementDataSummary(const string& ElementName)
{
    _ElementName = ElementName;

    _NumberOfEntries = 0;
    _MinimumValue = 0;
    _MaximumValue = 0;
    _TotalValue = 0;
    _TotalSquaredValue = 0;
}

TKinokoElementDataSummary::~TKinokoElementDataSummary()
{
}

int TKinokoElementDataSummary::ProcessElementData(int Data)
{
    _NumberOfEntries += 1;
    _TotalValue += Data;
    _TotalSquaredValue += sqr(Data);

    if (_NumberOfEntries == 1) {
	_MinimumValue = Data;
	_MaximumValue = Data;
    }
    else {
	if (Data < _MinimumValue) {
	    _MinimumValue = Data;
	}
	if (Data > _MaximumValue) {
	    _MaximumValue = Data;
	}
    }

    return 1;
}

const string& TKinokoElementDataSummary::ElementName(void) const
{
    return _ElementName;
}

int TKinokoElementDataSummary::NumberOfEntries(void) const
{
    return _NumberOfEntries;
}

int TKinokoElementDataSummary::MinimumValue(void) const
{
    return _MinimumValue;
}

int TKinokoElementDataSummary::MaximumValue(void) const
{
    return _MaximumValue;
}

long TKinokoElementDataSummary::Total(void) const
{
    return _TotalValue;
}

double TKinokoElementDataSummary::Mean(void) const
{
    return (double) _TotalValue / _NumberOfEntries;
}

double TKinokoElementDataSummary::Deviation(void) const
{
    double SquaredMean = sqr(Mean());
    double MeanOfSquared = _TotalSquaredValue / _NumberOfEntries;
    return sqrt(MeanOfSquared - SquaredMean);
}
