/* KinokoSectionDataAnalyzer.cc */
/* Created by Enomoto Sanshiro on 2 September 2001. */
/* Last updated by Enomoto Sanshiro on 2 September 2001 */


#include <strstream>
#include <string>
#include "KinokoDataSource.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoSectionDataAnalyzer.hh"

using namespace std;


TKinokoSectionDataAnalyzer::TKinokoSectionDataAnalyzer(const string& SectionPath)
{
    if (! SectionPath.empty()) {
	_SectionPathList.push_back(SectionPath);
    }
}

TKinokoSectionDataAnalyzer::~TKinokoSectionDataAnalyzer()
{
    for (unsigned i = 0; i < _SectionDataList.size(); i++) {
	delete _SectionDataList[i];
    }
}

int TKinokoSectionDataAnalyzer::AddSection(const string& SectionPath)
{
    int SectionIndex = _SectionPathList.size();
    _SectionPathList.push_back(SectionPath);

    return SectionIndex;
}

void TKinokoSectionDataAnalyzer::ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException)
{
    for (unsigned i = 0; i < _SectionPathList.size(); i++) {
	string SectionPath = _SectionPathList[i];
	string SectionName = TKinokoDataSection::SectionNameOf(SectionPath);

	TKinokoDataSection* DataSection = DataSource->DataSection(SectionName);
	if (DataSection == 0) {
	    // section not found  //
	    continue;
	}

	TKinokoSectionData* SectionData = (
	    TKinokoSectionData::CreateSectionDataFor(SectionPath, DataSection)
	);
	if (SectionData != 0) {
	    SectionData->SetSectionIndex(i);
	    _SectionDataList.push_back(SectionData);
	}
    }
}

int TKinokoSectionDataAnalyzer::ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    int Result = 0;
    for (unsigned i = 0; i < _SectionDataList.size(); i++) {
	if (_SectionDataList[i]->ProcessDataPacket(DataPacket)) {
	    Result += ProcessData(_SectionDataList[i]);
	}
    }

    return Result;
}

int TKinokoSectionDataAnalyzer::ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource) throw(TKinokoException)
{
    return ProcessTrailer(TKinokoDataStreamScanner::TrailerValueOf(Packet));
}

void TKinokoSectionDataAnalyzer::Dump(ostream& os, const string& Indent)
{
    os << Indent << "UNKNOWN" << endl;
}



TKinokoSectionData::TKinokoSectionData(TKinokoDataSection* DataSection)
{
    _DataSection = DataSection;
    _SectionIndex = 0;
}

TKinokoSectionData::~TKinokoSectionData()
{
}

void TKinokoSectionData::SetSectionIndex(int SectionIndex)
{
    _SectionIndex = SectionIndex;
}

TKinokoSectionData* TKinokoSectionData::CreateSectionDataFor(const string& SectionPath, TKinokoDataSection* DataSection)
{
    if (DataSection == 0) {
	return 0;
    }

    TKinokoSectionData* SectionData = 0;

    switch (DataSection->SectionType()) {
    case TKinokoDataSection::SectionType_Indexed:
	SectionData = new TKinokoIndexedSectionData(SectionPath, DataSection);
	break;
    case TKinokoDataSection::SectionType_Tagged:
	SectionData = new TKinokoTaggedSectionData(SectionPath, DataSection);
	break;
    case TKinokoDataSection::SectionType_Block:
	SectionData = new TKinokoBlockSectionData(SectionPath, DataSection);
	break;
    case TKinokoDataSection::SectionType_Nested:
	SectionData = new TKinokoNestedSectionData(SectionPath, DataSection);
	break;
    default:
	SectionData = 0;
	break;
    }

    return SectionData;
}

int TKinokoSectionData::IndexOf(const string& FieldName)
{
    return -1;
}

void TKinokoSectionData::InitializeElementSpecifier(TKinokoDataElementSpecifier& ElementSpecifier)
{
    if (! ElementSpecifier.IsInitialized()) {
	int SpecifiedAddress = IndexOf(ElementSpecifier.FieldName());
	if (SpecifiedAddress < 0) {
	    SpecifiedAddress = TKinokoDataElementSpecifier::Address_None;
	}
	ElementSpecifier.SetAddress(SpecifiedAddress);
    }
}

bool TKinokoSectionData::GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, int& Address, int& Data)
{
    if (! ElementSpecifier.IsInitialized()) {
	InitializeElementSpecifier(ElementSpecifier);
    }

    Address = ElementSpecifier.Address();
    if (Address == TKinokoDataElementSpecifier::Address_None) {
	return false;
    }
    else if (Address == TKinokoDataElementSpecifier::Address_All) {
	return GetNext(Address, Data);
    }
    else {
	return GetNextAt(Address, Data);
    }
}

bool TKinokoSectionData::GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, string& FieldName, int& Data)
{
    if (! ElementSpecifier.IsInitialized()) {
	InitializeElementSpecifier(ElementSpecifier);
    }

    int Address = ElementSpecifier.Address();
    if (Address == TKinokoDataElementSpecifier::Address_None) {
	return false;
    }
    else if (Address == TKinokoDataElementSpecifier::Address_All) {
	return GetNext(FieldName, Data);
    }
    else {
	FieldName = ElementSpecifier.FieldName();
	return GetNextAt(Address, Data);
    }
}



TKinokoIndexedSectionData::TKinokoIndexedSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _IndexedDataSection = (TKinokoIndexedDataSection*) DataSection;
    _Scanner = _IndexedDataSection->Scanner();
    _SectionId = _IndexedDataSection->SectionId();

    _NumberOfElements = 0;
    _CurrentIndex = 0;
}

TKinokoIndexedSectionData::~TKinokoIndexedSectionData()
{
}

bool TKinokoIndexedSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	_NumberOfElements = 0;
	return false;
    }

    _DataPacket = DataPacket;
    _CurrentIndex = 0;
    _NumberOfElements = _Scanner->NumberOfElements(_DataPacket);

    return true;
}

bool TKinokoIndexedSectionData::GetNext(int& Address, int& Data)
{
    if (_CurrentIndex >= _NumberOfElements) {
	return false;
    }
    
    _Scanner->ReadFrom(_DataPacket, _CurrentIndex, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoIndexedSectionData::GetNext(string& FieldName, int& Data)
{
    int Address;
    bool Result = GetNext(Address, Data);

    if (Result) {
	char FieldNameBuffer[16];
	ostrstream FieldNameStream(FieldNameBuffer, sizeof(FieldNameBuffer));
	FieldNameStream << Address << ends;
	FieldName = FieldNameBuffer;
    }

    return Result;
}

bool TKinokoIndexedSectionData::GetNextAt(int Address, int& Data)
{
    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	if (_CurrentIndex < _NumberOfElements) {
	    Data = _NumberOfElements;
	    _CurrentIndex = _NumberOfElements;
	    return true;
	}
	else {
	    return false;
	}
    }

    int ThisAddress;
    while (GetNext(ThisAddress, Data)) {
	if (ThisAddress == Address) {
	    return true;
	}
    }

    return false;
}

void* TKinokoIndexedSectionData::GetDataBlock(void)
{
    return TKinokoDataSectionScanner::DataAreaOf(_DataPacket);
}

size_t TKinokoIndexedSectionData::DataBlockSize(void)
{
    return TKinokoDataSectionScanner::DataSizeOf(_DataPacket);
}



TKinokoTaggedSectionData::TKinokoTaggedSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _TaggedDataSection = (TKinokoTaggedDataSection*) DataSection;
    _Scanner = _TaggedDataSection->Scanner();
    _SectionId = _TaggedDataSection->SectionId();
    _NumberOfFields = _TaggedDataSection->NumberOfFields();
    
    _IsDataAvailable = false;
}

TKinokoTaggedSectionData::~TKinokoTaggedSectionData()
{
}

bool TKinokoTaggedSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	return _IsDataAvailable = false;
    }

    _DataPacket = DataPacket;
    _CurrentIndex = 0;

    return _IsDataAvailable = true;
}

int TKinokoTaggedSectionData::IndexOf(const string& FieldName)
{
    int Index;
    try {
	Index = _TaggedDataSection->FieldIndexOf(FieldName);
    }
    catch (TKinokoException &e) {
	Index = -1;
    }

    return Index;
}

bool TKinokoTaggedSectionData::GetNext(int& Address, int& Data)
{
    if (! _IsDataAvailable || (_CurrentIndex >= _NumberOfFields)) {
	return false;
    }
    
    Address = _CurrentIndex;
    _Scanner->ReadFrom(_DataPacket, Address, Data);
    _CurrentIndex++;

    return true;
}

bool TKinokoTaggedSectionData::GetNext(string& FieldName, int& Data)
{
    int Address;
    int Result = GetNext(Address, Data);

    if (Result) {
	FieldName = _TaggedDataSection->FieldNameOf(Address);
    }

    return Result;
}

bool TKinokoTaggedSectionData::GetNextAt(int Address, int& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	Data = _NumberOfFields;
    }
    else {
	_Scanner->ReadFrom(_DataPacket, Address, Data);
    }

    _IsDataAvailable = false;

    return true;
}

void* TKinokoTaggedSectionData::GetDataBlock(void)
{
    return TKinokoDataSectionScanner::DataAreaOf(_DataPacket);
}

size_t TKinokoTaggedSectionData::DataBlockSize(void)
{
    return TKinokoDataSectionScanner::DataSizeOf(_DataPacket);
}



TKinokoBlockSectionData::TKinokoBlockSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _BlockDataSection = (TKinokoBlockDataSection*) DataSection;
    _Scanner = _BlockDataSection->Scanner();
    _SectionId = _BlockDataSection->SectionId();

    _DataBlock = 0;
}

TKinokoBlockSectionData::~TKinokoBlockSectionData()
{
}

bool TKinokoBlockSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	_DataBlock = 0;
	return false;
    }

    _DataBlock = TKinokoDataSectionScanner::DataAreaOf(DataPacket);
    _DataSize = TKinokoDataSectionScanner::DataSizeOf(DataPacket);
    _CurrentAddress = 0;
	
    return true;
}

bool TKinokoBlockSectionData::GetNext(int& Address, int& Data)
{
    if ((_DataBlock == 0) || (_CurrentAddress >= _DataSize)) {
	return false;
    }

    Address = _CurrentAddress;
    Data = ((U32bit*) _DataBlock)[Address / sizeof(U32bit)];

    _CurrentAddress += sizeof(U32bit);

    return true;
}

bool TKinokoBlockSectionData::GetNext(string& FieldName, int& Data)
{
    int Address;
    bool Result = GetNext(Address, Data);
    
    if (Result) {
	static char FieldNameBuffer[32];
	ostrstream FieldNameStream(FieldNameBuffer, sizeof(FieldNameBuffer));
	FieldNameStream << Address << ends;
	FieldName = FieldNameBuffer;
    }

    return Result;
}

bool TKinokoBlockSectionData::GetNextAt(int Address, int& Data)
{
    if (_DataBlock == 0) {
	return false;
    }

    if (Address == TKinokoDataElementSpecifier::Address_DataSize) {
	Data = _DataSize;
    }
    else if (Address <= _DataSize) {
	Data = ((U32bit*) _DataBlock)[Address / sizeof(U32bit)];
    }
    else {
	return false;
    }
    
    _DataBlock = 0;

    return true;
}

void* TKinokoBlockSectionData::GetDataBlock(void)
{
    return _DataBlock;
}

size_t TKinokoBlockSectionData::DataBlockSize(void)
{
    return _DataSize;
}



TKinokoNestedSectionData::TKinokoNestedSectionData(const string& SectionPath, TKinokoDataSection* DataSection)
: TKinokoSectionData(DataSection)
{
    _SectionPath = SectionPath;
    _NestedDataSection = (TKinokoNestedDataSection*) DataSection;
    _Scanner = _NestedDataSection->Scanner();
    _SectionId = _NestedDataSection->SectionId();
    _IsDataAvailable = false;

    string SubSectionPath = TKinokoDataSection::SubSectionPathOf(SectionPath);
    string SubSectionName = TKinokoDataSection::SectionNameOf(SubSectionPath);

    TKinokoDataSection* SubSection = _NestedDataSection->DataSection(SubSectionName);
    if (SubSection != 0) {
	_SubSectionData = CreateSectionDataFor(SubSectionPath, SubSection);
    }
    else {
	_SubSectionData = 0;
    }
}

TKinokoNestedSectionData::~TKinokoNestedSectionData()
{
    delete _SubSectionData;
}

bool TKinokoNestedSectionData::ProcessDataPacket(void* DataPacket) throw(TKinokoException)
{
    if (_SubSectionData == 0) {
	return false;
    }

    if (_Scanner->SectionIdOf(DataPacket) != _SectionId) {
	_IsDataAvailable = false;
	return false;
    }

    int DataSize = _Scanner->DataSizeOf(DataPacket);
    U8bit* SubPacket = (U8bit*) _Scanner->DataAreaOf(DataPacket);

    int SubPacketSize;
    int ProcessedSize = 0;
    while (ProcessedSize < DataSize) {
	_IsDataAvailable = _SubSectionData->ProcessDataPacket(SubPacket);
	if (_IsDataAvailable) {
	    break;
	}

	SubPacketSize = TKinokoDataSectionScanner::PacketSizeOf(SubPacket);
	ProcessedSize += SubPacketSize;
	SubPacket += SubPacketSize;
    }

    return _IsDataAvailable;
}

bool TKinokoNestedSectionData::GetNext(int& Address, int& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNext(Address, Data);
}

bool TKinokoNestedSectionData::GetNext(string& FieldName, int& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNext(FieldName, Data);
}

bool TKinokoNestedSectionData::GetNextAt(int Address, int& Data)
{
    if (! _IsDataAvailable) {
	return false;
    }

    return _SubSectionData->GetNextAt(Address, Data);
}

int TKinokoNestedSectionData::IndexOf(const std::string& FieldName)
{
    if (_SubSectionData) {
	return _SubSectionData->IndexOf(FieldName);
    }
    else {
	return TKinokoSectionData::IndexOf(FieldName);
    }
}

void* TKinokoNestedSectionData::GetDataBlock(void)
{
    if (! _IsDataAvailable) {
	return 0;
    }

    return _SubSectionData->GetDataBlock();
}

size_t TKinokoNestedSectionData::DataBlockSize(void)
{
    if (! _IsDataAvailable) {
	return 0;
    }

    return _SubSectionData->DataBlockSize();
}



TKinokoDataElementSpecifier::TKinokoDataElementSpecifier(void)
{
    _Address = Address_All;

    _ConversionOffset = 0;
    _ConversionFactor = 1.0;
}

TKinokoDataElementSpecifier::TKinokoDataElementSpecifier(int Address)
{
    _Address = Address;

    char FieldNameBuffer[16];
    ostrstream FieldNameStream(FieldNameBuffer, sizeof(FieldNameBuffer));
    FieldNameStream << Address << ends;
    _FieldName = FieldNameBuffer;

    _ConversionOffset = 0;
    _ConversionFactor = 1.0;
}

TKinokoDataElementSpecifier::TKinokoDataElementSpecifier(const string& FieldName)
{
    _FieldName = FieldName;

    if (_FieldName == "$data_size") {
	_Address = Address_DataSize;
    }
    else {
	_Address = Address_Unknown;
    }

    _ConversionOffset = 0;
    _ConversionFactor = 1.0;
}

TKinokoDataElementSpecifier::~TKinokoDataElementSpecifier()
{
}

void TKinokoDataElementSpecifier::SetAddress(int Address)
{
    _Address = Address;
}

string TKinokoDataElementSpecifier::AsString(void) const
{
    if (_Address == Address_Unknown) {
	return '\"' + _FieldName + '\"';
    }
    else if (_Address == Address_All) {
	return string("ALL");
    }
    else {
	char Buffer[32];
	ostrstream(Buffer, sizeof(Buffer)) << _Address << ends;
	return string(Buffer);
    }
}

void TKinokoDataElementSpecifier::SetOffset(double Offset)
{
    _ConversionOffset = Offset;
}

void TKinokoDataElementSpecifier::SetFactor(double Factor)
{
    _ConversionFactor = Factor;
}
