/* KinokoSectionDataAnalyzer.hh */
/* Created by Enomoto Sanshiro on 2 September 2001. */
/* Last updated by Enomoto Sanshiro on 2 September 2001 */


#ifndef __KinokoSectionDataAnalyzer_hh__
#define __KinokoSectionDataAnalyzer_hh__


#include <string>
#include <vector>
#include "KinokoDataSource.hh"
#include "KinokoDataAnalyzer.hh"


class TKinokoSectionData;
class TKinokoDataElementSpecifier;


class TKinokoSectionDataAnalyzer: public TKinokoDataAnalyzer {
  public:
    TKinokoSectionDataAnalyzer(const std::string& SectionPath = "");
    virtual ~TKinokoSectionDataAnalyzer();
    virtual int AddSection(const std::string& SectionPath);
    virtual void ReadDataSource(TKinokoDataSource* DataSource) throw(TKinokoException);
    virtual int ProcessDataPacket(void* DataPacket, TKinokoDataSource* DataSource = 0, TKinokoDataSection* DataSection = 0) throw(TKinokoException);
    virtual int ProcessTrailerPacket(void* Packet, TKinokoDataSource* DataSource = 0) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    virtual int ProcessData(TKinokoSectionData* SectionData) throw(TKinokoException) = 0;
    virtual int ProcessTrailer(int TrailerValue) throw(TKinokoException) { return 0; }
  protected:
    std::vector<std::string> _SectionPathList;
    std::vector<TKinokoSectionData*> _SectionDataList;
};


class TKinokoSectionData {
  public:
    TKinokoSectionData(TKinokoDataSection* DataSection);
    virtual ~TKinokoSectionData();
    void SetSectionIndex(int SectionIndex);
    inline int SectionIndex(void);
    inline int SectionId(void) const;
    inline const std::string& SectionName(void) const;
    inline const TKinokoDataSection* DataSection(void) const;
    inline TKinokoDataSection* DataSection(void);
  public:
    virtual bool ProcessDataPacket(void* DataPacket) throw(TKinokoException) = 0;
    virtual bool GetNext(int& Address, int& Data) = 0;
    virtual bool GetNext(std::string& FieldName, int& Data) = 0;
    virtual bool GetNextAt(int Address, int& Data) = 0;
    virtual bool GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, int& Address, int& Data);
    virtual bool GetNextOf(TKinokoDataElementSpecifier& ElementSpecifier, std::string& FieldName, int& Data);
    virtual void* GetDataBlock(void) = 0;
    virtual size_t DataBlockSize(void) = 0;
    virtual int IndexOf(const std::string& FieldName);
  public:
    static TKinokoSectionData* CreateSectionDataFor(const std::string& SectionPath, TKinokoDataSection* DataSection);
    void InitializeElementSpecifier(TKinokoDataElementSpecifier& ElementSpecifier);
  protected:
    int _SectionIndex;
    TKinokoDataSection* _DataSection;
};


class TKinokoIndexedSectionData: public TKinokoSectionData {
  public:
    TKinokoIndexedSectionData(const std::string& SectionPath, TKinokoDataSection* DataSection);
    virtual ~TKinokoIndexedSectionData();
    virtual bool ProcessDataPacket(void* DataPacket) throw(TKinokoException);
    virtual bool GetNext(int& Address, int& Data);
    virtual bool GetNext(std::string& FieldName, int& Data);
    virtual bool GetNextAt(int Address, int& Data);
    virtual void* GetDataBlock(void);
    virtual size_t DataBlockSize(void);
  protected:
    std::string _SectionPath;
    TKinokoIndexedDataSection* _IndexedDataSection;
    TKinokoIndexedDataSectionScanner* _Scanner;
    int _SectionId;
    void* _DataPacket;
    int _CurrentIndex;
    int _NumberOfElements;
};


class TKinokoTaggedSectionData: public TKinokoSectionData {
  public:
    TKinokoTaggedSectionData(const std::string& SectionPath, TKinokoDataSection* DataSection);
    virtual ~TKinokoTaggedSectionData();
    virtual bool ProcessDataPacket(void* DataPacket) throw(TKinokoException);
    virtual bool GetNext(int& Address, int& Data);
    virtual bool GetNext(std::string& FieldName, int& Data);
    virtual bool GetNextAt(int Address, int& Data);
    virtual int IndexOf(const std::string& FieldName);
    virtual void* GetDataBlock(void);
    virtual size_t DataBlockSize(void);
  protected:
    std::string _SectionPath;
    TKinokoTaggedDataSection* _TaggedDataSection;
    TKinokoTaggedDataSectionScanner* _Scanner;
    int _SectionId;
    void* _DataPacket;
    bool _IsDataAvailable;
    int _CurrentIndex;
    int _NumberOfFields;
};


class TKinokoBlockSectionData: public TKinokoSectionData {
  public:
    TKinokoBlockSectionData(const std::string& SectionPath, TKinokoDataSection* DataSection);
    virtual ~TKinokoBlockSectionData();
    virtual bool ProcessDataPacket(void* DataPacket) throw(TKinokoException);
    virtual bool GetNext(int& Address, int& Data);
    virtual bool GetNext(std::string& FieldName, int& Data);
    virtual bool GetNextAt(int Address, int& Data);
    virtual void* GetDataBlock(void);
    virtual size_t DataBlockSize(void);
  protected:
    std::string _SectionPath;
    TKinokoBlockDataSection* _BlockDataSection;
    TKinokoBlockDataSectionScanner* _Scanner;
    int _SectionId;
    void* _DataBlock;
    int _DataSize;
    int _CurrentAddress;
};


class TKinokoNestedSectionData: public TKinokoSectionData {
  public:
    TKinokoNestedSectionData(const std::string& SectionPath, TKinokoDataSection* DataSection);
    virtual ~TKinokoNestedSectionData();
    virtual bool ProcessDataPacket(void* DataPacket) throw(TKinokoException);
    virtual bool GetNext(int& Address, int& Data);
    virtual bool GetNext(std::string& FieldName, int& Data);
    virtual bool GetNextAt(int Address, int& Data);
    virtual int IndexOf(const std::string& FieldName);
    virtual void* GetDataBlock(void);
    virtual size_t DataBlockSize(void);
  protected:
    std::string _SectionPath;
    TKinokoNestedDataSection* _NestedDataSection;
    TKinokoSectionData* _SubSectionData;
    TKinokoNestedDataSectionScanner* _Scanner;
    int _SectionId;
    bool _IsDataAvailable;
};


class TKinokoDataElementSpecifier {
  public:
    TKinokoDataElementSpecifier(void);
    TKinokoDataElementSpecifier(int Address);
    TKinokoDataElementSpecifier(const std::string& FieldName);
    ~TKinokoDataElementSpecifier();
    void SetAddress(int Address);
    inline bool IsInitialized(void) const;
    inline int Address(void) const;
    inline const std::string& FieldName(void) const;
    std::string AsString(void) const;
  public:
    void SetOffset(double Offset);
    void SetFactor(double Factor);
    inline double ConvertData(int Data);
  public:
    enum TMetaDataAddress {
	Address_DataSize = -1,
	Address_All = -1024,
	Address_None = -1025,
	Address_Unknown = -1026
    };
  public:
    int _Address;
    std::string _FieldName;
    double _ConversionOffset, _ConversionFactor;
};



inline int TKinokoSectionData::SectionIndex(void)
{
    return _SectionIndex;
}

inline int TKinokoSectionData::SectionId(void) const
{
    return _DataSection->SectionId();
}

inline const std::string& TKinokoSectionData::SectionName(void) const
{
    return _DataSection->SectionName();
}

inline const TKinokoDataSection* TKinokoSectionData::DataSection(void) const
{
    return _DataSection;
}

inline TKinokoDataSection* TKinokoSectionData::DataSection(void)
{
    return _DataSection;
}


inline bool TKinokoDataElementSpecifier::IsInitialized(void) const
{
    return (_Address != Address_Unknown);
}

inline int TKinokoDataElementSpecifier::Address(void) const
{
    return _Address;
}

inline const std::string& TKinokoDataElementSpecifier::FieldName(void) const
{
    return _FieldName;
}

inline double TKinokoDataElementSpecifier::ConvertData(int Data)
{
    return _ConversionOffset + _ConversionFactor * (double) Data;
}


#endif
