/* KinokoArena.cc */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Last updated by Enomoto Sanshiro on 15 January 2002. */


#include <sys/types.h>
#include "KinokoArena.hh"


#ifndef __KinokoArena_cc__
#define __KinokoArena_cc__


TKinokoArenaEntry::TKinokoArenaEntry(int EntryIndex)
{
    _EntryIndex = EntryIndex;
}

TKinokoArenaEntry::~TKinokoArenaEntry()
{
}

void TKinokoArenaEntry::Attach(void* Address, size_t Capacity, int NumberOfAttached)
{
    _Address = Address;
    _Capacity = Capacity;
    _ReferenceCount = NumberOfAttached;
    _PrevEntry = 0;
    _NextEntry = 0;
}

void TKinokoArenaEntry::Detach(void) 
{
    if (--_ReferenceCount <= 0) {
	if (_PrevEntry != 0) {
	    _PrevEntry->_NextEntry = _NextEntry;
	}
	if (_NextEntry != 0) {
	    _NextEntry->_PrevEntry = _PrevEntry;
	}
    }
}

void TKinokoArenaEntry::AddNext(TKinokoArenaEntry* Entry)
{
    _NextEntry = Entry;
    Entry->_PrevEntry = this;
}

int TKinokoArenaEntry::EntryIndex(void) const 
{
    return _EntryIndex;
}

void* TKinokoArenaEntry::Address(void) const 
{
    return _Address;
}

int TKinokoArenaEntry::ReferenceCount(void) const 
{
    return _ReferenceCount;
}

size_t TKinokoArenaEntry::Capacity(void) const 
{
    return _Capacity;
}

TKinokoArenaEntry* TKinokoArenaEntry::Prev(void) const 
{
    return _PrevEntry;
}

TKinokoArenaEntry* TKinokoArenaEntry::Next(void) const 
{
    return _NextEntry;
}    



TKinokoArenaEntryPool::TKinokoArenaEntryPool(int MaxNumberOfEntries)
{
    _EntryTableSize = MaxNumberOfEntries;
    
    _EntryTable = new TKinokoArenaEntry* [_EntryTableSize];
    _OccupancyTable = new bool[_EntryTableSize];

    for (int i = 0; i < _EntryTableSize; i++) {
	_EntryTable[i] = new TKinokoArenaEntry(i);
	_OccupancyTable[i] = false;
    }

    _LastEntryIndex = -1;
}

TKinokoArenaEntryPool::~TKinokoArenaEntryPool()
{
    for (int i = 0; i < _EntryTableSize; i++) {
	delete _EntryTable[i];
    }

    delete[] _OccupancyTable;
    delete[] _EntryTable;
}

TKinokoArenaEntry* TKinokoArenaEntryPool::AllocateEntry(void)
{
    int Offset = 0;
    while (_OccupancyTable[(_LastEntryIndex + (++Offset)) % _EntryTableSize]) {
	if (Offset >= _EntryTableSize) {
#if 0
	    throw TKinokoException(
		"TKinokoArenaEntryPool::AllocateEntry()",
		"buffer entry pool is exhausted"
	    );
#else
	    return 0;
#endif
	}
    }
    _LastEntryIndex = (_LastEntryIndex + Offset) % _EntryTableSize;
    _OccupancyTable[_LastEntryIndex] = true;

    return _EntryTable[_LastEntryIndex];
}

void TKinokoArenaEntryPool::ReleaseEntry(TKinokoArenaEntry* Entry)
{
    _OccupancyTable[Entry->EntryIndex()] = false;
}



TKinokoArena::TKinokoArena(void* BaseAddress, size_t ArenaSize, int MaxNumberOfEntries)
{
    _BaseAddress = BaseAddress;
    _ArenaSize = ArenaSize;
    _MaxNumberOfEntries = MaxNumberOfEntries;

    _HeadEntry = 0;
    _TailEntry = 0;

    _EntryPool = new TKinokoArenaEntryPool(_MaxNumberOfEntries);
    
    _NumberOfEntries = 0;
    _UsedSize = 0;
}

TKinokoArena::~TKinokoArena()
{
    delete _EntryPool;
}

bool TKinokoArena::CreateEntry(size_t Capacity, void*& Address, int NumberOfAttached)
{
    if (NumberOfAttached == 0) {
	return false;
    }

    if (_HeadEntry == 0) {	
	Address = _BaseAddress;
    }
    else {
	Address = (unit*) _TailEntry->Address() + _TailEntry->Capacity();
	if (AvailableSize(Address) < Capacity) {
	    Address = _BaseAddress;
	}
    }

    if (AvailableSize(Address) < Capacity) {
#if 0
	throw TKinokoException(
	    "TKinokoArena::CreateEntry()", "buffer is exhausted"
	);
#else
	return false;
#endif
    }

    if (_NumberOfEntries >= _MaxNumberOfEntries) {
#if 0
	throw TKinokoException(
	    "TKinokoArena::CreateEntry()", "entry pool is exhausted"
	);
#else
	return false;
#endif
    }

    TKinokoArenaEntry* Entry = _EntryPool->AllocateEntry();
    if (Entry == 0) {
#if 0
	throw TKinokoException(
	    "TKinokoArena::CreateEntry()", "unable to create entry"
	);
#else
	return false;
#endif
    }

    Entry->Attach(Address, Capacity, NumberOfAttached);
    if (_HeadEntry == 0) {
	_HeadEntry = Entry;
    }
    else {
	_TailEntry->AddNext(Entry);
    }
    _TailEntry = Entry;

    _NumberOfEntries++;
    _UsedSize += Capacity;

    return true;
}

bool TKinokoArena::DetachEntry(void* Address)
{
    TKinokoArenaEntry* Entry;
    for (Entry = _HeadEntry; Entry != 0; Entry = Entry->Next()) {
	if (Entry->Address() == Address) {
	    break;
	}
    } 
    
    if (Entry != 0) {
	Entry->Detach();
	if (Entry->ReferenceCount() == 0) {
	    if (_HeadEntry == Entry) {
		_HeadEntry = Entry->Next();
	    }
	    if (_TailEntry == Entry) {
		_TailEntry = Entry->Prev();
	    }

	    _NumberOfEntries--;
	    _UsedSize -= Entry->Capacity();

	    _EntryPool->ReleaseEntry(Entry);
	}
    }
    else {
#if 0
	std::ostrstream Message;
	Message << "invalid address: " << Address << std::ends;
	throw TKinokoException(
	    "TKinokoArena::DetachEntry()", Message.str()
	);
#else
	return false;
#endif
    }

    return true;
}

size_t TKinokoArena::AvailableSize(void) const
{
    size_t Size;
    void* TailAddress;

    if (_HeadEntry == 0) {	
	Size = _ArenaSize;
    }
    else {
	TailAddress = (unit*) _TailEntry->Address() + _TailEntry->Capacity();
	Size = AvailableSize(_BaseAddress) + AvailableSize(TailAddress);
    }

    return Size;
}

size_t TKinokoArena::AvailableSize(const void* EntryAddress) const
{
    if (_HeadEntry == 0) {
	return _ArenaSize;
    }
    
    size_t Size;
    unit* HeadAddress = (unit*) _HeadEntry->Address();
    unit* TailAddress = (unit*) _TailEntry->Address() + _TailEntry->Capacity();
    
    if (HeadAddress < TailAddress) {
	if (EntryAddress < HeadAddress) {
	    // _____E---H...T_____    (...: Used, ___: Free, ---: ContinuousSize)
	    Size = (unit*) HeadAddress - (unit*) EntryAddress;
	}
	else if (EntryAddress < TailAddress) {
	    // _____H...E...T_____
	    Size = 0;
	}
	else {
	    // _____H...T___E-----
	    Size = _ArenaSize - ((unit*) EntryAddress - (unit*) _BaseAddress);
	}
    }
    else {
	if (EntryAddress < TailAddress) {
	    // .....E...T___H.....
	    Size = 0;
	}
	else if (EntryAddress < HeadAddress) {
	    // .....T___E---H.....
	    Size = (unit*) HeadAddress - (unit*) EntryAddress;
	}
	else {
	    // .....T___H...E.....
	    Size = 0;
	}
    }
    
    return Size;
}

int TKinokoArena::NumberOfAvailableEntries(void) const
{
    return _MaxNumberOfEntries - _NumberOfEntries;
}

int TKinokoArena::IsAllocatable(size_t Size) const
{
    if (_NumberOfEntries >= _MaxNumberOfEntries) {
	return false;
    }

    if (_HeadEntry == 0) {
	return (Size <= AvailableSize(_BaseAddress));
    }
    else {
	void* TailAddress = (unit*) _TailEntry->Address() + _TailEntry->Capacity();
	return (Size <= AvailableSize(TailAddress)) || (Size <= AvailableSize(_BaseAddress));
    }
}

int TKinokoArena::NumberOfEntries(void) const
{
    return _NumberOfEntries;
}

size_t TKinokoArena::UsedSize(void) const
{
    return _UsedSize;
}

long TKinokoArena::DataHeadOffset(void) const
{
    if (_HeadEntry == 0) {
	return 0;
    }
    else {
	return (long) _HeadEntry->Address() - (long) _BaseAddress;
    }
}

long TKinokoArena::NextStartOffset(void) const
{
    if (_HeadEntry == 0) {
	return 0;
    }
    else {
	return (long) _TailEntry->Address() + (long) _TailEntry->Capacity() - (long) _BaseAddress;
    }
}


#endif
