/* KinokoBufferClient.cc */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Last updated by Enomoto Sanshiro on 21 August 2002. */


#include <iostream>
#include <fstream>
#include "MushDefs.hh"
#include "MushIpc.hh"
#include "KinokoDefs.hh"
#include "KinokoBufferServer.hh"
#include "KinokoBufferClient.hh"

using namespace std;


TKinokoBufferClient::TKinokoBufferClient(int SharedMemoryProjectId, int MessageQueueProjectId, TMushIntervalTimer* TimeKeeper, TKinokoBufferLogger* Logger) throw(TKinokoException)
{
    _SharedMemory = 0;
    _MessageQueue = 0;

    try {
	_SharedMemory = new TMushClientSharedMemory(
	    SharedMemoryProjectId, TKinokoBufferServer::IpcProjectPath()

	);
	_MessageQueue = new TMushClientMessageQueue(
	    MessageQueueProjectId, TKinokoBufferServer::IpcProjectPath()
	);
    }
    catch (TSystemCallException &e) {
	delete _MessageQueue;
	delete _SharedMemory;
	_SharedMemory = 0;
	_MessageQueue = 0;

        throw TKinokoException(
	    "TKinokoBufferClient::TKinokoBufferClient()",
	    "system call exception: " + e.Message()
        );
    }

    _TimeKeeper = TimeKeeper;
    _Logger = Logger;

    _BaseAddress = (unsigned long) _SharedMemory->BasePointer();
}

TKinokoBufferClient::~TKinokoBufferClient()
{
    delete _MessageQueue;
    delete _SharedMemory;
}

long TKinokoBufferClient::MessageQueueAddress(void)
{
    return _MessageQueue->Address();
}



TKinokoBufferWriter::TKinokoBufferWriter(int SharedMemoryProjectId, int MessageQueueProjectId, TMushIntervalTimer* TimeKeeper, TKinokoBufferLogger* Logger) throw(TKinokoException)
:TKinokoBufferClient(SharedMemoryProjectId, MessageQueueProjectId, TimeKeeper, Logger)
{
    _NumberOfReserved = 0;
}

TKinokoBufferWriter::~TKinokoBufferWriter()
{
}

int TKinokoBufferWriter::ReserveWriteEntry(size_t Capacity)
{
    if (_NumberOfReserved > 0) {
	return _NumberOfReserved;
    }

    _Packet.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _Packet.SenderAddress = MessageQueueAddress();
    _Packet.PacketType = TKinokoBufferPacket::ptWriteRequest;
    _Packet.Size = (unsigned long) Capacity;

    //_TimeKeeper->Start();
    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    //_TimeKeeper->Stop();

    if (SentLength > 0) {
	_NumberOfReserved++;
    }

    return _NumberOfReserved;
}

bool TKinokoBufferWriter::GetWriteEntry(void*& Address, size_t Capacity)
{
    if (ReserveWriteEntry(Capacity) == 0) {
	// Timed out //
	Address = 0;
	return false;
    }

    _TimeKeeper->Start();
    int ReceivedLength = _MessageQueue->Receive(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    _TimeKeeper->Stop();

    if (ReceivedLength > 0) {
	Address = (void*) (_BaseAddress + _Packet.OffsetAddress);
	_NumberOfReserved--;
    }
    else {
	// timed out //
	Address = 0;
    }

    return (Address != 0);
}

bool TKinokoBufferWriter::SendStrobe(const void* Address, size_t Size)
{
    _Packet.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _Packet.SenderAddress = MessageQueueAddress();
    _Packet.PacketType = TKinokoBufferPacket::ptDataStrobe;
    _Packet.OffsetAddress = (unsigned long) Address - _BaseAddress;
    _Packet.Size = (unsigned long) Size;

    //_TimeKeeper->Start();
    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    //_TimeKeeper->Stop();

    if (SentLength <= 0) {
	_Logger->Write(
	    "no answer from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
	if (SentLength <= 0) {
	    _Logger->Write(
		"no answer from BufferServer (or interrupted)", 
		TKinokoBufferLogger::mlError
	    );
	}
    }

    return (SentLength > 0);
}



TKinokoBufferReader::TKinokoBufferReader(int SharedMemoryProjectId, int MessageQueueProjectId, TMushIntervalTimer* TimeKeeper, TKinokoBufferLogger* Logger) throw(TKinokoException)
: TKinokoBufferClient(SharedMemoryProjectId, MessageQueueProjectId, TimeKeeper, Logger)
{
    _IsAttached = false;
    _IsPacketDataAvailable = false;

    Attach();
}

TKinokoBufferReader::~TKinokoBufferReader()
{
    if (_IsAttached) {
	Detach();
    }
}

void TKinokoBufferReader::Attach(void)
{
    if (_IsAttached) {
	return;
    }

    _OutgoingPacket.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _OutgoingPacket.SenderAddress = MessageQueueAddress();
    _OutgoingPacket.PacketType = TKinokoBufferPacket::ptReaderAttachRequest;

    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_OutgoingPacket, sizeof(_OutgoingPacket)
    );

    if (SentLength <= 0) {
	_Logger->Write(
	    "no answer from BufferServer (or interrupted)",
	    TKinokoBufferLogger::mlError
	);
    }
    else {
	_IsAttached = true;
    }
}

void TKinokoBufferReader::Detach(void)
{
    if (! _IsAttached) {
	return;
    }

    _OutgoingPacket.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _OutgoingPacket.SenderAddress = MessageQueueAddress();
    _OutgoingPacket.PacketType = TKinokoBufferPacket::ptReaderDetachRequest;

    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_OutgoingPacket, sizeof(_OutgoingPacket)
    );

    if (SentLength <= 0) {
	_Logger->Write(
	    "no answer from BufferServer (or interrupted)",
	    TKinokoBufferLogger::mlError
	);
	return;
    }
    
    // flush remaining packets and wait for detach acknowledge //
    int NumberOfTrys = 0;
    int MaxNumberOfTrys = 128;
    while (_IsAttached) {
	_TimeKeeper->Start();
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket *) &_IncomingPacket, sizeof(_IncomingPacket)
	);
	_TimeKeeper->Stop();
	
	if (ReceivedLength > 0) {
	    switch (_IncomingPacket.PacketType) {
              case TKinokoBufferPacket::ptReadRequest:
		SendAcknowledge(
		    (void*) (_BaseAddress + _IncomingPacket.OffsetAddress)
		);
		break;
	      case TKinokoBufferPacket::ptDetachAcknowledge:
		_IsAttached = false;
		break;
	      default:
		break;
	    }
	}
	else if (++NumberOfTrys > MaxNumberOfTrys) {
	    _Logger->Write(
		"no answer from BufferServer for detach request",
		TKinokoBufferLogger::mlWarning
	    );
	}
    }
}

bool TKinokoBufferReader::HasData(void)
{
    if (! _IsPacketDataAvailable) {
	static const bool NoWait = true;
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_IncomingPacket, 
	    sizeof(_IncomingPacket), 
	    NoWait
	);

	_IsPacketDataAvailable = (ReceivedLength > 0);
    }

    return _IsPacketDataAvailable;
}

int TKinokoBufferReader::NextDataSize(void)
{
    if (! _IsPacketDataAvailable) {
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_IncomingPacket, 
	    sizeof(_IncomingPacket)
	);

	_IsPacketDataAvailable = (ReceivedLength > 0);
    }

    return _IsPacketDataAvailable ? _IncomingPacket.Size : 0;
}

bool TKinokoBufferReader::GetReadEntry(void*& Address, size_t& Size)
{
    if (_IsPacketDataAvailable) {
	Address = (void*) (_BaseAddress + _IncomingPacket.OffsetAddress);
	Size = (size_t) _IncomingPacket.Size;
    }
    else {
	int ReceivedLength = _MessageQueue->Receive(
	    (TMushMessageQueuePacket*) &_IncomingPacket, 
	    sizeof(_IncomingPacket)
	);

	if (ReceivedLength > 0) {
	    Address = (void*) (_BaseAddress + _IncomingPacket.OffsetAddress);
	    Size = (size_t) _IncomingPacket.Size;
	}
	else {
	    Address = 0;
	    Size = 0;
	}
    }

    _IsPacketDataAvailable = false;

    return (Address != 0);
}

bool TKinokoBufferReader::SendAcknowledge(const void* Address)
{
    _OutgoingPacket.ReceiverAddress = TKinokoBufferServer::MessageQueueAddress();
    _OutgoingPacket.SenderAddress = MessageQueueAddress();
    _OutgoingPacket.PacketType = TKinokoBufferPacket::ptDataAcknowledge;
    _OutgoingPacket.OffsetAddress = (unsigned long) Address - _BaseAddress;

    //_TimeKeeper->Start();
    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_OutgoingPacket, 
	sizeof(_OutgoingPacket)
    );
    //_TimeKeeper->Stop();

    if (SentLength <= 0) {
	_Logger->Write(
	    "no answer from BufferServer (trying again...)", 
	    TKinokoBufferLogger::mlWarning
        );

	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_OutgoingPacket, 
	    sizeof(_OutgoingPacket)
	);
	if (SentLength <= 0) {
	    _Logger->Write(
		"no answer from BufferServer (interrupted)", 
		TKinokoBufferLogger::mlError
	    );
	}
    }

    return (SentLength > 0);
}
