/* KinokoBufferServer.cc */
/* Created by Enomoto Sanshiro on 19 April 1998. */
/* Last updated by Enomoto Sanshiro on 21 August 2002. */


#include <iostream>
#include <fstream>
#include <strstream>
#include <vector>
#include <deque>
#include <algorithm>
#include "MushDefs.hh"
#include "MushIpc.hh"
#include "MushMisc.hh"
#include "KinokoDefs.hh"
#include "KinokoArena.hh"
#include "KinokoBufferLogger.hh"
#include "KinokoBufferServer.hh"

using namespace std;


//#define DEBUG
#define blog ofstream("Dump-Buffer", ios::app)


static const char* KinokoBufferIpcProjectPath = "";
static const int KinokoBufferServerMessageQueueAddress = 1;
static const int PrescaledLoggingInterval_sec = 10;


TKinokoBufferServer::TKinokoBufferServer(int SharedMemoryProjectId, int MessageQueueProjectId, unsigned long BufferSize, int BufferEntryTableSize, TKinokoBufferLogger* Logger) throw(TKinokoException)
{
    _BufferSize = BufferSize;
    _BufferEntryTableSize = BufferEntryTableSize;
    _Logger = Logger;

    _PrescaledLogger = new TKinokoPrescaledBufferLogger(
	_Logger, PrescaledLoggingInterval_sec
    );

    _SharedMemory = 0;
    _MessageQueue = 0;
    try {
	string ProjectPath = this->IpcProjectPath();
	int AccessPermission = 0600;
	bool IsSelfish = true;

	_SharedMemory = new TMushServerSharedMemory(
	    BufferSize, SharedMemoryProjectId, ProjectPath, 
	    AccessPermission, IsSelfish
	);
	_MessageQueue = new TMushServerMessageQueue(
	    MessageQueueProjectId, ProjectPath, 
	    AccessPermission, IsSelfish
	);
    }
    catch (TSystemCallException &e) {
	delete _MessageQueue;
	delete _SharedMemory;

        throw TKinokoException(
	    "TKinokoBufferServer::TKinokoBufferServer()",
	    "system call exception: " + e.Message()
        );
    }

    _MessageQueue->SetAddress(this->MessageQueueAddress());
    _BaseAddress = (unsigned long) _SharedMemory->BasePointer();

    _Arena = new TKinokoArena(
	(void *) _BaseAddress, (size_t) BufferSize, BufferEntryTableSize
    );

    _NumberOfReaders = 0;

    int QueuePacketCapacity = _MessageQueue->QueueCapacity() / sizeof(TKinokoBufferPacket);
    _MaxNumberOfUnreadPackets = QueuePacketCapacity - 128;
    _NumberOfUnreadPackets = 0;

    _WrittenDataSize = 0;
    _NumberOfWrittenPackets = 0;
    _LastReportTime = TMushDateTime().SecSinceEpoch();
}

TKinokoBufferServer::~TKinokoBufferServer()
{
    delete _Arena;
    delete _MessageQueue;
    delete _SharedMemory;
    delete _PrescaledLogger;
}

const char* TKinokoBufferServer::IpcProjectPath(void)
{
    return KinokoBufferIpcProjectPath;
}

long TKinokoBufferServer::MessageQueueAddress(void)
{
    return KinokoBufferServerMessageQueueAddress;
}

int TKinokoBufferServer::DoTransaction(TMushIntervalTimer *TimeKeeper)
{
    // firstly try nowait receive since starting a timer may take time
    static const bool NoWait = true;
    int Length = _MessageQueue->Receive(
	(TMushMessageQueuePacket *) &_Packet, sizeof(_Packet), NoWait
    );

    // secondly use an interval timer for timeout processings
    if (Length <= 0) {
	TimeKeeper->Start();
	Length = _MessageQueue->Receive(
	    (TMushMessageQueuePacket *) &_Packet, sizeof(_Packet)
	);
	TimeKeeper->Stop();
    }

    if (Length > 0) {
#ifdef DEBUG
	DumpKinokoBufferPacket(Packet);
	blog << "FreeArea (before transaction): " << _Arena->AvailableSize() << endl;
#endif

	DispatchMessage(_Packet);

#ifdef DEBUG
	blog << "FreeArea (after transaction): " << _Arena->AvailableSize() << endl << endl;	
#endif
    }

    return (Length > 0);
}

void TKinokoBufferServer::DispatchMessage(const TKinokoBufferPacket &Packet)
{
    switch (Packet.PacketType) {
      case TKinokoBufferPacket::ptWriteRequest:
        if (! ProcessWriteRequest(TKinokoBufferRequest(Packet))) {
	    _WriteRequestQueue.push_back(TKinokoBufferRequest(Packet));
	}
	break;
      case TKinokoBufferPacket::ptDataStrobe:
        if (! ProcessReadRequest(TKinokoBufferRequest(Packet))) {
	    _ReadRequestQueue.push_back(TKinokoBufferRequest(Packet));
	}
	break;
      case TKinokoBufferPacket::ptDataAcknowledge:
	DetachEntry(Packet.OffsetAddress);
	ProcessPendingRequests();
	break;
      case TKinokoBufferPacket::ptReaderAttachRequest:
	AttachReader(Packet.SenderAddress);
	break;
      case TKinokoBufferPacket::ptReaderDetachRequest:
	DetachReader(Packet.SenderAddress);
	break;
      default:
	_Logger->Write("unknown buffer request", TKinokoBufferLogger::mlPanic);
    }
}

void TKinokoBufferServer::ProcessPendingRequests(void)
{
    while (! _ReadRequestQueue.empty()) {
	const TKinokoBufferRequest& ReadRequest = _ReadRequestQueue.front();
	if (ProcessReadRequest(ReadRequest)) {
	    _ReadRequestQueue.pop_front();
	}
	else {
	    break;
	}
    }

    while (! _WriteRequestQueue.empty()) {
	const TKinokoBufferRequest& WriteRequest = _WriteRequestQueue.front();
	if (ProcessWriteRequest(WriteRequest)) {
	    _WriteRequestQueue.pop_front();
	}
	else {
	    break;
	}
    }
}

bool TKinokoBufferServer::ProcessWriteRequest(const TKinokoBufferRequest& WriteRequest)
{
    bool IsRequestProcessed = false;

    if (_NumberOfReaders == 0) {
	_PrescaledLogger->Write(
            "no reader process (allocation deferred)", 
            TKinokoBufferLogger::mlWarning
        );
    }

    else if (_Arena->IsAllocatable(WriteRequest.Size)) {
	void *Address;
	if (_Arena->CreateEntry(
	    (size_t) WriteRequest.Size, Address, _NumberOfReaders
	)){
	    SendEntryReady(
		WriteRequest, (unsigned long) Address - _BaseAddress
            );
	    _WrittenDataSize += WriteRequest.Size;
	    _NumberOfWrittenPackets++;

	    IsRequestProcessed = true;
	}
	else {
	    if (_PrescaledLogger->Write(
		"unable to create buffer entry (allocation deferred)", 
		TKinokoBufferLogger::mlWarning
	    )){
		ReportStatus();
	    }
	}
    }
    else if (WriteRequest.Size > _BufferSize) {
	_Logger->Write(
	    "too large data area requested (request ignored)", 
	    TKinokoBufferLogger::mlError
	);

        // throw away the request
	IsRequestProcessed = true;
    }
    else if (_Arena->NumberOfAvailableEntries() <= 0) {
	if (_PrescaledLogger->Write(
	    "buffer entry exhausted (allocation deferred)", 
	    TKinokoBufferLogger::mlWarning
	)){
	    ReportStatus();
	}
    }
    else {
	if (_PrescaledLogger->Write(
	    "buffer full (allocation deferred)", 
	    TKinokoBufferLogger::mlWarning
	)){
	    ReportStatus();
	}
    }

    return IsRequestProcessed;
}

void TKinokoBufferServer::SendEntryReady(const TKinokoBufferRequest& WriteRequest, unsigned long OffsetAddress)
{
    _Packet.ReceiverAddress = WriteRequest.WriterMessageAddress;
    _Packet.SenderAddress = _MessageQueue->Address();
    _Packet.PacketType = TKinokoBufferPacket::ptEntryReady;
    _Packet.OffsetAddress = OffsetAddress;
    _Packet.Size = WriteRequest.Size;

#ifdef DEBUG
    blog << endl;
    DumpKinokoBufferPacket(_Packet);
    blog << "FreeArea (after allocation): " << _Arena->AvailableSize() << endl;
#endif
    
    int SentLength =  _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    if (SentLength == 0) {
	_Logger->Write(
	    "SendEntryReady(): "
	    "incomplete packet processing (interrupted)", 
	    TKinokoBufferLogger::mlError
	);
	ReportStatus();
    }
}

bool TKinokoBufferServer::ProcessReadRequest(const TKinokoBufferRequest& ReadRequest)
{
    if (_NumberOfUnreadPackets < _MaxNumberOfUnreadPackets) {
	SendReadRequest(ReadRequest);
	return true;
    }
    else {
	return false;
    }
}

void TKinokoBufferServer::SendReadRequest(const TKinokoBufferRequest& ReadRequest)
{
    _Packet.SenderAddress = _MessageQueue->Address();
    _Packet.PacketType = TKinokoBufferPacket::ptReadRequest;
    _Packet.OffsetAddress = ReadRequest.OffsetAddress;
    _Packet.Size = ReadRequest.Size;

    int SentLength;
    for (int i = 0; i < _NumberOfReaders; i++) {
	_Packet.ReceiverAddress = _ReaderAddressList[i];

	SentLength = _MessageQueue->Send(
	    (TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
	);
	if (SentLength == 0) {
	    _Logger->Write(
		"SendReadRequest(): "
		"incomplete packet processing (interrupted)", 
		TKinokoBufferLogger::mlError
	    );
	    ReportStatus();
	    break;
	}

	_NumberOfUnreadPackets++;
    }
}

void TKinokoBufferServer::DetachEntry(unsigned long OffsetAddress)
{
    if (! _Arena->DetachEntry((void*) (_BaseAddress + OffsetAddress))) {
	ostrstream MessageStream;
	MessageStream << "releasing invalid address: " << OffsetAddress << ends;
	_Logger->Write(
	    MessageStream.str(), TKinokoBufferLogger::mlPanic
	);
    }

    _NumberOfUnreadPackets--;
}

void TKinokoBufferServer::AttachReader(long ReaderAddress)
{
    _ReaderAddressList.push_back(ReaderAddress);
    _NumberOfReaders++;
}

void TKinokoBufferServer::DetachReader(long ReaderAddress)
{
    vector<long>::iterator Reader;
    Reader = find(_ReaderAddressList.begin(), _ReaderAddressList.end(), ReaderAddress);

    if (Reader == _ReaderAddressList.end()) {
	_Logger->Write(
	    "detach request from unattached reader", 
            TKinokoBufferLogger::mlPanic
	);
	return;
    }

    _ReaderAddressList.erase(Reader);
    _NumberOfReaders--;

    _Packet.ReceiverAddress = ReaderAddress;
    _Packet.SenderAddress = _MessageQueue->Address();
    _Packet.PacketType = TKinokoBufferPacket::ptDetachAcknowledge;

    int SentLength = _MessageQueue->Send(
	(TMushMessageQueuePacket*) &_Packet, sizeof(_Packet)
    );
    if (SentLength == 0) {
	_Logger->Write(
	    "DetachReader(): "
	    "incomplete packet processing (interrupted)", 
	    TKinokoBufferLogger::mlError
	);
	ReportStatus();
    }
}

void TKinokoBufferServer::ReportStatus(void)
{
    char StatusMessage[512];
    ostrstream MessageStream(StatusMessage, sizeof(StatusMessage));
    
    MessageStream <<
	"used: " << _Arena->UsedSize() << 
	"/" << _BufferSize << ", " <<
	"entries: " << _Arena->NumberOfEntries() <<
	"/" << _BufferEntryTableSize << ", " <<
	"head:tail: " << _Arena->DataHeadOffset() <<
	":" << _Arena->NextStartOffset() << ", " <<
	"written(size:entries)/time: " << _WrittenDataSize <<
	":" << _NumberOfWrittenPackets << 
	"/" << TMushDateTime::SecSince(_LastReportTime) << ", " <<
	"unread(queue1:queue2): " << (_ReadRequestQueue.size() * _NumberOfReaders) << 
	":" << _NumberOfUnreadPackets << 
	"/" << _MaxNumberOfUnreadPackets <<
    ends;

    _Logger->Write(StatusMessage, TKinokoBufferLogger::mlLog);

    _WrittenDataSize = 0;
    _NumberOfWrittenPackets = 0;
    _LastReportTime = TMushDateTime::SecSinceEpoch();
}

void DumpKinokoBufferPacket(TKinokoBufferPacket& KinokoBufferPacket)
{
#ifdef DEBUG
    if (KinokoBufferPacket.PacketType == TKinokoBufferPacket::ptWriteRequest) {
	blog << "========================" << endl;
    }

    blog << "From: " << KinokoBufferPacket.SenderAddress << endl;
    blog << "To: " << KinokoBufferPacket.ReceiverAddress << endl;
    
    switch (KinokoBufferPacket.PacketType) {
      case TKinokoBufferPacket::ptWriteRequest:
	blog << "Message-Type: " << "WriteRequest" << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptEntryReady:
	blog << "Message_Type: " << "EntryReady" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptDataStrobe:
	blog << "Message-Type: " << "DataStrobe" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptReadRequest:
	blog << "Message-Type: " << "ReadRequest" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptDataAcknowledge:
	blog << "Message-Type: " << "DataAcknowledge" << endl;
	blog << "Address: " << KinokoBufferPacket.OffsetAddress << endl;
	blog << "Size: " << KinokoBufferPacket.Size << endl;
	break;
      case TKinokoBufferPacket::ptReaderAttachRequest:
	blog << "Message-Type: " << "ReaderAttachRequest" << endl;
	break;
      case TKinokoBufferPacket::ptReaderDetachRequest:
	blog << "Message-Type: " << "ReaderDetachRequest" << endl;
	break;
      default:
	blog << "Message-Type: " << "UNDEFINED" << endl;
    }
#endif
}
