/* KinokoBufferCom.cc */
/* Created by Enomoto Sanshiro on 20 October 2000. */
/* Last updated by Enomoto Sanshiro on 9 October 2000. */


#include <strstream>
#include <string>
#include "MushNetworkSocket.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KinokoBufferServer.hh"
#include "KinokoBufferCom.hh"

using namespace std;


static const int g_IpcKeyBase = 100;  //... temporary
static const int DefaultBufferSize = 32 * 1024 * 1024; 
static const int DefaultBufferEntryTableSize = 1024 * 1024;

static const int DefaultReportInterval = 180;
static const int ReportTimeCheckPrescaleRate = 100;


TKinokoBufferCom::TKinokoBufferCom(void)
: TKinokoStreamComponent("KinokoBufferCom")
{
    _BufferSize = DefaultBufferSize;
    _BufferEntryTableSize = DefaultBufferEntryTableSize;

    _BufferServer = 0;
    _BufferLogger = 0;

    _ReportInterval = DefaultReportInterval;
    _LastReportTime = TMushDateTime::SecSinceEpoch();
    _NumberOfTransactionsSinceLastReport = 0;
}

TKinokoBufferCom::~TKinokoBufferCom()
{
    delete _BufferServer;
    delete _BufferLogger;
}

void TKinokoBufferCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamComponent::BuildDescriptor(Descriptor);

    Descriptor.RegisterProperty(
	PropertyId_SharedMemoryId, TKcomPropertyDeclaration(
	    "shared_memory_id", TKcomPropertyDeclaration::Type_Int
	)
    );
    Descriptor.RegisterProperty(
	PropertyId_MessageQueueId, TKcomPropertyDeclaration(
	    "message_queue_id", TKcomPropertyDeclaration::Type_Int
	)
    );

    TKcomEventDeclaration SetBufferSizeEvent("setBufferSize");
    SetBufferSizeEvent.AddArgument(TKcomPropertyDeclaration(
        "buffer_size", TKcomPropertyDeclaration::Type_Int
    ));
    SetBufferSizeEvent.AddArgument(TKcomPropertyDeclaration(
        "entry_table_size", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetBufferSize, SetBufferSizeEvent);

    TKcomEventDeclaration SetReportIntervalEvent("setReportInterval");
    SetReportIntervalEvent.AddArgument(TKcomPropertyDeclaration(
        "interval_sec", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetReportInterval, SetReportIntervalEvent);

    TKcomEventDeclaration StartEvent("start");
    Descriptor.RegisterEventSlot(EventId_Start, StartEvent);

    TKcomEventDeclaration QuitEvent("quit");
    Descriptor.RegisterEventSlot(EventId_Quit, QuitEvent);
}

int TKinokoBufferCom::DoTransaction(void) throw(TKcomException)
{
    try {
	if ((_State == State_Running) && (_BufferServer != 0)) {
	    return ProcessData();
	}
	
	switch (_State) {
          case State_Initial:
	    SetProperties();
	    ChangeState(State_StreamReady);
	    break;
	    
          case State_StreamReady:
	    // wait start() event
	    _TimeKeeper->Suspend();
	    break;
	    
	  default:
	    return TKinokoStreamComponent::DoTransaction();
	}
    }
    catch (TKinokoException &e) {
	throw TKcomException(
	    "TKinokoBufferCom::DoTransaction()",
	    "kinoko exception: " + e.Message()
	);
    }

    return 1;
}

int TKinokoBufferCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetBufferSize:
	Result = ProcessSetBufferSizeEvent(Event);
	break;

      case EventId_SetReportInterval:
	Result = ProcessSetReportIntervalEvent(Event);
	break;

      case EventId_Start:
	Result = ProcessStartEvent(Event);
	break;

      case EventId_Quit:
	Result = ProcessQuitEvent(Event);
	break;
	
      default:
	Result = TKinokoStreamComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoBufferCom::ProcessSetBufferSizeEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setBufferSize(): too few arguments"
        );
	return 0;
    }

    if (_BufferServer != 0) {
	_Logger->WriteError(
	    ComponentName(), "setBufferSize(): buffer already constructed"
        );
	return 0;
    }

    string BufferSizeString = "*";
    if (Event.ArgumentList().size() > 0) {
	BufferSizeString = Event.ArgumentList()[0];
	istrstream BufferSizeStream(BufferSizeString.c_str());
	if (! (BufferSizeStream >> _BufferSize)) {
	    _Logger->WriteError(
		ComponentName(), 
		"setBufferSize(): invalid argument: " + BufferSizeString
	    );
	    return 0;
	}
    }

    string EntryTableSizeString = "*";
    if (Event.ArgumentList().size() > 1) {
	EntryTableSizeString = Event.ArgumentList()[1];
	istrstream EntryTableSizeStream(EntryTableSizeString.c_str());
	if (! (EntryTableSizeStream >> _BufferEntryTableSize)) {
	    _Logger->WriteError(
		ComponentName(), 
		"setBufferSize(): invalid argument: " + EntryTableSizeString
	    );
	    return 0;
	}
    }
    
    _Logger->WriteDebug(
	ComponentName(), 
	"setBufferSize(): " + BufferSizeString + ", " + EntryTableSizeString
    );

    return 1;
}

int TKinokoBufferCom::ProcessSetReportIntervalEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setReportInterval(): too few arguments"
        );
	return 0;
    }
    string ReportIntervalString = Event.ArgumentList()[0];

    istrstream ReportIntervalStream(ReportIntervalString.c_str());
    if (! (ReportIntervalStream >> _ReportInterval)) {
	_Logger->WriteError(
	    ComponentName(), 
	    "setReportInterval(): invalid argument: " + ReportIntervalString
	);
	return 0;
    }

    _Logger->WriteDebug(
	ComponentName(), "setReportInterval(): " + ReportIntervalString
    );

    return 1;
}

int TKinokoBufferCom::ProcessStartEvent(TKcomEvent& Event)
{
    Construct();
    ChangeState(State_Running);
    
    return 1;
}

int TKinokoBufferCom::ProcessQuitEvent(TKcomEvent& Event)
{
    Destruct();
    ChangeState(State_Shutdown);

    return 1;
}

void TKinokoBufferCom::SetProperties(void) throw(TKinokoException)
{
    int BufferId = Registry()->GetSequenceValue("data_buffer_id");
    
    _SharedMemoryId = g_IpcKeyBase + 2 * BufferId;
    _MessageQueueId = g_IpcKeyBase + 2 * BufferId + 1;

    SetProperty(PropertyId_StreamType, _StreamTypeName[StreamType_Buffer]);
    SetProperty(PropertyId_Host, TMushNetworkSocket::LocalIPAddress());

    SetLongProperty(PropertyId_SharedMemoryId, _SharedMemoryId);
    SetLongProperty(PropertyId_MessageQueueId, _MessageQueueId);
}

void TKinokoBufferCom::Construct(void) throw(TKinokoException)
{
    _BufferLogger = new TKinokoLoggerBufferLogger(ComponentName(), _Logger);
    try {
	_BufferServer = new TKinokoBufferServer(
	    _SharedMemoryId, _MessageQueueId, 
	    _BufferSize, _BufferEntryTableSize,
	    _BufferLogger
	);
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());
	delete _BufferServer;
	_BufferServer = 0;
    }
}

void TKinokoBufferCom::Destruct(void) throw(TKinokoException)
{
    delete _BufferServer;
    delete _BufferLogger;

    _BufferServer = 0;
    _BufferLogger = 0;
}

int TKinokoBufferCom::ProcessData(void)throw(TKinokoException)
{
    if (_NumberOfTransactionsSinceLastReport++ > ReportTimeCheckPrescaleRate) {
	if (TMushDateTime::SecSince(_LastReportTime) >= _ReportInterval) {
	    _LastReportTime = TMushDateTime::SecSinceEpoch();
	    _NumberOfTransactionsSinceLastReport = 0;

	    _BufferServer->ReportStatus();
	}
    }

    return _BufferServer->DoTransaction(_TimeKeeper);
}
