/* KinokoComponent.cc */
/* Created by Enomoto Sanshiro on 9 October 2000. */
/* Last updated by Enomoto Sanshiro on 19 May 2003. */


#include <strstream>
#include "MushTimer.hh"
#include "KcomComponent.hh"
#include "KinokoComponent.hh"

using namespace std;


TKinokoComponent::TKinokoComponent(const string& TypeName)
: TKcomComponent(TypeName)
{
    //... this value should be set by an environmental variable
    _TimeKeeper = new TMushRealTimeTimer(0, 10000);

    _IsConditionMonitorEnabled = 0;
    _LastConditionCheckTime = 0;
    _HeartBeatCount = 0;
    _VitalLevel = 0;
}

TKinokoComponent::~TKinokoComponent()
{
    delete _TimeKeeper;
}

void TKinokoComponent::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKcomComponent::BuildDescriptor(Descriptor);

    Descriptor.RegisterProperty(
	PropertyId_Host, TKcomPropertyDeclaration(
	    "host", TKcomPropertyDeclaration::Type_String
	)
    );

    Descriptor.RegisterProperty(
	PropertyId_ConditionCheckTime, TKcomPropertyDeclaration(
	    "condition_check_time", TKcomPropertyDeclaration::Type_Long
	)
    );
    Descriptor.RegisterProperty(
	PropertyId_HeartBeatCount, TKcomPropertyDeclaration(
	    "heart_beat_count", TKcomPropertyDeclaration::Type_Long
	)
    );
    Descriptor.RegisterProperty(
	PropertyId_VitalLevel, TKcomPropertyDeclaration(
	    "vital_level", TKcomPropertyDeclaration::Type_Long
	)
    );

    TKcomEventDeclaration SystemEvent("system");
    Descriptor.RegisterEventSlot(EventId_System, SystemEvent);

    TKcomEventDeclaration EnableConditionMonitorEvent("enableConditionMonitor");
    Descriptor.RegisterEventSlot(
	EventId_EnableConditionMonitor, EnableConditionMonitorEvent
    );
}

int TKinokoComponent::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 1;

    switch (EventId) {
      case EventId_System:
	Result = ProcessSystemEvent(Event, EventResponse);
	break;

      case EventId_EnableConditionMonitor:
	Result = ProcessEnableConditionMonitorEvent(Event);
	break;

      default:
	return TKcomComponent::ProcessEvent(EventId, Event, EventResponse);
    }

    return 1;
}

int TKinokoComponent::ProcessSystemEvent(TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    if (Event.ArgumentList().size() != 1) {
	return 0;
    }

    string Command = Event.ArgumentList()[0];
    long ReturnValue = system(Command.c_str());

    char Buffer[32];
    ostrstream os(Buffer, sizeof(Buffer));
    os << ReturnValue << ends;
    EventResponse.ReturnValue() = Buffer;

    return 1;
}

int TKinokoComponent::ProcessEnableConditionMonitorEvent(TKcomEvent& Event)
{
    _IsConditionMonitorEnabled = true;
    return 1;
}

int TKinokoComponent::DoTransaction(void) throw(TKcomException)
{
    _TimeKeeper->Suspend();
    return 0;
}

void TKinokoComponent::OnTransaction(void) throw(TKcomException)
{
    _HeartBeatCount++;
    _VitalLevel++;  //... temporary

    if (_IsConditionMonitorEnabled) {
	if (TMushDateTime::SecSince(_LastConditionCheckTime) > 60) {
	    _LastConditionCheckTime = TMushDateTime::SecSinceEpoch();

	    SetLongProperty(PropertyId_ConditionCheckTime, _LastConditionCheckTime);
	    SetLongProperty(PropertyId_HeartBeatCount, _HeartBeatCount);
	    SetLongProperty(PropertyId_VitalLevel, _VitalLevel);

	    _HeartBeatCount = 0;
	    _VitalLevel = 0;
	}
    }
}

void TKinokoComponent::EmitKcomEvent(const string& EventName, const vector<string>& EventArgumentList)
{
    int EventId = EventId_Any;
    TKcomEvent Event(EventName);

    for (unsigned i = 0; i < EventArgumentList.size(); i++) {
	Event.ArgumentList().push_back(EventArgumentList[i]);
    }

    EmitEventOneWay(EventId, Event);
}

bool TKinokoComponent::EmitKcomEventTo(const string& TargetName, const string& EventName, const vector<string>& EventArgumentList, string& ReturnValue)
{
    TKcomEvent Event(EventName);
    TKcomEventResponse EventResponse;

    for (unsigned i = 0; i < EventArgumentList.size(); i++) {
	Event.ArgumentList().push_back(EventArgumentList[i]);
    }

    EmitEventTo(TargetName, Event, EventResponse);

    ReturnValue = EventResponse.ReturnValue();

    return ! EventResponse.IsError();
}
