/* KinokoControllerCom.cc */
/* Created by Enomoto Sanshiro on 7 December 2000. */
/* Last updated by Enomoto Sanshiro on 22 August 2002. */


#include <string>
#include <vector>
#include <iostream>
#include <strstream>
#include "KinokoLoggerOrb.hh"
#include "KinokoComponentPlatform.hh"
#include "KinokoSystemComponent.hh"
#include "KinokoController.hh"
#include "KinokoControllerCom.hh"

using namespace std;


static const string Comment = "Controller: Kinoko System Controller";


TKinokoControllerCom::TKinokoControllerCom(void)
: TKinokoSystemComponent("KinokoControllerCom")
{
    _Logger = 0;
    _EventEmitter = 0;
    _Registry = 0;

    _Controller = 0;
}

TKinokoControllerCom::~TKinokoControllerCom()
{
    delete _Controller;

    delete _Logger;
    delete _EventEmitter;
    delete _Registry;
}

void TKinokoControllerCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoSystemComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    TKcomEventDeclaration OpenEvent("open");
    OpenEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_Open, OpenEvent);

    TKcomEventDeclaration ChangeStateEvent("changeState");
    ChangeStateEvent.AddArgument(TKcomPropertyDeclaration(
        "state_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_ChangeState, ChangeStateEvent);

    TKcomEventDeclaration SetEvent("set");
    SetEvent.AddArgument(TKcomPropertyDeclaration(
        "widget_name", TKcomPropertyDeclaration::Type_String
    ));
    SetEvent.AddArgument(TKcomPropertyDeclaration(
        "value", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_Set, SetEvent);

    TKcomEventDeclaration ExecuteEvent("execute");
    ExecuteEvent.AddArgument(TKcomPropertyDeclaration(
        "action_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_Execute, ExecuteEvent);

    TKcomEventDeclaration OpenPopupEvent("openPopup");
    OpenPopupEvent.AddArgument(TKcomPropertyDeclaration(
        "type", TKcomPropertyDeclaration::Type_String
    ));
    OpenPopupEvent.AddArgument(TKcomPropertyDeclaration(
        "action_list", TKcomPropertyDeclaration::Type_String
    ));
    OpenPopupEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_OpenPopup, OpenPopupEvent);

    TKcomEventDeclaration OpenQueryPopupEvent("openQueryPopup");
    OpenQueryPopupEvent.AddArgument(TKcomPropertyDeclaration(
        "type", TKcomPropertyDeclaration::Type_String
    ));
    OpenQueryPopupEvent.AddArgument(TKcomPropertyDeclaration(
        "action_list", TKcomPropertyDeclaration::Type_String
    ));
    OpenQueryPopupEvent.AddArgument(TKcomPropertyDeclaration(
        "message", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_OpenQueryPopup, OpenQueryPopupEvent);

    TKcomEventDeclaration SaveEvent("save");
    SaveEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_Save, SaveEvent);

    TKcomEventDeclaration LoadEvent("load");
    LoadEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_Load, LoadEvent);

    _Logger = new TKinokoLoggerProxy();
    Descriptor.RegisterImportObject(
    	_Logger, TKcomObjectDeclaration(_Logger->ClassName(), "logger")
    );
}

int TKinokoControllerCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_Open:
	Result = ProcessOpenEvent(Event);
	break;

      case EventId_ChangeState:
	Result = ProcessChangeStateEvent(Event);
	break;

      case EventId_Set:
	Result = ProcessSetEvent(Event);
	break;

      case EventId_Execute:
	Result = ProcessExecuteEvent(Event);
	break;

      case EventId_OpenPopup:
	Result = ProcessOpenPopupEvent(Event);
	break;

      case EventId_OpenQueryPopup:
	Result = ProcessOpenQueryPopupEvent(Event, EventResponse);
	break;

      case EventId_Save:
	Result = ProcessSaveEvent(Event);
	break;

      case EventId_Load:
	Result = ProcessLoadEvent(Event);
	break;

      default:
	Result = TKinokoSystemComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

void TKinokoControllerCom::Initialize(void) throw(TKcomException)
{
    _EventEmitter = new TKinokoComponentEventEmitter(this);
    _Registry = new TKinokoComponentRegistry(Registry());

    _Controller = new TKinokoController(
	InputStream(), OutputStream(), _EventEmitter, _Registry
    );

    //... for backward compatibility ...//
    TKcomEvent Event("ready");
    EmitEventOneWay(EventId_Any, Event);
}

void TKinokoControllerCom::Finalize(void) throw(TKcomException)
{
    delete _Registry;
    delete _EventEmitter;
    _Registry = 0;
    _EventEmitter = 0;

    if (_Controller != 0) {
	delete _Controller;
	_Controller = 0;
    }
}

int TKinokoControllerCom::DoTransaction(void) throw(TKcomException)
{
    if (_Controller->ProcessInput() == 0) {
	TKinokoSystemComponent::DoTransaction();
	return 0;
    }

    return 1;
}

int TKinokoControllerCom::ProcessOpenEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "Open(): too few arguments"
	);
	return 0;
    }

    string FileName = Event.ArgumentList()[0];
    _Controller->Open(FileName);

    _Logger->WriteDebug(
	ComponentName(), "open(): " + FileName
    );

    return 1;
}

int TKinokoControllerCom::ProcessChangeStateEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "ChangeState(): too few arguments"
	);
	return 0;
    }

    string StateName = Event.ArgumentList()[0];
    _Controller->ChangeState(StateName);

    _Logger->WriteDebug(
	ComponentName(), "changeState(): " + StateName
    );

    return 1;
}

int TKinokoControllerCom::ProcessSetEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 2) {
	_Logger->WriteError(
	    ComponentName(), "set(): too few arguments"
	);
	return 0;
    }

    string WidgetName = Event.ArgumentList()[0];
    string Value = Event.ArgumentList()[1];
    _Controller->SetWidgetProperty(WidgetName, Value);

#if 0
    _Logger->WriteDebug(
	ComponentName(), "set(): " + WidgetName + " " + Value
    );
#endif

    return 1;
}

int TKinokoControllerCom::ProcessExecuteEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "execute(): too few arguments"
	);
	return 0;
    }

    string ActionName = Event.ArgumentList()[0];
    _Controller->ExecuteAction(ActionName);

#if 0
    _Logger->WriteDebug(ComponentName(), "execute(): " +ActionName);
#endif

    return 1;
}

int TKinokoControllerCom::ProcessOpenPopupEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 3) {
	_Logger->WriteError(
	    ComponentName(), "openPopup(): too few arguments"
	);
	return 0;
    }

    string Type = Event.ArgumentList()[0];
    string ActionList = Event.ArgumentList()[1];
    string Message = Event.ArgumentList()[2];

    _Controller->OpenPopup(Type, ActionList, Message);

    return 1;
}

int TKinokoControllerCom::ProcessOpenQueryPopupEvent(TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    if (Event.ArgumentList().size() < 3) {
	_Logger->WriteError(
	    ComponentName(), "openQueryPopup(): too few arguments"
	);
	return 0;
    }

    string Type = Event.ArgumentList()[0];
    string ActionList = Event.ArgumentList()[1];
    string Message = Event.ArgumentList()[2];

    string Reply = _Controller->OpenQueryPopup(Type, ActionList, Message);
    EventResponse.ReturnValue() = Reply;

    return 1;
}

int TKinokoControllerCom::ProcessSaveEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "Save(): too few arguments"
	);
	return 0;
    }
    
    string FileName = Event.ArgumentList()[0];
    _Controller->Save(FileName);

    return 1;
}

int TKinokoControllerCom::ProcessLoadEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "Load(): too few arguments"
	);
	return 0;
    }
    
    string FileName = Event.ArgumentList()[0];
    _Controller->Load(FileName);

    return 1;
}

void TKinokoControllerCom::OnQuit(void) throw(TKcomException)
{
    _Controller->Quit();

    TKinokoSystemComponent::OnQuit();
}
