/* KinokoReproducerCom.cc */
/* Created by Enomoto Sanshiro on 15 January 2002. */
/* Last updated by Enomoto Sanshiro on 15 January 2002. */


#include <iostream>
#include <strstream>
#include <string>
#include <set>
#include "KinokoStream.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataProcessor.hh"
#include "KinokoDataFilterCom.hh"

using namespace std;


static const string Comment = "DataFilter: select specific datasoruces from stream";


TKinokoDataFilterCom::TKinokoDataFilterCom(void)
: TKinokoStreamPipeComponent("KinokoDataFilterCom")
{
    _DataReceiver = 0;
    _DataStreamScanner = 0;
}

TKinokoDataFilterCom::~TKinokoDataFilterCom()
{
    delete _DataStreamScanner;
    delete _DataReceiver;
}

void TKinokoDataFilterCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamPipeComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    TKcomEventDeclaration AddDataSourceEvent("addDataSource");
    AddDataSourceEvent.AddArgument(TKcomPropertyDeclaration(
        "datasource_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_AddDataSource, AddDataSourceEvent);
}

int TKinokoDataFilterCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_AddDataSource:
	Result = ProcessAddDataSourceEvent(Event);
	break;

      default:
	Result = TKinokoStreamPipeComponent::ProcessEvent(
            EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoDataFilterCom::ProcessAddDataSourceEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "addDataSource(): too few arguments"
        );
	return 0;
    }

    string DataSourceName = Event.ArgumentList()[0];
    _DataSourceNameSet.insert(DataSourceName);
    
    _Logger->WriteDebug(
	ComponentName(), "addDataSource(): " + DataSourceName
    );

    return 1;
}

void TKinokoDataFilterCom::Construct(void) throw(TKinokoException)
{
    _DataReceiver = new TKinokoDataReceiver();
    _DataStreamScanner = new TKinokoDataStreamScanner();

    _DataReceiver->SetStreamCommandProcessor(_StreamCommandProcessor);
    _DataReceiver->ConstructInlet(_InputDataStream);
}

void TKinokoDataFilterCom::Destruct(void) throw(TKinokoException)
{
    delete _DataStreamScanner;
    delete _DataReceiver;

    _DataStreamScanner = 0;
    _DataReceiver = 0;
}

int TKinokoDataFilterCom::ProcessData(void)throw(TKinokoException)
{
    if (! _InputDataStream->HasData()) {
	_TimeKeeper->Suspend();
        return 0;
    }

    void* Packet;
    int PacketSize = _InputDataStream->NextEntry(Packet);
    if (PacketSize <= 0) {
	_TimeKeeper->Suspend();
	return 0;
    }

    _DataStreamScanner->CorrectByteOrder(Packet, PacketSize);
    long DataSourceId = _DataStreamScanner->DataSourceIdOf(Packet);
    
    if (_DataReceiver->ProcessDataDescriptorPacket(Packet, PacketSize)) {
	TKinokoDataSource* DataSource = _DataReceiver->InputDataDescriptor()->DataSource(DataSourceId);
	if (DataSource != 0) {
	    string DataSourceName = DataSource->DataSourceName();
	    if (_DataSourceNameSet.count(DataSourceName) > 0) {
		_DataSourceIdSet.insert(DataSourceId);
	    }
	}
    }

    if (
	(! _DataSourceNameSet.empty()) &&
	(_DataSourceIdSet.count(DataSourceId) == 0)
    ){
	_InputDataStream->Flush(Packet);
	return 0;
    }

    int WrittenLength;
    while ((WrittenLength = _OutputDataStream->Write(Packet, PacketSize)) == 0) {
	_Logger->WriteWarning(
	    ComponentName(), "unable to write to stream"
        );
	_TimeKeeper->Suspend();
    }
    
    if (WrittenLength != PacketSize) {
	_Logger->WritePanic(
	    ComponentName(), "stream I/O error (internal)"
        );
	_InputDataStream->Flush(Packet);
	return 0;
    }

    _InputDataStream->Flush(Packet);

    return (PacketSize > 0);
}
