/* KinokoEventCounterCom.cc */
/* Created by Enomoto Sanshiro on 20 May 2001. */
/* Last updated by Enomoto Sanshiro on 20 May 2001. */


#include <iostream>
#include <strstream>
#include "KinokoDataStreamFormatter.hh"
#include "KinokoEventCounterCom.hh"

using namespace std;


TKinokoEventCounterCom::TKinokoEventCounterCom(void)
: TKinokoStreamSinkComponent("KinokoEventCounterCom")
{
    _MaxNumberOfEvents = 0;
    _StreamScanner = new TKinokoDataStreamScanner();
}

TKinokoEventCounterCom::~TKinokoEventCounterCom()
{
    delete _StreamScanner;
}

void TKinokoEventCounterCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamSinkComponent::BuildDescriptor(Descriptor);

    TKcomEventDeclaration SetNumberOfEventsEvent("setNumberOfEvents");
    SetNumberOfEventsEvent.AddArgument(TKcomPropertyDeclaration(
        "number_of_events", TKcomPropertyDeclaration::Type_Long
    ));
    Descriptor.RegisterEventSlot(
	EventId_SetNumberOfEvents, SetNumberOfEventsEvent
    );

    TKcomEventDeclaration AlarmEvent("alarm");
    Descriptor.RegisterEventSource(EventId_Alarm, AlarmEvent);
}

int TKinokoEventCounterCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetNumberOfEvents:
	Result = ProcessSetNumberOfEventsEvent(Event);
	break;

      default:
	Result = TKinokoStreamSinkComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoEventCounterCom::ProcessSetNumberOfEventsEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setNumberOfEvents(): too few argument"
        );
	return 0;
    }

    istrstream NumberOfEventsStream(Event.ArgumentList()[0].c_str());
    if (! (NumberOfEventsStream >> _MaxNumberOfEvents)) {
	_Logger->WriteError(
	    ComponentName(), "setNumberOfEvents(): invalid argument"
        );
	return 0;
    }

    return 1;
}

void TKinokoEventCounterCom::EmitAlarmEvent(void)
{
    TKcomEvent Event;
    EmitEventOneWay(EventId_Alarm, Event);
}

void TKinokoEventCounterCom::Construct(void) throw(TKinokoException)
{
    _EventCount = 0;
}

void TKinokoEventCounterCom::Destruct(void) throw(TKinokoException)
{
}

int TKinokoEventCounterCom::ProcessData(void)throw(TKinokoException)
{
    if (! _InputDataStream->HasData()) {
	_TimeKeeper->Suspend();
        return 0;
    }

    void* Data;
    int DataSize = _InputDataStream->NextEntry(Data);
    if (DataSize <= 0) {
	_TimeKeeper->Suspend();
	return 0;
    }

    _StreamScanner->CorrectByteOrder(Data, DataSize);

    static const int Trailer_Event = TKinokoDataStreamScanner::Trailer_Event;
    if (_StreamScanner->IsDataDescriptorPacket(Data)) {
	;
    }
    else if (_StreamScanner->IsCommandPacket(Data)) {
	int CommandValue = _StreamScanner->CommandValueOf(Data);
	_StreamCommandProcessor->ProcessCommand(CommandValue);
    }
    else if (
	_StreamScanner->IsTrailerPacket(Data) &&
	(_StreamScanner->TrailerValueOf(Data) == Trailer_Event)
    ){
	_EventCount++;

	if (_EventCount % _MaxNumberOfEvents == 0) {
	    _Logger->WriteNotice(
		ComponentName(), "event counts reached the maximum value"
	    );
	    EmitAlarmEvent();
	}
    }     

    _InputDataStream->Flush(Data);

    return 1;
}
