/* KinokoRecorderCom.cc */
/* Created by Enomoto Sanshiro on 12 October 2000. */
/* Last updated by Enomoto Sanshiro on 27 April 2002. */


#include <strstream>
#include <string>
#include <vector>
#include "MushFileSystem.hh"
#include "MushMisc.hh"
#include "KinokoKdfRecorder.hh"
#include "KinokoRecorderCom.hh"

using namespace std;


static const string Comment = "Recorder: Data Storage Interface";


TKinokoRecorderCom::TKinokoRecorderCom(void)
: TKinokoStreamSinkComponent("KinokoRecorderCom")
{
    _Recorder = 0;

    _IsOverWriteAllowed = false;
    _DataFileAccessMode = -1;

    _IsDataCompressionEnabled = false;
    _IsIndexEnabled = false;
    _IsCommonHeaderDisabled = false;

    _RequiredSize = -1;
    _RequiredBlocks = -1;
    _LastDiskSpaceCheckTime = TMushDateTime::SecSinceEpoch();

    _ReportInterval = -1;
    _LastReportTime = TMushDateTime::SecSinceEpoch();

    _FileSystem = 0;
}

TKinokoRecorderCom::~TKinokoRecorderCom()
{
    delete _Recorder;
    delete _FileSystem;
}

void TKinokoRecorderCom::BuildDescriptor(TKcomComponentDescriptor& Descriptor)
{
    TKinokoStreamSinkComponent::BuildDescriptor(Descriptor);
    Descriptor.AddComment(Comment);

    TKcomEventDeclaration SetDataFileEvent("setDataFile");
    SetDataFileEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_SetDataFile, SetDataFileEvent);

    TKcomEventDeclaration FinishRecordingEvent("finishRecording");
    Descriptor.RegisterEventSource(EventId_FinishRecording, FinishRecordingEvent);

    TKcomEventDeclaration MakeDirectoryEvent("makeDirectory");
    MakeDirectoryEvent.AddArgument(TKcomPropertyDeclaration(
        "directory_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_MakeDirectory, MakeDirectoryEvent);

    TKcomEventDeclaration IsFileReadableEvent("isFileReadable");
    IsFileReadableEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_IsFileReadable, IsFileReadableEvent);

    TKcomEventDeclaration IsFileWritableEvent("isFileWritable");
    IsFileWritableEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_IsFileWritable, IsFileWritableEvent);

    TKcomEventDeclaration AllowOverWriteEvent("allowOverWrite");
    Descriptor.RegisterEventSlot(EventId_AllowOverWrite, AllowOverWriteEvent);

    TKcomEventDeclaration SetDataFileReadOnlyEvent("setDataFileReadOnly");
    Descriptor.RegisterEventSlot(EventId_SetDataFileReadOnly, SetDataFileReadOnlyEvent);

    TKcomEventDeclaration EnableDataCompressionEvent("enableDataCompression");
    Descriptor.RegisterEventSlot(EventId_EnableDataCompression, EnableDataCompressionEvent);

    TKcomEventDeclaration EnableIndexEvent("enableIndex");
    Descriptor.RegisterEventSlot(EventId_EnableIndex, EnableIndexEvent);

    TKcomEventDeclaration AddHeaderEntryEvent("addHeaderEntry");
    AddHeaderEntryEvent.AddArgument(TKcomPropertyDeclaration(
        "entry_name", TKcomPropertyDeclaration::Type_String
    ));
    AddHeaderEntryEvent.AddArgument(TKcomPropertyDeclaration(
        "value", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_AddHeaderEntry, AddHeaderEntryEvent);

    TKcomEventDeclaration ImportHeaderEvent("importHeader");
    ImportHeaderEvent.AddArgument(TKcomPropertyDeclaration(
        "file_name", TKcomPropertyDeclaration::Type_String
    ));
    Descriptor.RegisterEventSlot(EventId_ImportHeader, ImportHeaderEvent);

    TKcomEventDeclaration DisableCommonHeaderEvent("disableCommonHeader");
    Descriptor.RegisterEventSlot(EventId_DisableCommonHeader, DisableCommonHeaderEvent);

    TKcomEventDeclaration RequireDiskSpaceEvent("requireDiskSpace");
    RequireDiskSpaceEvent.AddArgument(TKcomPropertyDeclaration(
        "size", TKcomPropertyDeclaration::Type_Float
    ));
    Descriptor.RegisterEventSlot(EventId_RequireDiskSpace, RequireDiskSpaceEvent);

    TKcomEventDeclaration SetReportIntervalEvent("setReportInterval");
    SetReportIntervalEvent.AddArgument(TKcomPropertyDeclaration(
        "interval_sec", TKcomPropertyDeclaration::Type_Int
    ));
    Descriptor.RegisterEventSlot(EventId_SetReportInterval, SetReportIntervalEvent);

    TKcomEventDeclaration DiskSpaceExhaustedEvent("diskSpaceExhausted");
    Descriptor.RegisterEventSource(EventId_DiskSpaceExhausted, DiskSpaceExhaustedEvent);

}

int TKinokoRecorderCom::ProcessEvent(int EventId, TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    int Result = 0;

    switch (EventId) {
      case EventId_SetDataFile:
	Result = ProcessSetDataFileEvent(Event);
	break;

      case EventId_MakeDirectory:
	Result = ProcessMakeDirectoryEvent(Event, EventResponse);
	break;

      case EventId_IsFileReadable:
	Result = ProcessIsFileReadableEvent(Event, EventResponse);
	break;

      case EventId_IsFileWritable:
	Result = ProcessIsFileWritableEvent(Event, EventResponse);
	break;

      case EventId_AllowOverWrite:
	Result = ProcessAllowOverWriteEvent(Event);
	break;

      case EventId_SetDataFileReadOnly:
	Result = ProcessSetDataFileReadOnlyEvent(Event);
	break;

      case EventId_EnableDataCompression:
	Result = ProcessEnableDataCompressionEvent(Event);
	break;

      case EventId_EnableIndex:
	Result = ProcessEnableIndexEvent(Event);
	break;

      case EventId_AddHeaderEntry:
	Result = ProcessAddHeaderEntryEvent(Event);
	break;

      case EventId_ImportHeader:
	Result = ProcessImportHeaderEvent(Event);
	break;

      case EventId_DisableCommonHeader:
	Result = ProcessDisableCommonHeaderEvent(Event);
	break;

      case EventId_RequireDiskSpace:
	Result = ProcessRequireDiskSpaceEvent(Event);
	break;

      case EventId_SetReportInterval:
	Result = ProcessSetReportIntervalEvent(Event);
	break;

      default:
	Result = TKinokoStreamSinkComponent::ProcessEvent(
	    EventId, Event, EventResponse
	);
    }

    return Result;
}

int TKinokoRecorderCom::ProcessMakeDirectoryEvent(TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "makeDirectory(): too few arguments"
        );
    }
    else {
	string DirectoryName = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "makeDirectory(): " + DirectoryName
	);

	try {
	    if (TMushFileSystem::MakeDirectory(DirectoryName)) {
		EventResponse.ReturnValue() = "1";
	    }
	}
	catch (TSystemCallException &e) {
	    _Logger->WriteError(
		ComponentName(), "makeDirectory(): " + e.Message()
            );

	    EventResponse.IsError() = 1;
	    EventResponse.ReturnValue() = e.Message();
	}
    }

    return 1;
}

int TKinokoRecorderCom::ProcessIsFileReadableEvent(TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "isFileReadable(): too few arguments"
        );
    }
    else {
	string FileName = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "isFileReadable(): " + FileName
	);

	if (! FileName.empty() && TMushFileAttribute(FileName).IsReadable()) {
	    EventResponse.ReturnValue() = "1";
	}
	else {
	    EventResponse.ReturnValue() = "0";
	}
    }

    return 1;
}

int TKinokoRecorderCom::ProcessIsFileWritableEvent(TKcomEvent& Event, TKcomEventResponse& EventResponse)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "isFileWritable(): too few arguments"
        );
    }
    else {
	string FilePathName = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "isFileWritable(): " + FilePathName
	);

	string ErrorMessage;
	TMushFileAttribute FileAttribute(FilePathName);
	if (FilePathName.empty()) {
	    ;
	}
	if (FileAttribute.IsReadable()) {
	    if (FileAttribute.IsDirectory()) {
		ErrorMessage = "invalid file name: " + FilePathName;
	    }
	    if (! FileAttribute.IsWritable()) {
		ErrorMessage = "unable to overwrite file: " + FilePathName;
	    }
	}
	else {
	    string PathName = FileAttribute.PathName();
	    if (PathName.empty()) {
		PathName = TMushFileSystem::CurrentDirectory();
	    }

	    TMushFileAttribute DirectoryAttribute(PathName);
	    if (! DirectoryAttribute.IsReadable()) {
		ErrorMessage = "unable to read directory: " + PathName;
	    }
	    if (! DirectoryAttribute.IsWritable()) {
		ErrorMessage = "unable to make file at directory: " + PathName;
	    }
	}

	if (ErrorMessage.empty()) {
	    EventResponse.ReturnValue() = "1";
	}
	else {
	    EventResponse.ReturnValue() = "0";
	}
    }

    return 1;
}

int TKinokoRecorderCom::ProcessSetDataFileEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setDataFile(): too few argument[s]"
        );
    }
    else {
	_OutputFileName = Event.ArgumentList()[0];
    
	_Logger->WriteDebug(
	    ComponentName(), "setDataFile(): " + _OutputFileName
	);
    }

    return 1;
}

int TKinokoRecorderCom::ProcessAllowOverWriteEvent(TKcomEvent& Event)
{
    _IsOverWriteAllowed = true;

    _Logger->WriteDebug(ComponentName(), "allowOverWrite()");

    return 1;
}

int TKinokoRecorderCom::ProcessSetDataFileReadOnlyEvent(TKcomEvent& Event)
{
    _DataFileAccessMode = 0444;
    
    _Logger->WriteDebug(ComponentName(), "setDataFileReadOnly()");

    return 1;
}

int TKinokoRecorderCom::ProcessEnableDataCompressionEvent(TKcomEvent& Event)
{
    _IsDataCompressionEnabled = true;
    _Logger->WriteDebug(ComponentName(), "enableDataCompression()");

    return 1;
}

int TKinokoRecorderCom::ProcessEnableIndexEvent(TKcomEvent& Event)
{
    _IsIndexEnabled = true;
    _Logger->WriteDebug(ComponentName(), "enableIndex()");

    return 1;
}

int TKinokoRecorderCom::ProcessAddHeaderEntryEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 2) {
	_Logger->WriteError(
	    ComponentName(), "addHeaderEntry(): too few arguments"
        );
    }
    else {
	string EntryName = Event.ArgumentList()[0];
	string Value = Event.ArgumentList()[1];
	_HeaderEntryList.push_back(make_pair(EntryName, Value));
	
	_Logger->WriteDebug(
	    ComponentName(), "addHeaderEntry(): " + EntryName + ", " + Value
	);
    }

    return 1;
}

int TKinokoRecorderCom::ProcessImportHeaderEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "importHeader(): too few arguments"
        );
    }
    else {
	string FileName = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "importHeader(): " + FileName
	);

	_HeaderImportingFileList.push_back(FileName);
    }

    return 1;
}

int TKinokoRecorderCom::ProcessDisableCommonHeaderEvent(TKcomEvent& Event)
{
    _IsCommonHeaderDisabled = true;
    _Logger->WriteDebug(ComponentName(), "disableCommonHeader()");

    return 1;
}

int TKinokoRecorderCom::ProcessRequireDiskSpaceEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "requireDiskSpace(): too few arguments"
        );
    }
    else {
	string RequiredSizeString = Event.ArgumentList()[0];
	_Logger->WriteDebug(
	    ComponentName(), "requireDiskSpace(): " + RequiredSizeString
	);

	if (! (istrstream(RequiredSizeString.c_str()) >> _RequiredSize)) {
            _RequiredSize = -1;
	    _Logger->WriteError(
		ComponentName(), "requireDiskSpace(): invalid argument"
	    );
	}
    }

    return 1;
}

int TKinokoRecorderCom::ProcessSetReportIntervalEvent(TKcomEvent& Event)
{
    if (Event.ArgumentList().size() < 1) {
	_Logger->WriteError(
	    ComponentName(), "setReportInterval(): too few arguments"
        );
	return 0;
    }
    string ReportIntervalString = Event.ArgumentList()[0];

    istrstream ReportIntervalStream(ReportIntervalString.c_str());
    if (! (ReportIntervalStream >> _ReportInterval)) {
	_Logger->WriteError(
	    ComponentName(), 
	    "setReportInterval(): invalid argument: " + ReportIntervalString
	);
	return 0;
    }

    _Logger->WriteDebug(
	ComponentName(), "setReportInterval(): " + ReportIntervalString
    );

    return 1;
}

void TKinokoRecorderCom::Construct(void) throw(TKinokoException)
{
    if (_OutputFileName.empty()) {
	_Logger->WriteWarning(
	    ComponentName(), 
	    "data file is not specified (null stream is used instead)"
        );
    }

    if (_RequiredSize >= 0) {
	try {
	    string Path = TMushFileAttribute(_OutputFileName).PathName();
	    _FileSystem = new TMushFileSystem(Path);
	    _RequiredBlocks = (long) (_RequiredSize / _FileSystem->BlockSize());
	}
	catch (TSystemCallException &e) {
	    _Logger->WriteError(ComponentName(), e.Message());
	}
    }

    _Recorder = new TKinokoKdfRecorder();
    _Recorder->AttachIO(InputStream(), OutputStream(), _Logger);
    _Recorder->SetStreamCommandProcessor(_StreamCommandProcessor);
    _StreamMonitor = _Recorder->DataStreamMonitor();

    _Recorder->SetDataFile(_OutputFileName);
    if (_IsOverWriteAllowed || _OutputFileName.empty()) {
	_Recorder->AllowOverWrite();
    }
    if (_DataFileAccessMode >= 0) {
	_Recorder->SetDataFileAccessMode(_DataFileAccessMode);
    }
    if (_IsDataCompressionEnabled) {
	_Recorder->EnableDataCompression();
    }
    if (_IsIndexEnabled) {
	_Recorder->EnableIndex();
    }

    for (unsigned i = 0; i < _HeaderImportingFileList.size(); i++) {
	_Recorder->ImportHeaderFrom(_HeaderImportingFileList[i]);
    }

    if (! _IsCommonHeaderDisabled) {
	string Creator = TypeName();
	string DateTime = TMushDateTime().AsString("%Y-%m-%d %H:%M:%S %Z");
	string UserName = TMushUser().Name();
	string Host = TMushLocalHost::HostName();
	string Directory = TMushFileSystem::CurrentDirectory();
	_Recorder->AddHeaderEntry("Creator", Creator);
	_Recorder->AddHeaderEntry("DateTime", DateTime);
	_Recorder->AddHeaderEntry("UserName", UserName);
	_Recorder->AddHeaderEntry("Host", Host);
	_Recorder->AddHeaderEntry("Directory", Directory);
    }

    for (unsigned i = 0; i < _HeaderEntryList.size(); i++) {
	string EntryName = _HeaderEntryList[i].first;
	string Value = _HeaderEntryList[i].second;
	_Recorder->AddHeaderEntry(EntryName, Value);
    }

    try {
	_Recorder->Construct(ComponentName(), _InputDataStream);

	if (_IsOutputStreamUsed) {
	    _Recorder->SetChainedOutput(_OutputDataStream);
	}
    }
    catch (TKinokoException &e) {
	_Logger->WriteError(ComponentName(), e.Message());

	delete _Recorder;
	_Recorder = new TKinokoNullRecorder();
	_Recorder->AttachIO(InputStream(), OutputStream(), _Logger);
	_Recorder->SetStreamCommandProcessor(_StreamCommandProcessor);
	_Recorder->Construct(ComponentName(), _InputDataStream);
    }
}

void TKinokoRecorderCom::Destruct(void) throw(TKinokoException)
{
    _Recorder->Destruct();

    delete _Recorder;
    delete _FileSystem;
    _Recorder = 0;
    _FileSystem = 0;

    _OutputFileName = "";
    _IsOverWriteAllowed = false;
    _DataFileAccessMode = -1;
    _IsDataCompressionEnabled = false;
    _IsIndexEnabled = false;
    _RequiredSize = -1;
    _RequiredBlocks = -1;
    _LastDiskSpaceCheckTime = TMushDateTime::SecSinceEpoch();

    _HeaderEntryList.erase(_HeaderEntryList.begin(), _HeaderEntryList.end());
}

int TKinokoRecorderCom::ProcessData(void) throw(TKinokoException)
{
    int Result = _Recorder->ProcessData();
    if (Result <= 0) {
        _TimeKeeper->Suspend();
    }

    long Time = TMushDateTime::SecSinceEpoch();

    if ((_RequiredBlocks >= 0) && (Time - _LastDiskSpaceCheckTime > 10)) {
	if (_FileSystem->NumberOfAvailableBlocks() < _RequiredBlocks) {
	    TKcomEvent Event;
	    EmitEventOneWay(EventId_DiskSpaceExhausted, Event);

	    _RequiredBlocks = -1;
	}
	_LastDiskSpaceCheckTime = Time;
    }

    if ((_ReportInterval > 0) && (Time - _LastReportTime >= _ReportInterval)) {
	ReportStatistics();
	_LastReportTime = Time;
    }

    return Result;
}

void TKinokoRecorderCom::OnStop(void) throw(TKinokoException)
{
    _Logger->WriteRemarkable(ComponentName(), "data recording completed");

    TKcomEvent Event;
    EmitEventOneWay(EventId_FinishRecording, Event);
}

void TKinokoRecorderCom::ReportStatistics(void) throw(TKinokoException)
{
    char Message[512];
    ostrstream MessageStream(Message, sizeof(Message));
    
    MessageStream <<
	"written(size:packets)/time: " << _StreamMonitor->ProcessedDataSize() <<
	":" << _StreamMonitor->NumberOfProcessedPackets() << 
	"/" << _StreamMonitor->PassedTime() <<
    ends;

    _Logger->WriteNotice(ComponentName(), Message);

    _StreamMonitor->Clear();
}
