/* KinokoController.cc */
/* Created by Enomoto Sanshiro on 7 December 2000. */
/* Last updated by Enomoto Sanshiro on 22 August 2002. */


#include <string>
#include <iostream>
#include <strstream>
#include "KinokoPlatform.hh"
#include "KinokoControlInputWatcher.hh"
#include "KinokoController.hh"

using namespace std;


TKinokoController::TKinokoController(istream& InputStream, ostream& OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry)
: _InputStream(InputStream), _OutputStream(OutputStream)
{
    _EventEmitter = EventEmitter;
    _Registry = Registry;

    _InputWatcher = new TKinokoControlInputWatcher(_InputStream, ';');
    _InputWatcher->Start();
}

TKinokoController::~TKinokoController()
{
    delete _InputWatcher;
}

int TKinokoController::ProcessInput(void)
{
    if (_InputStream.eof() || (! _InputWatcher->IsDataAvailable())) {
	return 0;
    }

    const string& Line = _InputWatcher->Line();
    ProcessCommand(Line);

    _InputWatcher->Flush();
    
    return 1;
}

void TKinokoController::ProcessCommand(const string& Line)
{
    istrstream InputStream(Line.c_str());

    string Method, Command, ParameterList;
    getline(InputStream >> ws >> Method >> Command >> ws, ParameterList);

    while (! ParameterList.empty()) {
	int NameLength = ParameterList.find_first_of('=');
	int ParameterLength = ParameterList.find_first_of('&');
	int ValueLength = ParameterLength - NameLength - 1;
	if (NameLength > 0) {
	    string Name = ParameterList.substr(0, NameLength);
	    string Value = ParameterList.substr(NameLength + 1, ValueLength);

	    _Registry->SetValue("control/" + Name, DecodeEscape(Value));
	}
	if (ParameterLength > 0) {
	    ParameterList.erase(0, ParameterLength + 1);
	}
	else {
	    ParameterList.erase(0, ParameterLength);
	}
    }

    if (Method == ".invoke") {
	if (! Command.empty()) {
	    vector<string> ArgumentList;
	    _EventEmitter->EmitEvent(Command, ArgumentList);
	}
    }
    else {
	//... error: unknown method
	;
    }
}

void TKinokoController::Open(const string& FileName)
{
    _OutputStream << ".open " << FileName << ";" << endl;
    _OutputStream << ".start;" << endl;
}

void TKinokoController::ChangeState(const string& StateName)
{
    _OutputStream << ".changeState " << StateName << ";" << endl;
}

void TKinokoController::SetWidgetProperty(const string& WidgetName, const string& Value)
{
    _OutputStream << ".set " << WidgetName << " " << Value << ";" << endl;
}

void TKinokoController::ExecuteAction(const string& ActionName)
{
    _OutputStream << ".execute " << ActionName << ";" << endl;
}

void TKinokoController::OpenPopup(const string& Type, const string& ActionList, const string& Message)
{
    _OutputStream << ".openPopup ";
    _OutputStream << Type << ", " << ActionList << ", " << Message;
    _OutputStream << ";" << endl;
}

string TKinokoController::OpenQueryPopup(const string& Type, const string& ActionList, const string& Message)
{
    _OutputStream << ".openQueryPopup ";
    _OutputStream << Type << ", " << ActionList << ", " << Message;
    _OutputStream << ";" << endl;

    string Reply;
    while (1) {
	if (_InputWatcher->WaitData()) {
	    Reply = _InputWatcher->Line();
	    _InputWatcher->Flush();

	    if ((! Reply.empty()) && (Reply[0] == '.')) {
		ProcessCommand(Reply);
	    }
	    else {
		break;
	    }
	}
    }

    return Reply;
}

void TKinokoController::Save(const string& FileName)
{
    _OutputStream << ".save " << FileName << ";" << endl;
}

void TKinokoController::Load(const string& FileName)
{
    _OutputStream << ".load " << FileName << ";" << endl;
}

void TKinokoController::Quit(void)
{
    _InputWatcher->Stop();

    _OutputStream << ".stop;" << endl;
    _OutputStream << ".quit;" << endl;
}

string TKinokoController::DecodeEscape(const string& Value)
{
    string Result;
    bool IsEscaped = false;
    for (unsigned i = 0; i < Value.size(); i++) {
	if (! IsEscaped && (Value[i] == '%')) {
	    IsEscaped = true;
	    continue;
	}
	if (! IsEscaped) {
	    Result += Value[i];
	    continue;
	}

	switch (Value[i]) {
          case 'p':
	    Result += '%';
	    break;
          case 'a':
	    Result += '&';
	    break;
          case 'e':
	    Result += '=';
	    break;
          case 's':
	    Result += ';';
	    break;
          default:
	    Result += Value[i];
	}

	IsEscaped = false;
    }

    return Result;
}
