/* KinokoLogger.cc */
/* Created by Enomoto Sanshiro on 11 October 2000. */
/* Last updated by Enomoto Sanshiro on 7 March 2002. */


#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include "MushMisc.hh"
#include "KinokoLogger.hh"

using namespace std;



TKinokoLogMessage::TKinokoLogMessage(int LogLevel, const string& Source, const string& Contents)
{
    _LogLevel = LogLevel;
    _Date = TMushDateTime().AsString("%Y-%m-%d %H:%M:%S %Z");
    _Source = Source;
    _Contents = Contents;

    switch (_LogLevel) {
      case LogLevel_Debug:
	_LogLevelName = "debug";
	_LogLevelSymbol = "D";
	break;
      case LogLevel_Notice:
	_LogLevelName = "notice";
	_LogLevelSymbol = "N";
	break;
      case LogLevel_Remarkable:
	_LogLevelName = "remarkable";
	_LogLevelSymbol = "R";
	break;
      case LogLevel_Warning:
	_LogLevelName = "warning";
	_LogLevelSymbol = "W";
	break;
      case LogLevel_Error:
	_LogLevelName = "error";
	_LogLevelSymbol = "E";
	break;
      case LogLevel_Panic:
	_LogLevelName = "panic";
	_LogLevelSymbol = "P";
	break;
      default:
	_LogLevelName = "?";
	_LogLevelSymbol = "?";
	break;
    }
}

TKinokoLogMessage::~TKinokoLogMessage()
{
}

int TKinokoLogMessage::LogLevel(void) const
{
    return _LogLevel;
}

const string& TKinokoLogMessage::LogLevelName(void) const
{
    return _LogLevelName;
}

const string& TKinokoLogMessage::LogLevelSymbol(void) const
{
    return _LogLevelSymbol;
}

const string& TKinokoLogMessage::Date(void) const
{
    return _Date;
}

const string& TKinokoLogMessage::Source(void) const
{
    return _Source;
}

const string& TKinokoLogMessage::Contents(void) const
{
    return _Contents;
}

string TKinokoLogMessage::Message(void) const
{
    return _Source + ": " + _Contents;
}

string TKinokoLogMessage::LongMessage(void) const
{
    return _LogLevelSymbol + " " + _Date + ", " + Message();
}



TKinokoLogger::TKinokoLogger(ostream* OutputStream)
{
    _OutputStream = OutputStream;
    _LogFile = 0;
}

TKinokoLogger::~TKinokoLogger()
{
}

void TKinokoLogger::SetLogFile(TKinokoLogFile* LogFile)
{
    _LogFile = LogFile;
}

void TKinokoLogger::AddMessageHandler(TKinokoLogMessageHandler* Handler)
{
    _MessageHandlerList.push_back(Handler);
}

int TKinokoLogger::Write(int LogLevel, const string& Source, const string& Contents)
{
    TKinokoLogMessage LogMessage(LogLevel, Source, Contents);

    if (_LogFile) {
	_LogFile->Write(LogMessage);
    }

    ProcessLogMessage(LogMessage);

    for (unsigned i = 0; i < _MessageHandlerList.size(); i++) {
	_MessageHandlerList[i]->ProcessLogMessage(LogMessage);
    }

    return 0;
}

int TKinokoLogger::WriteDebug(const string& Source, const string& Contents)
{
    return Write(TKinokoLogMessage::LogLevel_Debug, Source, Contents);
}

int TKinokoLogger::WriteNotice(const string& Source, const string& Contents)
{
    return Write(TKinokoLogMessage::LogLevel_Notice, Source, Contents);
}

int TKinokoLogger::WriteRemarkable(const string& Source, const string& Contents)
{
    return Write(TKinokoLogMessage::LogLevel_Remarkable, Source, Contents);
}

int TKinokoLogger::WriteWarning(const string& Source, const string& Contents)
{
    return Write(TKinokoLogMessage::LogLevel_Warning, Source, Contents);
}

int TKinokoLogger::WriteError(const string& Source, const string& Contents)
{
    return Write(TKinokoLogMessage::LogLevel_Error, Source, Contents);
}

int TKinokoLogger::WritePanic(const string& Source, const string& Contents)
{
    return Write(TKinokoLogMessage::LogLevel_Panic, Source, Contents);
}

void TKinokoLogger::ProcessLogMessage(const TKinokoLogMessage& LogMessage)
{
    *_OutputStream << LogMessage.LongMessage() << endl;
}



TKinokoKoapLogger::TKinokoKoapLogger(ostream* OutputStream)
: TKinokoLogger(OutputStream)
{
}

TKinokoKoapLogger::~TKinokoKoapLogger()
{
}

void TKinokoKoapLogger::ProcessLogMessage(const TKinokoLogMessage& LogMessage)
{
    int LogLevel = LogMessage.LogLevel();

    string ColorString;
    switch (LogLevel) {
      case TKinokoLogMessage::LogLevel_Debug:
	ColorString = "gray";
	break;
      case TKinokoLogMessage::LogLevel_Notice:
	ColorString = "black";
	break;
      case TKinokoLogMessage::LogLevel_Remarkable:
	ColorString = "blue";
	break;
      case TKinokoLogMessage::LogLevel_Warning:
	ColorString = "green";
	break;
      case TKinokoLogMessage::LogLevel_Error:
	ColorString = "red";
	break;
      case TKinokoLogMessage::LogLevel_Panic:
	ColorString = "red";
	break;
      default:
	ColorString = "red";
	break;
    }

    *_OutputStream << ".set color " << ColorString << ";" << endl;
    *_OutputStream << ".write " << LogMessage.LongMessage()  << ";" << endl;

    if (
	(LogLevel == TKinokoLogMessage::LogLevel_Error) || 
	(LogLevel == TKinokoLogMessage::LogLevel_Panic)
    ){
	if (_StatusBarMessage.empty()) {
	    _StatusBarMessage = "ERROR: " + LogMessage.Message();
	    *_OutputStream << ".writeStatus ";
	    *_OutputStream << _StatusBarMessage << ";" << endl;
	}
    }
}



TKinokoLogMessageHandler::TKinokoLogMessageHandler(void)
{
}

TKinokoLogMessageHandler::~TKinokoLogMessageHandler()
{
}

void TKinokoLogMessageHandler::ProcessLogMessage(const TKinokoLogMessage& LogMessage)
{
    switch (LogMessage.LogLevel()) {
      case TKinokoLogMessage::LogLevel_Debug:
        OnDebug(LogMessage.Message());
	break;
      case TKinokoLogMessage::LogLevel_Notice:
        OnNotice(LogMessage.Message());
	break;
      case TKinokoLogMessage::LogLevel_Remarkable:
        OnRemarkable(LogMessage.Message());
	break;
      case TKinokoLogMessage::LogLevel_Warning:
        OnWarning(LogMessage.Message());
	break;
      case TKinokoLogMessage::LogLevel_Error:
        OnError(LogMessage.Message());
	break;
      case TKinokoLogMessage::LogLevel_Panic:
        OnPanic(LogMessage.Message());
	break;
      default:
	break;
    }
}




TKinokoLogFile::TKinokoLogFile(void)
{
    _File = new ofstream("/dev/null");
}

TKinokoLogFile::~TKinokoLogFile()
{
    delete _File;
}

void TKinokoLogFile::Open(const string& FileName)
{
    delete _File;
    _File = new ofstream(FileName.c_str());

    if (! *_File) {
	delete _File;
	_File = new ofstream("/dev/null");
    }
}

void TKinokoLogFile::Close(void)
{
    delete _File;
    _File = new ofstream("/dev/null");
}

void TKinokoLogFile::Write(const TKinokoLogMessage& LogMessage)
{
    *_File << LogMessage.LongMessage() << endl;
}




TKinokoXmlLogFile::TKinokoXmlLogFile(void)
{
}

TKinokoXmlLogFile::~TKinokoXmlLogFile()
{
}

void TKinokoXmlLogFile::Open(const string& FileName)
{
    TKinokoLogFile::Open(FileName);

    *_File << "<?xml version=\"1.0\"?>" << endl;
    *_File << endl;
    *_File << "<log-message>" << endl;
}

void TKinokoXmlLogFile::Close(void)
{
    *_File << "</log-message>" << endl;

    TKinokoLogFile::Close();
}

void TKinokoXmlLogFile::Write(const TKinokoLogMessage& LogMessage)
{
    *_File << "  <log";
    *_File << " level=\"" << LogMessage.LogLevelName() << "\"";
    *_File << " date=\"" << LogMessage.Date() << "\"";
    *_File << " source=\"" << LogMessage.Source() << "\"";
    *_File << ">";

    const char* Contents = LogMessage.Contents().c_str();
    while (*Contents != '\0') {
	if (*Contents == '&') {
	    *_File << "&amp;";
	}
	else if (*Contents == '<') {
	    *_File << "&lt;";
	}
	else if (*Contents == '>') {
	    *_File << "&gt;";
	}
	else {
	    *_File << *Contents;
	}
	Contents++;
    }

    *_File << "</log>" << endl;
}
