/* KinokoDataSection.hh */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 19 November 2000. */


#ifndef __KinokoDataSection_hh__
#define __KinokoDataSection_hh__


#include <iostream>
#include <string>
#include "RoomConfig.hh"
#include "ParaTokenizer.hh"
#include "KinokoDefs.hh"
#include "KinokoDataStreamFormatter.hh"

class TKinokoDataSource;


class TKinokoDataSection {
  public:
    TKinokoDataSection(TKinokoDataSource* DataSource, const std::string& SectionName);
    TKinokoDataSection(TKinokoDataSource* DataSource, const std::string& SectionName, int SectionId);
    virtual ~TKinokoDataSection();
    virtual void SetSectionId(int SectionId);
    virtual void SetParent(TKinokoDataSection* Parent);
    static int AutomaticSectionIdAssignmentBase(void);
  public:
    virtual int SectionType(void) const = 0;
    inline int SectionId(void) const;
    inline const std::string& SectionName(void) const;
    inline std::string FullSectionName(void) const;
    inline TKinokoDataSource* DataSource(void) const;
    inline TKinokoDataSection* Parent(void) const;
    virtual void ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException) = 0;
    virtual void WriteTo(std::ostream& os, const std::string& Indent) = 0;
  public:
    static std::string SectionNameOf(const std::string& SectionPath);
    static std::string SubSectionPathOf(const std::string& SectionPath);
  protected:
    int _SectionId;
    std::string _SectionName;
    TKinokoDataSource* _DataSource;
    TKinokoDataSection* _Parent;
  private:
    static int _NextSectionId;
  public:
    enum TSectionType {
	SectionType_Block,
	SectionType_Indexed,
	SectionType_Tagged,
	SectionType_Nested,
	_NumberOfSectionTypes
    };
    enum TSpecialSectionId {
        SectionId_Unknown = -1,
	_NumberOfSpecialSectionIds
    };
};


class TKinokoDataSectionFactory {
  protected:
    TKinokoDataSectionFactory(void);
  public:
    virtual ~TKinokoDataSectionFactory();
    static TKinokoDataSectionFactory* GetInstance(void);
    virtual TKinokoDataSection* ReadFrom(TParaTokenizer& Tokenizer, TKinokoDataSource* DataSource) throw(TKinokoException);
  protected:
    static TKinokoDataSectionFactory* _Instance;
};


class TKinokoDataSectionFormatter {
  public:
    TKinokoDataSectionFormatter(TKinokoDataSection* DataSection) throw(TKinokoException);
    virtual ~TKinokoDataSectionFormatter();
    inline int HeaderSize(void) const;
    inline int PacketSizeFor(int DataSize) const;
    inline void* DataAreaOf(void* Buffer) const;
    inline void WriteHeaderTo(void* Buffer, int DataSize) const;
  protected:
    long _DataSourceId;
    int _SectionId;
  public:
    enum {
	Offset_DataSourceId = TKinokoDataStreamFormatter::Offset_DataSourceId,
	Offset_PacketType = TKinokoDataStreamFormatter::Offset_PacketType,
	Offset_DataAreaSize = TKinokoDataStreamFormatter::Offset_Contents,
	Offset_DataArea
    };
    enum {
	_PacketType = TKinokoDataStreamFormatter::PacketType_DataSection
    };
};


class TKinokoDataSectionScanner {
  public:
    TKinokoDataSectionScanner(void);
    virtual ~TKinokoDataSectionScanner();
    static inline int SectionIdOf(void* Buffer);
    static inline int PacketSizeOf(void* Buffer);
    static inline int DataSizeOf(void* Buffer);
    static inline void* DataAreaOf(void* Buffer);
    static inline int DataWordOf(void* Buffer, int WordIndex);
  public:
    enum {
	Offset_PacketType = TKinokoDataSectionFormatter::Offset_PacketType,
	Offset_DataAreaSize = TKinokoDataSectionFormatter::Offset_DataAreaSize,
	Offset_DataArea = TKinokoDataSectionFormatter::Offset_DataArea
    };
    enum {
	_SectionIdMask = ~TKinokoDataStreamFormatter::PacketTypeMask
    };
};



inline int TKinokoDataSection::SectionId(void) const
{
    return _SectionId;
}

inline const std::string& TKinokoDataSection::SectionName(void) const
{
    return _SectionName;
}

inline std::string TKinokoDataSection::FullSectionName(void) const
{
    if (_Parent == 0) {
	return _SectionName;
    }
    else {
	return _Parent->FullSectionName() + ":" + _SectionName;
    }
}

inline TKinokoDataSource* TKinokoDataSection::DataSource(void) const
{
    return _DataSource;
}

inline TKinokoDataSection* TKinokoDataSection::Parent(void) const
{
    return _Parent;
}



inline int TKinokoDataSectionFormatter::HeaderSize(void) const
{
    return sizeof(U32bit) * Offset_DataArea;
}

inline int TKinokoDataSectionFormatter::PacketSizeFor(int DataSize) const
{
    return sizeof(U32bit) * Offset_DataArea + ((DataSize - 1) / 4 + 1) * 4;
}

inline void* TKinokoDataSectionFormatter::DataAreaOf(void* Buffer) const
{
    return (U32bit*) Buffer + Offset_DataArea;
}

inline void TKinokoDataSectionFormatter::WriteHeaderTo(void* Buffer, int DataSize) const
{
    ((U32bit*) Buffer)[Offset_DataSourceId] = _DataSourceId;
    ((U32bit*) Buffer)[Offset_PacketType] = _PacketType | _SectionId;
    ((U32bit*) Buffer)[Offset_DataAreaSize] = DataSize;
}



inline int TKinokoDataSectionScanner::SectionIdOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & _SectionIdMask;
}

inline int TKinokoDataSectionScanner::PacketSizeOf(void* Buffer)
{
    return (sizeof(U32bit) * Offset_DataArea) + DataSizeOf(Buffer);
}

inline int TKinokoDataSectionScanner::DataSizeOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_DataAreaSize];
}

inline void* TKinokoDataSectionScanner::DataAreaOf(void* Buffer)
{
    return (U32bit*) Buffer + Offset_DataArea;
}

inline int TKinokoDataSectionScanner::DataWordOf(void* Buffer, int WordIndex)
{
    return ((U32bit*) Buffer)[Offset_DataArea + WordIndex];
}

#endif
