/* KinokoDataSource.cc */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 25 January 2002. */


#include <iostream>
#include <string>
#include <vector>
#include <map>
#include "ParaTokenizer.hh"
#include "KinokoDataSection.hh"
#include "KinokoDataSectionContainer.hh"
#include "KinokoDataSource.hh"

using namespace std;



int TKinokoDataSource::AutomaticDataSourceIdAssignmentBase(void)
{
    return 1024;
}

TKinokoDataSource::TKinokoDataSource(const string& DataSourceName, long DataSourceId)
{
    _DataSourceName = DataSourceName;
    _DataSourceId = DataSourceId;
}

TKinokoDataSource::~TKinokoDataSource()
{
}

void TKinokoDataSource::SetName(const string& DataSourceName, long DataSourceId)
{
    _DataSourceName = DataSourceName;
    _DataSourceId = DataSourceId;
}

void TKinokoDataSource::AddAttribute(const string& Name, const string& Value)
{
    _AttributeList.push_back(make_pair(Name, Value));
}

bool TKinokoDataSource::GetAttributeOf(const std::string& Name, std::string& Value)
{
    bool Result = false;
    for (unsigned i = 0; i < _AttributeList.size(); i++) {
	if (_AttributeList[i].first == Name) {
	    Value = _AttributeList[i].second;
	    Result = true;
	    break;
	}
    }

    return Result;
}

void TKinokoDataSource::ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    try {
	Tokenizer.Next().MustBe("{");
	
	while (1) {
	    TParaToken Token = Tokenizer.LookAhead();
	    if (Token.Is("attribute")) {
		ParseAttribute(Tokenizer);
	    }
	    else if (Token.Is("section")) {
		ParseSection(Tokenizer);
	    }
	    else {
		break;
	    }
	}
	
	Tokenizer.Next().MustBe("}");
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoDataSource::ReadFrom()",
	    "script exception: " + e.Message()
	);
    }
}

void TKinokoDataSource::ParseAttribute(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    string AttributeName;
    string AttributeValue;
    try {
	Tokenizer.Next().MustBe("attribute");
	AttributeName = Tokenizer.Next().AsString();
	Tokenizer.Next().MustBe("=");
	AttributeValue = Tokenizer.Next().RemoveQuotation().AsString();
	Tokenizer.Next().MustBe(";");
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoDataSource::ParseAttribute()",
	    "script exception: " + e.Message()
	);
    }

    AddAttribute(AttributeName, AttributeValue);
}

void TKinokoDataSource::ParseSection(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    TKinokoDataSection* DataSection;
    TKinokoDataSource* DataSource;
    TKinokoDataSectionFactory* SectionFactory;

    DataSource = this;
    SectionFactory = TKinokoDataSectionFactory::GetInstance();
    DataSection = SectionFactory->ReadFrom(Tokenizer, DataSource);
    
    AddDataSection(DataSection);
}

void TKinokoDataSource::WriteTo(ostream& os)
{
    os << "datasource \"" << _DataSourceName << "\"";
    os << "<" << _DataSourceId << ">" << endl;
    os << "{";

    for (unsigned i = 0; i < _AttributeList.size(); i++) {
	os << endl;
	os << "    " << "attribute " << _AttributeList[i].first;
	os << " = \"" <<  _AttributeList[i].second << "\";";
    }
    os << endl;

    string Indent;
    for (unsigned j = 0; j < _DataSectionList.size(); j++) {
	os << endl;
	_DataSectionList[j]->WriteTo(os, Indent = "    ");
    }

    os << "}" << endl;
}
