/* KinokoDataStreamFormatter.hh */
/* Created by Enomoto Sanshiro on 23 November 2000. */
/* Last updated by Enomoto Sanshiro on 23 November 2000. */


#ifndef __KinokoDataStreamFormatter_hh__
#define __KinokoDataStreamFormatter_hh__


#include "MushDataEncoder.hh"
#include "KinokoDefs.hh"
class TKinokoDataSource;


class TKinokoDataStreamFormatter {
  public:
    TKinokoDataStreamFormatter(TKinokoDataSource* DataSource);
    ~TKinokoDataStreamFormatter();
    inline TKinokoDataSource* DataSource(void);
    int DescriptorPacketSize(void);
    static inline int CommandPacketSize(void);
    static inline int TrailerPacketSize(void);
    void WriteDescriptor(void* Buffer);
    inline void WriteCommand(void* Buffer, int CommandValue);
    inline void WriteTrailer(void* Buffer, int TrailerValue);
  protected:
    TKinokoDataSource* _DataSource;
    int _DataSourceId;
    int _DescriptorLength;
  public:
    enum {
	Offset_DataSourceId = 0,
	Offset_PacketType = 1,
	Offset_Contents = 2
    };
    enum {
        ByteOrderCheckMask = 0xffff0000,
        DataSourceIdMask = 0x0000ffff
    };
    enum {
	PacketTypeMask = 0xffff0000,
	PacketType_Null = 0,
	PacketType_DataDescriptor = (0x0001 << 16),
	PacketType_Command = (0x0001 << 17),
	PacketType_Trailer = (0x0001 << 18),
	PacketType_DataSection = (0x0001 << 19)
    };
    enum {
	CommandMask = ~PacketTypeMask,
	Command_RunBegin = 0x0001,
	Command_RunEnd = 0x0002,
	Command_RunSuspend = 0x0003,
	Command_RunResume = 0x0004
    };
    enum {
	TrailerMask = ~PacketTypeMask,
	Trailer_Event = 0x0001,
	Trailer_Command = 0x0002,
	Trailer_Trap = 0x0003
    };
};


class TKinokoDataStreamScanner {
  public:
    TKinokoDataStreamScanner(void);
    ~TKinokoDataStreamScanner();
    static inline bool IsByteOrderReversed(void* Buffer);
    static inline bool IsDataDescriptorPacket(void* Buffer);
    static inline bool IsCommandPacket(void* Buffer);
    static inline bool IsTrailerPacket(void* Buffer);
    static inline bool IsDataPacket(void* Buffer);
    static inline bool IsNullPacket(void* Buffer);
    static inline int DataSourceIdOf(void* Buffer);
    static inline char* DataDescriptorOf(void* Buffer);
    static inline int CommandValueOf(void* Buffer);
    static inline int TrailerValueOf(void* Buffer);
    static void CorrectByteOrder(void* Buffer, int BufferSize);
  protected:
    static TMushBinaryDataEncoder* _DataEncoder;
  public:
    enum {
	Offset_DataSourceId = TKinokoDataStreamFormatter::Offset_DataSourceId,
	Offset_PacketType = TKinokoDataStreamFormatter::Offset_PacketType,
	Offset_Contents = TKinokoDataStreamFormatter::Offset_Contents
    };
    enum {
        ByteOrderCheckMask = TKinokoDataStreamFormatter::ByteOrderCheckMask,
	DataSourceIdMask = TKinokoDataStreamFormatter::DataSourceIdMask
    };
    enum {
	PacketTypeMask = TKinokoDataStreamFormatter::PacketTypeMask,
	PacketType_Null = TKinokoDataStreamFormatter::PacketType_Null,
	PacketType_DataDescriptor = TKinokoDataStreamFormatter::PacketType_DataDescriptor,
	PacketType_Command = TKinokoDataStreamFormatter::PacketType_Command,
	PacketType_Trailer = TKinokoDataStreamFormatter::PacketType_Trailer,
	PacketType_DataSection = TKinokoDataStreamFormatter::PacketType_DataSection
    };
    enum {
	CommandMask = TKinokoDataStreamFormatter::CommandMask,
	Command_RunBegin = TKinokoDataStreamFormatter::Command_RunBegin,
	Command_RunEnd = TKinokoDataStreamFormatter::Command_RunEnd,
	Command_RunSuspend = TKinokoDataStreamFormatter::Command_RunSuspend,
	Command_RunResume = TKinokoDataStreamFormatter::Command_RunResume
    };
    enum {
	TrailerMask = TKinokoDataStreamFormatter::TrailerMask,
	Trailer_Event = TKinokoDataStreamFormatter::Trailer_Event,
	Trailer_Command = TKinokoDataStreamFormatter::Trailer_Command,
	Trailer_Trap = TKinokoDataStreamFormatter::Trailer_Trap
    };
};


class TKinokoStreamCommandProcessor {
  public:
    TKinokoStreamCommandProcessor(void);
    virtual ~TKinokoStreamCommandProcessor();
    virtual void ProcessCommand(int CommandValue);
    inline int NumberOfRunningDataSources(void);
    inline int NumberOfSuspendingDataSources(void);
  protected:
    virtual void OnReceiveRunBeginPacket(void) {}
    virtual void OnReceiveRunEndPacket(void) {}
    virtual void OnReceiveRunSuspendPacket(void) {}
    virtual void OnReceiveRunResumePacket(void) {}
    virtual void OnReceiveFirstRunBeginPacket(void) {}
    virtual void OnReceiveLastRunEndPacket(void) {}
    virtual void OnReceiveLastRunSuspendPacket(void) {}
    virtual void OnReceiveFirstRunResumePacket(void) {}
  protected:
    int _NumberOfRunningDataSources;
    int _NumberOfSuspendingDataSources;
};



inline TKinokoDataSource* TKinokoDataStreamFormatter::DataSource(void)
{
    return _DataSource;
}

inline int TKinokoDataStreamFormatter::CommandPacketSize(void)
{
    return sizeof(U32bit) * 2;
}

inline int TKinokoDataStreamFormatter::TrailerPacketSize(void)
{
    return sizeof(U32bit) * 2;
}

inline void TKinokoDataStreamFormatter::WriteCommand(void* Buffer, int CommandValue)
{
    ((U32bit*) Buffer)[Offset_DataSourceId] = _DataSourceId & DataSourceIdMask;
    ((U32bit*) Buffer)[Offset_PacketType] = PacketType_Command | CommandValue;
}

inline void TKinokoDataStreamFormatter::WriteTrailer(void* Buffer, int TrailerValue)
{
    ((U32bit*) Buffer)[Offset_DataSourceId] = _DataSourceId & DataSourceIdMask;
    ((U32bit*) Buffer)[Offset_PacketType] = PacketType_Trailer | TrailerValue;
}



inline bool TKinokoDataStreamScanner::IsByteOrderReversed(void* Buffer)
{
    return (((U32bit*) Buffer)[Offset_DataSourceId] & ByteOrderCheckMask) != 0;
}

inline bool TKinokoDataStreamScanner::IsDataDescriptorPacket(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & PacketType_DataDescriptor;
}

inline bool TKinokoDataStreamScanner::IsCommandPacket(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & PacketType_Command;
}

inline bool TKinokoDataStreamScanner::IsTrailerPacket(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & PacketType_Trailer;
}

inline bool TKinokoDataStreamScanner::IsDataPacket(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & PacketType_DataSection;
}

inline bool TKinokoDataStreamScanner::IsNullPacket(void* Buffer)
{
    return (((U32bit*) Buffer)[Offset_PacketType]  == 0);
}

inline int TKinokoDataStreamScanner::DataSourceIdOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_DataSourceId] & DataSourceIdMask;
}

inline char* TKinokoDataStreamScanner::DataDescriptorOf(void* Buffer)
{
    return (char*) ((U32bit*) Buffer + Offset_Contents);
}

inline int TKinokoDataStreamScanner::CommandValueOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & CommandMask;
}

inline int TKinokoDataStreamScanner::TrailerValueOf(void* Buffer)
{
    return ((U32bit*) Buffer)[Offset_PacketType] & TrailerMask;
}


inline int TKinokoStreamCommandProcessor::NumberOfRunningDataSources(void)
{
    return _NumberOfRunningDataSources;
}

inline int TKinokoStreamCommandProcessor::NumberOfSuspendingDataSources(void)
{
    return _NumberOfSuspendingDataSources;
}

#endif
