/* KinokoIndexedDataSection.hh */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 19 November 2000. */


#ifndef __KinokoIndexedDataSection_hh__
#define __KinokoIndexedDataSection_hh__

#include <iostream>
#include <string>
#include "ParaTokenizer.hh"
#include "KinokoDataSection.hh"


class TKinokoIndexedDataSectionFormatter;
class TKinokoIndexedDataSectionScanner;


class TKinokoIndexedDataSection: public TKinokoDataSection {
  public:
    TKinokoIndexedDataSection(TKinokoDataSource* DataSource, const std::string& SectionName);
    TKinokoIndexedDataSection(TKinokoDataSource* DataSource, const std::string& SectionName, int SectionId);
    virtual ~TKinokoIndexedDataSection();
    virtual int SectionType(void) const;
    virtual void ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException);
    virtual void WriteTo(std::ostream& os, const std::string& Indent);
    inline TKinokoIndexedDataSectionFormatter* Formatter(void);
    inline TKinokoIndexedDataSectionScanner* Scanner(void);
  public:
    void SetWidth(int AddressWidth, int DataWidth);
    int AddressWidth(void) const;
    int DataWidth(void) const;
  protected:
    int _AddressWidth, _DataWidth;
    TKinokoIndexedDataSectionFormatter* _Formatter;
    TKinokoIndexedDataSectionScanner* _Scanner;
};


class TKinokoIndexedDataSectionFormatter: public TKinokoDataSectionFormatter {
  public:
    TKinokoIndexedDataSectionFormatter(TKinokoIndexedDataSection* DataSection);
    virtual ~TKinokoIndexedDataSectionFormatter();
    inline int DataSizeFor(int NumberOfElements) const;
    inline int WriteTo(void* Buffer, int Index, int Address, int Data) const;
  protected:
    int _AddressWidth, _DataWidth;
    bool _UsesTwoWords;
  public:
    enum {
	Offset_Base = TKinokoDataSectionFormatter::Offset_DataArea
    };
};


class TKinokoIndexedDataSectionScanner: public TKinokoDataSectionScanner {
  public:
    TKinokoIndexedDataSectionScanner(TKinokoIndexedDataSection* DataSection);
    virtual ~TKinokoIndexedDataSectionScanner();
    inline int NumberOfElements(void* Buffer) const;
    inline void ReadFrom(void* Buffer, int Index, int& Address, int& Data) const;
  protected:
    int _AddressWidth, _DataWidth;
    bool _UsesTwoWords;
    unsigned _DataBitMask;
  public:
    enum {
	Offset_DataAreaSize = TKinokoDataSectionFormatter::Offset_DataAreaSize,
	Offset_Base = TKinokoDataSectionFormatter::Offset_DataArea
    };
};



inline TKinokoIndexedDataSectionFormatter* TKinokoIndexedDataSection::Formatter(void)
{
    if (_Formatter == 0) {
	_Formatter = new TKinokoIndexedDataSectionFormatter(this);
    }

    return _Formatter;
}

inline TKinokoIndexedDataSectionScanner* TKinokoIndexedDataSection::Scanner(void)
{
    if (_Scanner == 0) {
	_Scanner = new TKinokoIndexedDataSectionScanner(this);
    }

    return _Scanner;
}



inline int TKinokoIndexedDataSectionFormatter::DataSizeFor(int NumberOfElements) const
{
    if (_UsesTwoWords) {
	return sizeof(U32bit) * (2 * NumberOfElements);
    }
    else {
	return sizeof(U32bit) * NumberOfElements;
    }
}

inline int TKinokoIndexedDataSectionFormatter::WriteTo(void* Buffer, int Index, int Address, int Data) const
{
    if (_UsesTwoWords) {
	((U32bit*) Buffer)[Offset_Base + 2 * Index] = Address;
	((U32bit*) Buffer)[Offset_Base + 2 * Index + 1] = Data;
	return 2 * sizeof(U32bit);
    }
    else {
	((U32bit*) Buffer)[Offset_Base + Index] = Address << _DataWidth | Data;
	return sizeof(U32bit);
    }
}



inline int TKinokoIndexedDataSectionScanner::NumberOfElements(void* Buffer) const
{
    if (_UsesTwoWords) {
	return ((U32bit*) Buffer)[Offset_DataAreaSize] / (2 * sizeof(U32bit));
    }
    else {
	return ((U32bit*) Buffer)[Offset_DataAreaSize] / sizeof(U32bit);
    }
}

inline void TKinokoIndexedDataSectionScanner::ReadFrom(void* Buffer, int Index, int& Address, int& Data) const
{
    if (_UsesTwoWords) {
	Address = ((U32bit*) Buffer)[Offset_Base + 2 * Index];
	Data = ((U32bit*) Buffer)[Offset_Base + 2 * Index + 1];
    }
    else {
	Address = ((U32bit*) Buffer)[Offset_Base + Index] >> _DataWidth;
	Data = ((U32bit*) Buffer)[Offset_Base + Index] & _DataBitMask;
    }
}


#endif
