/* KinokoTaggedDataSection.cc */
/* Created by Enomoto Sanshiro on 19 November 2000. */
/* Last updated by Enomoto Sanshiro on 19 November 2000. */


#include <iostream>
#include <string>
#include <vector>
#include <map>
#include "ParaTokenizer.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoTaggedDataSection.hh"

using namespace std;


TKinokoTaggedDataSection::TKinokoTaggedDataSection(TKinokoDataSource* DataSource, const string& SectionName)
: TKinokoDataSection(DataSource, SectionName)
{
    _Formatter = 0;
    _Scanner = 0;
    
    _NumberOfFields = 0;
}

TKinokoTaggedDataSection::TKinokoTaggedDataSection(TKinokoDataSource* DataSource, const string& SectionName, int SectionId)
: TKinokoDataSection(DataSource, SectionName, SectionId)
{
    _Formatter = 0;
    _Scanner = 0;

    _NumberOfFields = 0;
}

TKinokoTaggedDataSection::~TKinokoTaggedDataSection()
{
    delete _Formatter;
    delete _Scanner;

    for (unsigned j = 0; j < _ValueNameTableList.size(); j++) {
	delete _ValueNameTableList[j];
    }
}

int TKinokoTaggedDataSection::SectionType(void) const
{
    return TKinokoDataSection::SectionType_Tagged;
}

int TKinokoTaggedDataSection::AddField(const string& FieldName, int ValueWidth, TValueNameTable* ValueNameTable)
{
    int FieldIndex = _NumberOfFields;
    _NumberOfFields++;

    _FieldNameList.push_back(FieldName);
    _FieldNameTable[FieldName] = FieldIndex;

    _ValueWidthList.push_back(ValueWidth);

    _ValueNameTableList.push_back(ValueNameTable);

    return FieldIndex;
}

bool TKinokoTaggedDataSection::HasField(const string& FieldName)
{
    return (_FieldNameTable.count(FieldName) > 0);
}

int TKinokoTaggedDataSection::FieldIndexOf(const string& FieldName) throw(TKinokoException)
{
    if (_FieldNameTable.count(FieldName) == 0) {
	throw TKinokoException(
	    "TKinokoTaggedDataSection::FieldIndexOf()",
	    "unknown field name: " + _SectionName + "::" + FieldName
	);
    }
    
    return _FieldNameTable[FieldName];    
}

string TKinokoTaggedDataSection::FieldNameOf(int FieldIndex) throw(TKinokoException)
{
    if ((FieldIndex < 0) || (FieldIndex >= _NumberOfFields)) {
	throw TKinokoException(
	    "TKinokoTaggedDataSection::FieldNameOf()",
	    "bad field index"
	);
    }

    return _FieldNameList[FieldIndex];
}

int TKinokoTaggedDataSection::NumberOfFields(void)
{
    return _NumberOfFields;
}

void TKinokoTaggedDataSection::ReadFrom(TParaTokenizer& Tokenizer) throw(TKinokoException)
{
    try {
	TParaToken Token;
	Tokenizer.Next().MustBe("{");
	while ((Token = Tokenizer.Next()).Is("field")) {
	    string FieldName = Tokenizer.Next().RemoveQuotation().AsString();
	    Tokenizer.Next().MustBe(":");
	    Tokenizer.Next().MustBe("int");
	    Tokenizer.Next().MustBe("-");
	    int ValueWidth = Tokenizer.Next().AsLong();
	    Tokenizer.Next().MustBe("bit");

	    TValueNameTable* ValueNameTable = 0;
	    if ((Token = Tokenizer.Next()).Is("{")) {
		ValueNameTable = new TValueNameTable();
		long Value; 
		string ValueName;
		while ((Token = Tokenizer.Next()).IsNot("}")) {
		    Value = Token.AsLong();
		    Tokenizer.Next().MustBe("->");
		    ValueName = Tokenizer.Next().RemoveQuotation().AsString();
		    Tokenizer.Next().MustBe(";");

		    (*ValueNameTable)[Value] = ValueName;
		}
	    }
	    else {
		Token.MustBe(";");
	    }

	    AddField(FieldName, ValueWidth, ValueNameTable);
	}
	Token.MustBe("}");
    }
    catch (TScriptException &e) {
	throw TKinokoException(
	    "TKinokoTaggedDataSection::ReadFrom()",
	    "script exception: " + e.Message()
	);
    }
}

void TKinokoTaggedDataSection::WriteTo(ostream& os, const string& Indent)
{
    os << Indent << "section \"" << _SectionName << "\"<" << _SectionId << ">";
    os << ": tagged {";

    for (unsigned i = 0; i < _FieldNameList.size(); i++) {
	os << endl;
	os << Indent << "    " << "field \"" << _FieldNameList[i] << "\"";
	os << ": int-" << _ValueWidthList[i] << "bit";

	if (_ValueNameTableList[i] != 0) {
	    os << " {" << endl;

	    TValueNameTable::iterator ValueNameIterator;
	    for (
		ValueNameIterator = _ValueNameTableList[i]->begin();
		ValueNameIterator != _ValueNameTableList[i]->end();
		ValueNameIterator++
	    ){
		os << Indent << "    ""    ";
		os << (*ValueNameIterator).first << " -> ";
                os << "\"" << (*ValueNameIterator).second << "\";" << endl;
	    }

	    os << Indent << "    " << "}";
	}
	else {
	    os << ";";
	}
    }

    os << endl;
    os << Indent << "}" << endl;
}



TKinokoTaggedDataSectionFormatter::TKinokoTaggedDataSectionFormatter(TKinokoTaggedDataSection* DataSection)
: TKinokoDataSectionFormatter(DataSection)
{
    _TaggedDataSection = DataSection;
}

TKinokoTaggedDataSectionFormatter::~TKinokoTaggedDataSectionFormatter()
{
}



TKinokoTaggedDataSectionScanner::TKinokoTaggedDataSectionScanner(TKinokoTaggedDataSection* DataSection)
{
}

TKinokoTaggedDataSectionScanner::~TKinokoTaggedDataSectionScanner()
{
}
