/* KinokoDaqBuilder.cc */
/* Created by Enomoto Sanshiro on 9 October 1999. */
/* Last updated by Enomoto Sanshiro on 8 July 2001. */


#include <string>
#include <deque>
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoDevicePool.hh"
#include "KinokoTrigger.hh"
#include "KinokoTransaction.hh"
#include "KinokoPlatform.hh"
#include "KinokoStream.hh"
#include "KinokoDaqBuilder.hh"

using namespace std;


TKinokoDaqBuilder::TKinokoDaqBuilder(const string& DataSourceName, int DataSourceId, TKinokoOutputStream* OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry)
{
    _DataSource = new TKinokoDataSource(DataSourceName, DataSourceId);

    _DevicePool = new TKinokoDevicePool();
    _Trigger = new TKinokoTrigger();
    _Transaction = new TKinokoTransaction(_DataSource, OutputStream);

    _EventEmitter = EventEmitter;
    _Registry = Registry;

    _IsConstructed = false;
}

TKinokoDaqBuilder::~TKinokoDaqBuilder()
{
    delete _Transaction;
    delete _Trigger;
    delete _DevicePool;

    delete _DataSource;
}

void TKinokoDaqBuilder::Destruct(void) throw(TKinokoException)
{
    _DevicePool->Destroy();
}

TKinokoDevicePool* TKinokoDaqBuilder::DevicePool(void)
{
    if (! _IsConstructed) {
	_IsConstructed = true;
	Construct();
    }

    return _DevicePool;
}

TKinokoTransaction* TKinokoDaqBuilder::Transaction(void)
{
    if (! _IsConstructed) {
	_IsConstructed = true;
	Construct();
    }

    return _Transaction;
}

TKinokoTrigger* TKinokoDaqBuilder::Trigger(void)
{
    if (! _IsConstructed) {
	_IsConstructed = true;
	Construct();
    }

    return _Trigger;
}

TKinokoDataSource* TKinokoDaqBuilder::DataSource(void)
{
    return _DataSource;
}

void TKinokoDaqBuilder::RegisterDataSourceName(const string& Name, int DataSourceId)
{
    _DataSourceNameList.push_back(make_pair(Name, DataSourceId));
}

int TKinokoDaqBuilder::RegisterCrate(const string& Name, TRoomCrate* Crate)
{
    return _DevicePool->RegisterCrate(Name, Crate);
}

int TKinokoDaqBuilder::RegisterController(const string& Name, TRoomController* Controller)
{
    return _DevicePool->RegisterController(Name, Controller);
}

int TKinokoDaqBuilder::RegisterModule(const string& Name, TRoomModule* Module)
{
    return _DevicePool->RegisterModule(Name, Module);
}

int TKinokoDaqBuilder::RegisterReadoutUnit(const string& UnitName)
{
    return _DevicePool->RegisterDeviceName(UnitName);
}

int TKinokoDaqBuilder::RegisterRegister(const string& RegisterName, int* Register)
{
    return _DevicePool->RegisterRegister(RegisterName, Register);
}

int TKinokoDaqBuilder::RegisterDataRecord(const string& DataRecordName, TKinokoDataRecord* DataRecord)
{
    return _DevicePool->RegisterDataRecord(DataRecordName, DataRecord);
}

void TKinokoDaqBuilder::OpenReadoutSequence(int TriggerSourceId, int* TriggerIdRegister)
{
    TKinokoReadoutSequence* Sequence;
    if (TriggerIdRegister != 0) {
	Sequence = new TKinokoReadoutSequence(TriggerIdRegister);
    }
    else {
	Sequence = new TKinokoReadoutSequence();
    }

    _ReadoutSequenceStack.push_front(Sequence);
    _Transaction->AddReadoutSequence(TriggerSourceId, Sequence);
}

void TKinokoDaqBuilder::OpenReadoutSequence(const string& EventName, vector<int*> ArgumentRegisterList)
{
    TKinokoReadoutSequence* Sequence = new TKinokoReadoutSequence(
	ArgumentRegisterList
    );

    _ReadoutSequenceStack.push_front(Sequence);
    _Transaction->AddDaqEventSequence(EventName, Sequence);
}

void TKinokoDaqBuilder::CloseReadoutSequence(void)
{
    _ReadoutSequenceStack.pop_front();
}

void TKinokoDaqBuilder::OpenConditionalReadoutSequence(TKinokoReadoutCondition* Condition)
{
    if (_ReadoutSequenceStack.empty()) {
	throw TKinokoException("ReadoutCondition outside ReadoutSequence");
    }

    TKinokoReadoutSequence* ReadoutSequence;
    ReadoutSequence = new TKinokoConditionalReadoutSequence(Condition);

    AddReadoutAction(ReadoutSequence);
    _ReadoutSequenceStack.push_front(ReadoutSequence);
}

void TKinokoDaqBuilder::CloseConditionalReadoutSequence(void)
{
    _ReadoutSequenceStack.pop_front();
}

void TKinokoDaqBuilder::OpenReadoutUnit(TKinokoUnitedReadoutActions* UnitedReadoutActions, TKinokoNestedDataSection* DataSection) throw(TKinokoException)
{
    AddReadoutAction(UnitedReadoutActions, DataSection);
    
    _ReadoutSequenceStack.push_front(UnitedReadoutActions);
    _NestedDataSectionStack.push_front(DataSection);
}

void TKinokoDaqBuilder::CloseReadoutUnit(void)
{
    _ReadoutSequenceStack.pop_front();
    _NestedDataSectionStack.pop_front();
}

void TKinokoDaqBuilder::AddReadoutAction(TKinokoReadoutAction* ReadoutAction, TKinokoDataSection* DataSection) throw(TKinokoException)
{
    if (_ReadoutSequenceStack.empty()) {
	throw TKinokoException("ReadoutAction outside ReadoutSequence");
    }
    _ReadoutSequenceStack.front()->AddReadoutAction(ReadoutAction);

    if (DataSection != 0) {
	if (! _NestedDataSectionStack.empty()) {
	    _NestedDataSectionStack.front()->AddDataSection(DataSection);
	}
	else {
	    _DataSource->AddDataSection(DataSection);
	}
    }
}

unsigned TKinokoDaqBuilder::AddTriggerSource(const string& TriggerSourceName) throw(TKinokoException)
{
    TRoomModule* TriggerSource = _DevicePool->LookupModule(TriggerSourceName);
    if (TriggerSource == 0) {
	throw TKinokoException("unknown module: " + TriggerSourceName);
    }

    int TriggerSourceId = _DevicePool->DeviceIdOf(TriggerSourceName);
    _Trigger->AddTriggerSource(TriggerSourceId, TriggerSource);

    return TriggerSourceId;
}

void TKinokoDaqBuilder::AddAttribute(const std::string& Name, const std::string& Value) throw(TKinokoException)
{
    _DataSource->AddAttribute(Name, Value);
}

TKinokoEventEmitter* TKinokoDaqBuilder::EventEmitter(void)
{
    return _EventEmitter;
}

TKinokoRegistry* TKinokoDaqBuilder::Registry(void)
{
    return _Registry;
}
