/* KinokoDaqFrontend.cc */
/* Created by Enomoto Sanshiro on 8 July 2001. */
/* Last updated by Enomoto Sanshiro on 29 March 2002. */


#include <iostream>
#include <strstream>
#include <string>
#include "MushFileSystem.hh"
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KinokoScriptedDaqBuilder.hh"
#include "KinokoDevicePool.hh"
#include "KinokoTrigger.hh"
#include "KinokoTransaction.hh"
#include "KinokoStreamSourceComponent.hh"
#include "KinokoDaqFrontend.hh"

using namespace std;


TKinokoDaqFrontend::TKinokoDaqFrontend(const string& Name)
{
    _Name = Name;
    _DaqBuilder = 0;
}

TKinokoDaqFrontend::~TKinokoDaqFrontend()
{
    delete _DaqBuilder;
}

void TKinokoDaqFrontend::Construct(const string& ScriptFileName, const string& DataSourceName, int DataSourceId, TKinokoOutputStream* OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry) throw(TKinokoException)
{
    if (_DaqBuilder != 0) {
	throw TKinokoException(
	    "TKinokoDaqFrontend::Construct()", "already constructed"
	);
    }

    _DaqBuilder = new TKinokoScriptedDaqBuilder(
	ScriptFileName, DataSourceName, DataSourceId, 
	OutputStream, EventEmitter, Registry
    );
    
    _DevicePool = _DaqBuilder->DevicePool();
    _Trigger = _DaqBuilder->Trigger();
    _Transaction = _DaqBuilder->Transaction();
    
    try {
	_DevicePool->ProbeDevices();
    }
    catch (TKinokoException &e) {
	delete _DaqBuilder;
	_DaqBuilder = 0;
	throw TKinokoException(
	    "TKinokoDaqFrontend::Construct()", e.Message()
	);
    }

    char ScriptCheckSum[32];
    ostrstream os(ScriptCheckSum, sizeof(ScriptCheckSum));
    os << hex << TMushFileAttribute(ScriptFileName).CheckSum() << ends;

    string Date = TMushDateTime().AsString("%Y-%m-%d %H:%M:%S %Z");
    
    TKinokoDataSource* DataSource = _DaqBuilder->DataSource();
    DataSource->AddAttribute("creator", _Name);
    DataSource->AddAttribute("creation_date", Date);
    DataSource->AddAttribute("script_file", ScriptFileName);	
    DataSource->AddAttribute("script_file_fingerprint", ScriptCheckSum);

    _Transaction->SendDataDescriptorPacket();
}

void TKinokoDaqFrontend::Destruct(void) throw(TKinokoException)
{
    if (_DaqBuilder != 0) {
	_DaqBuilder->Destruct();
	delete _DaqBuilder;
	_DaqBuilder = 0;
    }
}

void TKinokoDaqFrontend::OnStart(void) throw(TKinokoException)
{
    _DevicePool->Initialize();
    _Trigger->Initialize();
    _Transaction->Initialize();
}

void TKinokoDaqFrontend::OnStop(void) throw(TKinokoException)
{
    _Transaction->Finalize();
    _Trigger->Finalize();
    _DevicePool->Finalize();
}

int TKinokoDaqFrontend::DoTransaction(void) throw(TKinokoException)
{
    int Result = 0;
    int TriggerSourceId;
    int TriggerId;

    if (_Trigger->Wait(TriggerSourceId, TriggerId)) {
        Result = _Transaction->OnTrigger(TriggerSourceId, &TriggerId, 1);
	_Trigger->Clear();
    }
    else {
	Result = 0;
    }

    return Result;
}

int TKinokoDaqFrontend::ExecuteDaqCommand(const string& Message, int* ParameterList, int NumberOfParameters) throw(TKinokoException)
{
    return _Transaction->OnCommand(Message, ParameterList, NumberOfParameters);
}

bool TKinokoDaqFrontend::IsStopRequested(void)
{
    return _Trigger->IsStopRequested();
}


TKinokoNullDaqFrontend::TKinokoNullDaqFrontend(void)
: TKinokoDaqFrontend("NULL")
{
}

TKinokoNullDaqFrontend::~TKinokoNullDaqFrontend()
{
}

void TKinokoNullDaqFrontend::Construct(const string& ScriptFileName, const string& DataSourceName, int DataSourceId, TKinokoOutputStream* OutputStream, TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry) throw(TKinokoException)
{
}

void TKinokoNullDaqFrontend::Destruct(void) throw(TKinokoException)
{
}

void TKinokoNullDaqFrontend::OnStart(void) throw(TKinokoException)
{
}

void TKinokoNullDaqFrontend::OnStop(void) throw(TKinokoException)
{
}

int TKinokoNullDaqFrontend::DoTransaction(void) throw(TKinokoException)
{
    TMushRealTimeTimer(0, 100000).Suspend();
    return 0;
}

int TKinokoNullDaqFrontend::ExecuteDaqCommand(const string& Message, int* ParameterList, int NumberOfParameters) throw(TKinokoException)
{
    return 0;
}
