/* KinokoDaqScriptStatement.cc */
/* Created by Enomoto Sanshiro on 11 October 1999. */
/* Last updated by Enomoto Sanshiro on 6 July 2001. */


#include <string>
#include "ParaStatement.hh"
#include "KinokoTrigger.hh"
#include "KinokoTransaction.hh"
#include "KinokoDaqBuilder.hh"
#include "KinokoReadoutAction.hh"
#include "KinokoMetaReadoutAction.hh"
#include "KinokoModuleMessenger.hh"
#include "KinokoControllerMessenger.hh"
#include "KinokoMetaDeviceMessenger.hh"
#include "KinokoDaqScriptStatement.hh"

using namespace std;


TKinokoDaqScriptOnStatement::TKinokoDaqScriptOnStatement(TKinokoDaqBuilder* DaqBuilder)
{
    _DaqBuilder = DaqBuilder;
    _ConcreteStatement = 0;
}

TKinokoDaqScriptOnStatement::~TKinokoDaqScriptOnStatement()
{
    delete _ConcreteStatement;
}

TParaStatement* TKinokoDaqScriptOnStatement::Clone(void)
{
    return new TKinokoDaqScriptOnStatement(_DaqBuilder);
}

string TKinokoDaqScriptOnStatement::FirstToken(void) const
{
    return string("on");
}

void TKinokoDaqScriptOnStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->LookAhead(1).MustBe(FirstToken());
    if (Tokenizer->LookAhead(2).Is("trigger")) {
	_ConcreteStatement = new TKinokoDaqScriptOnTriggerStatement(_DaqBuilder);
    }
    else if (Tokenizer->LookAhead(2).Is("command")) {
	_ConcreteStatement = new TKinokoDaqScriptOnCommandStatement(_DaqBuilder);
    }
    else {
	_ConcreteStatement = new TKinokoDaqScriptOnTrapStatement(_DaqBuilder);
    }

    _ConcreteStatement->Parse(Tokenizer, StatementParser, SymbolTable);
}

TParaStatement::TExecResult TKinokoDaqScriptOnStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _ConcreteStatement->Execute(SymbolTable);
}



TKinokoDaqScriptOnTriggerStatement::TKinokoDaqScriptOnTriggerStatement(TKinokoDaqBuilder* DaqBuilder)
: TKinokoDaqScriptOnStatement(DaqBuilder)
{
    _ModuleExpression = 0;
    _RegisterExpression = 0;
    _Statement = 0;
}

TKinokoDaqScriptOnTriggerStatement::~TKinokoDaqScriptOnTriggerStatement()
{
    delete _RegisterExpression;
    delete _ModuleExpression;
    delete _Statement;
}

TParaStatement* TKinokoDaqScriptOnTriggerStatement::Clone(void)
{
    return new TKinokoDaqScriptOnTriggerStatement(_DaqBuilder);
}

void TKinokoDaqScriptOnTriggerStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());
    Tokenizer->Next().MustBe("trigger");
    Tokenizer->Next().MustBe("(");

    _ModuleExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);

    if (Tokenizer->LookAhead().Is(",")) {
        Tokenizer->Next();
	_RegisterExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    }

    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoDaqScriptOnTriggerStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue ModuleValue = _ModuleExpression->Evaluate(SymbolTable);
    if (! ModuleValue.IsObject()) {
	throw TScriptException(
	    _ModuleExpression->Position() + 
	    "bad trigger module: " + ModuleValue.AsString()
	);
    }
    TParaObjectPrototype* ModuleMessenger = ModuleValue.AsObject();

    string ModuleName;
    if (
	//... temporary 
	// We should implement some mechanizm to handle class hierarchy.
	(ModuleMessenger->InternalClassName() == "VmeModule") ||
	(ModuleMessenger->InternalClassName() == "CamacModule") ||
	(ModuleMessenger->InternalClassName() == "SoftwareModule")
    ){
	ModuleName = ((TKinokoModuleMessenger*) ModuleMessenger)->ModuleName();
    }
    else {
	throw TScriptException(
	    _ModuleExpression->Position() + 
	    "bad trigger module: " + ModuleValue.AsString()
	);
    }

    int* TriggerIdRegister = 0;
    if (_RegisterExpression) {
	try {
	    TriggerIdRegister = TKinokoRegisterMessenger::GetRegisterOf(
		_RegisterExpression->Evaluate(SymbolTable), _DaqBuilder
	    );
	}
	catch (TScriptException &e) {
	    throw TScriptException(
		_RegisterExpression->Position() + e.Message()
	    );
	}
    }

    int TriggerSourceId = _DaqBuilder->AddTriggerSource(ModuleName);
    _DaqBuilder->OpenReadoutSequence(TriggerSourceId, TriggerIdRegister);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _DaqBuilder->CloseReadoutSequence();

    return Result;
}



TKinokoDaqScriptOnCommandStatement::TKinokoDaqScriptOnCommandStatement(TKinokoDaqBuilder* DaqBuilder)
: TKinokoDaqScriptOnStatement(DaqBuilder)
{
    _Statement = 0;
}

TKinokoDaqScriptOnCommandStatement::~TKinokoDaqScriptOnCommandStatement()
{
    delete _Statement;

    for (unsigned i = 0; i < _RegisterExpressionList.size(); i++) {
	delete _RegisterExpressionList[i];
    }
}

TParaStatement* TKinokoDaqScriptOnCommandStatement::Clone(void)
{
    return new TKinokoDaqScriptOnCommandStatement(_DaqBuilder);
}

void TKinokoDaqScriptOnCommandStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());
    Tokenizer->Next().MustBe("command");

    Tokenizer->Next().MustBe("(");
    _CommandName = Tokenizer->Next().RemoveQuotation('"').AsString();

    while (Tokenizer->LookAhead().Is(",")) {
	Tokenizer->Next().MustBe(",");

	_RegisterExpressionList.push_back(
	    ExpressionParser->Parse(Tokenizer, SymbolTable)
	);
    }

    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoDaqScriptOnCommandStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    vector<int*> ArgumentRegisterList;

    for (unsigned i = 0; i < _RegisterExpressionList.size(); i++) {
	int* Register;
	try {
	    Register = TKinokoRegisterMessenger::GetRegisterOf(
		_RegisterExpressionList[i]->Evaluate(SymbolTable), _DaqBuilder
	    );
	}
	catch (TScriptException &e) {
	    throw TScriptException(
		_RegisterExpressionList[i]->Position() + e.Message()
	    );
	}

	ArgumentRegisterList.push_back(Register);
    }

    _DaqBuilder->OpenReadoutSequence(_CommandName, ArgumentRegisterList);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _DaqBuilder->CloseReadoutSequence();

    return Result;
}



TKinokoDaqScriptOnTrapStatement::TKinokoDaqScriptOnTrapStatement(TKinokoDaqBuilder* DaqBuilder)
: TKinokoDaqScriptOnStatement(DaqBuilder)
{
    _Statement = 0;
}

TKinokoDaqScriptOnTrapStatement::~TKinokoDaqScriptOnTrapStatement()
{
    delete _Statement;
}

TParaStatement* TKinokoDaqScriptOnTrapStatement::Clone(void)
{
    return new TKinokoDaqScriptOnTrapStatement(_DaqBuilder);
}

void TKinokoDaqScriptOnTrapStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    string TrapName = Tokenizer->Next().RemoveQuotation('"').AsString();

    if (TrapName == "run_begin") {
	_TrapId = TKinokoTransaction::TrapId_RunBegin;
    }
    else if (TrapName == "run_end") {
	_TrapId = TKinokoTransaction::TrapId_RunEnd;
    }
    else if (TrapName == "run_suspend") {
	_TrapId = TKinokoTransaction::TrapId_RunSuspend;
    }
    else if (TrapName == "run_resume") {
	_TrapId = TKinokoTransaction::TrapId_RunResume;
    }
    else {
	throw TScriptException("unknown on-statement tag: " + TrapName);
    }

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoDaqScriptOnTrapStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _DaqBuilder->OpenReadoutSequence(_TrapId);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _DaqBuilder->CloseReadoutSequence();

    return Result;
}



TKinokoDaqScriptInvokeStatement::TKinokoDaqScriptInvokeStatement(TKinokoDaqBuilder* DaqBuilder)
{
    _DaqBuilder = DaqBuilder;
}

TKinokoDaqScriptInvokeStatement::~TKinokoDaqScriptInvokeStatement()
{
    for (unsigned i = 0; i < _EventArgumentExpressionList.size(); i++) {
	delete _EventArgumentExpressionList[i];
    }
}

TParaStatement* TKinokoDaqScriptInvokeStatement::Clone(void)
{
    return new TKinokoDaqScriptInvokeStatement(_DaqBuilder);
}

string TKinokoDaqScriptInvokeStatement::FirstToken(void) const
{
    return string("invoke");
}

void TKinokoDaqScriptInvokeStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());

    _EventName = Tokenizer->Next().AsString();
    if (Tokenizer->LookAhead().Is("(")) {
	_EventArgumentExpressionList = ExpressionParser->ParseExpressionList(
	    Tokenizer, SymbolTable, "(", ")", ","
	);
    }

    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKinokoDaqScriptInvokeStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    string EventName = _EventName;

    vector<string> EventArgumentList;
    for (unsigned i = 0; i < _EventArgumentExpressionList.size(); i++) {
	EventArgumentList.push_back(
	    _EventArgumentExpressionList[i]->Evaluate(SymbolTable).AsString()
	);
    }

    TKinokoDaqInvokeAction* InvokeAction =  new TKinokoDaqInvokeAction(
	_DaqBuilder->EventEmitter(), EventName, EventArgumentList
    );
    try {
	_DaqBuilder->AddReadoutAction(InvokeAction);
    }
    catch (TKinokoException &e) {
	delete InvokeAction;
	throw TScriptException("invoke: " + e.Message());
    }

    return TParaStatement::TExecResult();
}



TKinokoDaqScriptWhenStatement::TKinokoDaqScriptWhenStatement(TKinokoDaqBuilder* DaqBuilder)
{
    _DaqBuilder = DaqBuilder;
    _ConditionExpression = 0;
    _Statement = 0;
}

TKinokoDaqScriptWhenStatement::~TKinokoDaqScriptWhenStatement()
{
    delete _Statement;
    delete _ConditionExpression;
}

TParaStatement* TKinokoDaqScriptWhenStatement::Clone(void)
{
    return new TKinokoDaqScriptWhenStatement(_DaqBuilder);
}

string TKinokoDaqScriptWhenStatement::FirstToken(void) const
{
    return string("when");
}

void TKinokoDaqScriptWhenStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());

    Tokenizer->Next().MustBe("(");
    _ConditionExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoDaqScriptWhenStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaValue& ConditionValue = _ConditionExpression->Evaluate(SymbolTable);
    if (! ConditionValue.IsObject("ReadoutCondition")) {
	throw TScriptException(
	    _ConditionExpression->Position() +
	    "when: invalid sequence-condition"
	);
    }

    TKinokoReadoutCondition* Condition;
    try {
	Condition = ((TKinokoReadoutConditionMessenger*) ConditionValue.AsObject())->GetReadoutCondition();
    }
    catch (TScriptException &e) {
	throw TScriptException(_ConditionExpression->Position() + e.Message());
    }
	
    _DaqBuilder->OpenConditionalReadoutSequence(Condition);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _DaqBuilder->CloseConditionalReadoutSequence();
    
    return TExecResult();
}



TKinokoDaqScriptUnitStatement::TKinokoDaqScriptUnitStatement(TKinokoDaqBuilder* DaqBuilder)
{
    _DaqBuilder = DaqBuilder;
    _Statement = 0;
}

TKinokoDaqScriptUnitStatement::~TKinokoDaqScriptUnitStatement()
{
    delete _Statement;
}

TParaStatement* TKinokoDaqScriptUnitStatement::Clone(void)
{
    return new TKinokoDaqScriptUnitStatement(_DaqBuilder);
}

string TKinokoDaqScriptUnitStatement::FirstToken(void) const
{
    return string("unit");
}

void TKinokoDaqScriptUnitStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    _UnitName = Tokenizer->Next().RemoveQuotation('"').AsString();
    
    _SectionId = TKinokoDataSection::SectionId_Unknown;
    if (Tokenizer->LookAhead().Is("<")) {
	Tokenizer->Next().MustBe("<");
	_SectionId = Tokenizer->Next().AsLong();
	Tokenizer->Next().MustBe(">");
    }

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoDaqScriptUnitStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TKinokoDataSource* DataSource = _DaqBuilder->DataSource();

    TKinokoNestedDataSection* DataSection = new TKinokoNestedDataSection(
	DataSource, _UnitName, _SectionId
    );
    TKinokoUnitedReadoutActions* UnitedReadoutActions = new TKinokoUnitedReadoutActions(
	DataSection
    );

    _DaqBuilder->OpenReadoutUnit(UnitedReadoutActions, DataSection);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _DaqBuilder->CloseReadoutUnit();

    return Result;
}



TKinokoDaqScriptAttributeStatement::TKinokoDaqScriptAttributeStatement(TKinokoDaqBuilder* DaqBuilder)
{
    _DaqBuilder = DaqBuilder;
    _ValueExpression = 0;
}

TKinokoDaqScriptAttributeStatement::~TKinokoDaqScriptAttributeStatement()
{
    delete _ValueExpression;
}

TParaStatement* TKinokoDaqScriptAttributeStatement::Clone(void)
{
    return new TKinokoDaqScriptAttributeStatement(_DaqBuilder);
}

string TKinokoDaqScriptAttributeStatement::FirstToken(void) const
{
    return string("attribute");
}

void TKinokoDaqScriptAttributeStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());

    _Name = Tokenizer->Next().AsString();
    Tokenizer->Next().MustBe("=");

    _ValueExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);
    Tokenizer->Next().MustBe(";");
}

TParaStatement::TExecResult TKinokoDaqScriptAttributeStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    string Value = _ValueExpression->Evaluate(SymbolTable).AsString();
    _DaqBuilder->AddAttribute(_Name, Value);

    return TParaStatement::TExecResult();
}



TKinokoDaqScriptDataSourceEntry::TKinokoDaqScriptDataSourceEntry(TKinokoDaqBuilder* DaqBuilder)
: TParaPackageEntry("datasource")
{
    _DaqBuilder = DaqBuilder;
    _Statement = 0;
}

TKinokoDaqScriptDataSourceEntry::~TKinokoDaqScriptDataSourceEntry()
{
    delete _Statement;
}

TParaPackageEntry* TKinokoDaqScriptDataSourceEntry::Clone(void)
{
    return new TKinokoDaqScriptDataSourceEntry(_DaqBuilder);
}

bool TKinokoDaqScriptDataSourceEntry::HasEntryWordsOf(TParaTokenizer* Tokenizer)
{
    return Tokenizer->LookAhead().Is("datasource");
}

void TKinokoDaqScriptDataSourceEntry::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe("datasource");
    string DataSourceName = Tokenizer->Next().RemoveQuotation('"').AsString();

    int DataSourceId = -1;
    if (Tokenizer->LookAhead().Is("<")) {
	Tokenizer->Next().MustBe("<");
	DataSourceId = Tokenizer->Next().AsLong();
	Tokenizer->Next().MustBe(">");
    }

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);

    SetEntryName(DataSourceName);
    _DaqBuilder->RegisterDataSourceName(DataSourceName, DataSourceId);
}

TParaValue TKinokoDaqScriptDataSourceEntry::Execute(const vector<TParaValue*>& ArgumentList, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _Statement->Execute(SymbolTable).ReturnValue;
}
