/* KinokoMetaDeviceMessenger.cc */
/* Created by Enomoto Sanshiro on 4 July 2001. */
/* Last updated by Enomoto Sanshiro on 9 March 2003. */


#include <strstream>
#include "ParaOperator.hh"
#include "KinokoDaqBuilder.hh"
#include "KinokoReadoutAction.hh"
#include "KinokoMetaReadoutAction.hh"
#include "KinokoReadoutCondition.hh"
#include "KinokoMetaDeviceMessenger.hh"

using namespace std;


int TKinokoRegisterMessenger::_RegisterCount = 0;

TKinokoRegisterMessenger::TKinokoRegisterMessenger(TKinokoDaqBuilder* DaqBuilder)
: TParaObjectPrototype("Register")
{
    _DaqBuilder = DaqBuilder;

    _Register = 0;
}

TKinokoRegisterMessenger::~TKinokoRegisterMessenger()
{
}

TParaObjectPrototype* TKinokoRegisterMessenger::Clone(void)
{
    return new TKinokoRegisterMessenger(_DaqBuilder);
}

void TKinokoRegisterMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TParaObjectPrototype::Construct(ClassName, ArgumentList);

    _Register = new int(0);    
    _RegisterName = CreateRegisterName();

    _DaqBuilder->RegisterRegister(_RegisterName, _Register);
}

string TKinokoRegisterMessenger::RegisterName(void)
{
    return _RegisterName;
}

int TKinokoRegisterMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "load") || (Message == "Load")) {
        ReturnValue = AddRegisterAction(
            Message, ArgumentList, TKinokoRegisterAction::OperationId_Load
	);
    }
    else if ((Message == "add") || (Message == "Add")) {
        ReturnValue = AddRegisterAction(
            Message, ArgumentList, TKinokoRegisterAction::OperationId_Add
	);
    }
    else if ((Message == "subtract") || (Message == "Subtract")) {
        ReturnValue = AddRegisterAction(
            Message, ArgumentList, TKinokoRegisterAction::OperationId_Subtract
	);
    }
    else if ((Message == "multiply") || (Message == "Multiply")) {
        ReturnValue = AddRegisterAction(
            Message, ArgumentList, TKinokoRegisterAction::OperationId_Multiply
	);
    }
    else if ((Message == "divide") || (Message == "Divide")) {
        ReturnValue = AddRegisterAction(
            Message, ArgumentList, TKinokoRegisterAction::OperationId_Divide
	);
    }
    else if ((Message == "modulo") || (Message == "Modulo")) {
        ReturnValue = AddRegisterAction(
            Message, ArgumentList, TKinokoRegisterAction::OperationId_Modulo
	);
    }
    else if ((Message == "dump") || (Message == "Dump")) {
        ReturnValue = Dump(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue& TKinokoRegisterMessenger::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    int Parameter = RightValue.AsLong();

    //... BUG: This ConditionMessenger will not be deleted
    TKinokoRegisterConditionMessenger* ConditionMessenger;

    ConditionMessenger = new TKinokoRegisterConditionMessenger(
	Operator->Symbol(), _Register, Parameter
    );

    return Result = TParaValue(ConditionMessenger);
}

TParaValue TKinokoRegisterMessenger::AddRegisterAction(const string& Message, vector<TParaValue*>& ArgumentList, int RegisterOperationId) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    _RegisterName + "." + Message + "(...)", "too few argument[s]"
	);
    }	    

    int* ParameterRegister;
    try {
	ParameterRegister = GetConstRegisterOf(*ArgumentList[0], _DaqBuilder);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    _RegisterName + "." + Message + "(...): ", e.Message()
	);
    }
    
    TKinokoRegisterAction* RegisterAction = new TKinokoRegisterAction(
	_Register, RegisterOperationId, ParameterRegister
    );

    try {
	_DaqBuilder->AddReadoutAction(RegisterAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
            _RegisterName + "." + Message + "()", e.Message()
	);
    }

    return TParaValue((TParaObjectPrototype*) this);
}

TParaValue TKinokoRegisterMessenger::Dump(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoReadoutAction* Action = new TKinokoRegisterDumpAction(
	_Register, ObjectName()
    );

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_RegisterName + ".dump(): " + e.Message());
    }

    return TParaValue((long) 0);
}

string TKinokoRegisterMessenger::CreateRegisterName(void)
{
    char RegisterName[64];

    ostrstream RegisterNameStream(RegisterName, sizeof(RegisterName));
    RegisterNameStream << "__register_" << _RegisterCount++ << ends;

    return string(RegisterName);
}

int* TKinokoRegisterMessenger::GetRegisterOf(TParaValue& Value, TKinokoDaqBuilder* DaqBuilder) throw(TScriptException)
{
    if (! Value.IsObject("Register")) {
	throw TScriptException(
	    "register value is expexted: " + Value.AsString()
	);
    }

    string RegisterName = (
	((TKinokoRegisterMessenger*) Value.AsObject())->RegisterName()
    );

    int* Register = DaqBuilder->DevicePool()->LookupRegister(RegisterName);
    if (Register == 0) {
	throw TScriptException("invalid register name: " + RegisterName);
    }
    
    return Register;
}

int* TKinokoRegisterMessenger::GetConstRegisterOf(TParaValue& Value, TKinokoDaqBuilder* DaqBuilder) throw(TScriptException)
{
    int* Register;

    if (Value.IsLong()) {
	Register = new int(Value.AsLong());
	DaqBuilder->RegisterRegister(CreateRegisterName(), Register);
    }
    else if (Value.IsObject("Register")) {
	Register = GetRegisterOf(Value, DaqBuilder);
    }
    else {
	throw TScriptException(
	    "integer or register value is expexted: " + Value.AsString()
	);
    }

    return Register;
}



TKinokoReadoutConditionMessenger::TKinokoReadoutConditionMessenger(void)
: TParaObjectPrototype("ReadoutCondition")
{
}

TKinokoReadoutConditionMessenger::~TKinokoReadoutConditionMessenger()
{
}

TParaObjectPrototype* TKinokoReadoutConditionMessenger::Clone(void)
{
    return 0;
}

TParaValue& TKinokoReadoutConditionMessenger::EvaluateOperator(TParaOperator* Operator, TParaValue& LeftValue, TParaValue& RightValue, TParaSymbolTable* SymbolTable, TParaValue& Result) throw(TScriptException)
{
    //... BUG: This ConditionMessenger will not be deleted
    TKinokoReadoutConditionMessenger* ConditionMessenger;

    if (Operator->Name() == "Not") {
	ConditionMessenger = new TKinokoComplexReadoutConditionMessenger(
	    Operator->Symbol(), 0, this
	);
    }
    else if (RightValue.IsObject("ReadoutCondition")) {
	ConditionMessenger = new TKinokoComplexReadoutConditionMessenger(
	    Operator->Symbol(), 
	    this, 
	    (TKinokoReadoutConditionMessenger*) RightValue.AsObject()
	);
    }
    else {
	throw TScriptException("invalid operation: " + Operator->Symbol());
    }

    return Result = TParaValue(ConditionMessenger);
}



TKinokoRegisterConditionMessenger::TKinokoRegisterConditionMessenger(const string& OperatorSymbol, int* Register, int Parameter)
{
    _OperatorSymbol = OperatorSymbol;
    _Register = Register;
    _Parameter = Parameter;
}

TKinokoRegisterConditionMessenger::~TKinokoRegisterConditionMessenger()
{
}

TKinokoReadoutCondition* TKinokoRegisterConditionMessenger::GetReadoutCondition(void) throw(TScriptException)
{
    int OperatorId;
    if (_OperatorSymbol == "==") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_Equal;
    }
    else if (_OperatorSymbol == "!=") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_NotEqual;
    }
    else if (_OperatorSymbol == ">") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_GreaterThan;
    }
    else if (_OperatorSymbol == ">=") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_GreaterEqual;
    }
    else if (_OperatorSymbol == "<") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_LessThan;
    }
    else if (_OperatorSymbol == "<=") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_LessEqual;
    }
    else if (_OperatorSymbol == "&") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_BitAnd;
    }
    else if (_OperatorSymbol == "^") { 
	OperatorId = TKinokoRegisterReadoutCondition::Operator_BitXor;
    }
    else {
	throw TScriptException("invalid register operator: " + _OperatorSymbol);
    }

    TKinokoReadoutCondition* Condition = new TKinokoRegisterReadoutCondition(
	_Register, _Parameter, OperatorId
    );

    return Condition;
}



TKinokoComplexReadoutConditionMessenger::TKinokoComplexReadoutConditionMessenger(const string& OperatorSymbol, TKinokoReadoutConditionMessenger* LeftConditionMessenger, TKinokoReadoutConditionMessenger* RightConditionMessenger)
{
    _OperatorSymbol = OperatorSymbol;
    _LeftConditionMessenger = LeftConditionMessenger;
    _RightConditionMessenger = RightConditionMessenger;
}

TKinokoComplexReadoutConditionMessenger::~TKinokoComplexReadoutConditionMessenger()
{
    delete _LeftConditionMessenger;
    delete _RightConditionMessenger;
}

TKinokoReadoutCondition* TKinokoComplexReadoutConditionMessenger::GetReadoutCondition(void) throw(TScriptException)
{
    int OperatorId;
    if (_OperatorSymbol == "!") { 
	OperatorId = TKinokoComplexReadoutCondition::Operator_Not;
    }
    else if (_OperatorSymbol == "&&") { 
	OperatorId = TKinokoComplexReadoutCondition::Operator_And;
    }
    else if (_OperatorSymbol == "||") { 
	OperatorId = TKinokoComplexReadoutCondition::Operator_Or;
    }
    else {
	throw TScriptException("invalid register operator: " + _OperatorSymbol);
    }

    TKinokoReadoutCondition* LeftCondition = 0;
    TKinokoReadoutCondition* RightCondition = 0;

    if (_LeftConditionMessenger) {
	LeftCondition = _LeftConditionMessenger->GetReadoutCondition();
    }
    if (_RightConditionMessenger) {
	RightCondition = _RightConditionMessenger->GetReadoutCondition();
    }

    TKinokoReadoutCondition* Condition = new TKinokoComplexReadoutCondition(
	LeftCondition, RightCondition, OperatorId
    );

    return Condition;
}



TKinokoDataRecordMessenger::TKinokoDataRecordMessenger(TKinokoDaqBuilder* DaqBuilder)
: TParaObjectPrototype("DataRecord")
{
    _DaqBuilder = DaqBuilder;
    _DataRecord = 0;
}

TKinokoDataRecordMessenger::~TKinokoDataRecordMessenger()
{
}

TParaObjectPrototype* TKinokoDataRecordMessenger::Clone(void)
{
    return new TKinokoDataRecordMessenger(_DaqBuilder);
}

void TKinokoDataRecordMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() > 0) {
	_DataRecordName = ArgumentList[0]->AsString();
    }
    else {
	_DataRecordName = this->ObjectName();
    }

    int SectionId = TKinokoDataSection::SectionId_Unknown;
    if (ArgumentList.size() > 1) {
	SectionId = ArgumentList[1]->AsLong();
    }

    TKinokoDataSource* DataSource = _DaqBuilder->DataSource();
    _DataSection = new TKinokoTaggedDataSection(DataSource, _DataRecordName, SectionId);
    _DataRecord = new TKinokoDataRecord(_DataSection);

    _DaqBuilder->RegisterDataRecord(_DataRecordName, _DataRecord);
}

int TKinokoDataRecordMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "fill") || (Message == "Fill")) {
        ReturnValue = Fill(ArgumentList);
    }
    else if ((Message == "send") || (Message == "Send")) {
        ReturnValue = Send(ArgumentList);
    }
    else if ((Message == "dump") || (Message == "Dump")) {
        ReturnValue = Dump(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKinokoDataRecordMessenger::Fill(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: " + _DataRecordName + 
	    ".fill(string field_name, ...)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "bad argument: " + _DataRecordName + 
	    ".fill(string field_name, ...)"
	);
    }

    string FieldName = ArgumentList[0]->AsString();
    int FieldIndex = _DataRecord->AddField(FieldName);
    
    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(_DataRecordName + ".fill(): " + e.Message());
    }
    
    TKinokoDataRecordFillAction* DataRecordFillAction = (
	new TKinokoDataRecordFillAction(_DataRecord, FieldIndex, Register)
    );

    try {
	_DaqBuilder->AddReadoutAction(DataRecordFillAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_DataRecordName + ".fill(): " + e.Message());
    }

    return TParaValue((long) 0);
}

TParaValue TKinokoDataRecordMessenger::Send(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoDataRecordSendAction* Action = new TKinokoDataRecordSendAction(
	_DataRecord
    );

    try {
	_DaqBuilder->AddReadoutAction(Action, _DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_DataRecordName + ".send(): " + e.Message());
    }

    return TParaValue((long) 0);
}

TParaValue TKinokoDataRecordMessenger::Dump(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoDataRecordDumpAction* Action = new TKinokoDataRecordDumpAction(
	_DataRecord
    );

    try {
	_DaqBuilder->AddReadoutAction(Action, _DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_DataRecordName + ".dump(): " + e.Message());
    }

    return TParaValue((long) 0);
}



TKinokoDaqSystemMessenger::TKinokoDaqSystemMessenger(TKinokoDaqBuilder* DaqBuilder, const string& ScriptFileName)
: TParaObjectPrototype("DaqSystem")
{
    _DaqBuilder = DaqBuilder;
    _ScriptFileName = ScriptFileName;
}

TKinokoDaqSystemMessenger::~TKinokoDaqSystemMessenger()
{
}

TParaObjectPrototype* TKinokoDaqSystemMessenger::Clone(void)
{
    return new TKinokoDaqSystemMessenger(_DaqBuilder, _ScriptFileName);
}

void TKinokoDaqSystemMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
}

int TKinokoDaqSystemMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "getRegistry") || (Message == "GetRegistry")) {
        // normal registry access
        ReturnValue = GetRegistry(ArgumentList);
    }
    else if ((Message == "setRegistry") || (Message == "SetRegistry")) {
        // normal registry access
        ReturnValue = SetRegistry(ArgumentList);
    }
    else if ((Message == "readRegistry") || (Message == "ReadRegistry")) {
        // registry access action
        ReturnValue = ReadRegistry(ArgumentList);
    }
    else if ((Message == "writeRegistry") || (Message == "WriteRegistry")) {
        // registry access action
        ReturnValue = WriteRegistry(ArgumentList);
    }
    else if ((Message == "suspend") || (Message == "Suspend")) {
        ReturnValue = Suspend(ArgumentList);
    }
    else if ((Message == "terminate") || (Message == "Terminate")) {
        ReturnValue = Terminate(ArgumentList);
    }
    else if ((Message == "echo") || (Message == "Echo")) {
        ReturnValue = Echo(ArgumentList);
    }
    else if ((Message == "readTime") || (Message == "ReadTime")) {
        ReturnValue = ReadTime(ArgumentList);
    }
    else if ((Message == "scriptFileName") || (Message == "ScriptFileName")) {
        ReturnValue = ScriptFileName(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

TParaValue TKinokoDaqSystemMessenger::GetRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few arguments: getRegistry(string registry_path)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument:  getRegistry(string registry_path)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    TKinokoRegistry* Registry = _DaqBuilder->Registry();

    string Value = Registry->GetValue(RegistryPath);

    return TParaValue(Value);
}

TParaValue TKinokoDaqSystemMessenger::SetRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    TKinokoRegistry* Registry = _DaqBuilder->Registry();
    Registry->SetValue(RegistryPath, Value);

    return TParaValue(Value);
}

TParaValue TKinokoDaqSystemMessenger::ReadRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: "
	    "readRegistry(string registry_path, Register register)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument: "
	    "readRegistry(string registry_path, Register register)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid argument: "
	    "readRegistry(string registry_path, Register register): " +
	    e.Message()
	);
    }

    TKinokoRegistry* Registry = _DaqBuilder->Registry();

    TKinokoReadoutAction* Action = new TKinokoReadRegistryAction(
	Registry, RegistryPath, Register
    );
    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("readRegistry(): " + e.Message());
    }


    return TParaValue((long) 0);
}

TParaValue TKinokoDaqSystemMessenger::WriteRegistry(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: "
	    "writeRegistry(string registry_path, Registry register)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument: "
	    "writeRegistry(string registry_path, Register register)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid argument: "
	    "writeRegistry(string registry_path, Register register): " +
	    e.Message()
	);
    }

    TKinokoRegistry* Registry = _DaqBuilder->Registry();

    TKinokoReadoutAction* Action = new TKinokoWriteRegistryAction(
	Registry, RegistryPath, Register
    );
    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("writeRegistry(): " + e.Message());
    }


    return TParaValue((long) 0);
}

TParaValue TKinokoDaqSystemMessenger::Suspend(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException("too few arguments: suspend(int sec, int usec)");
    }
    if ((! ArgumentList[0]->IsLong()) || (! ArgumentList[1]->IsLong())) {
	throw TScriptException("bad argument: suspend(int sec, int usec)");
    }

    int Time_sec = ArgumentList[0]->AsLong();
    int Time_usec = ArgumentList[1]->AsLong();
    
    TKinokoReadoutAction* Action = new TKinokoSuspendAction(Time_sec, Time_usec);
    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("suspend(): " + e.Message());
    }

    return TParaValue((long) 0);
}

TParaValue TKinokoDaqSystemMessenger::Terminate(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoTerminateAction* TerminateAction =  new TKinokoTerminateAction(
	_DaqBuilder->Trigger()
    );

    //... BUG: "terminate" should abort the current sequence
    try {
	_DaqBuilder->AddReadoutAction(TerminateAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException("terminate() " + e.Message());
    }

    return TParaValue((long) 0);
}

TParaValue TKinokoDaqSystemMessenger::ReadTime(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few arguments: ", "readTime(Register register)"
	);
    }

    int* Register;
    try {
	Register = TKinokoRegisterMessenger::GetRegisterOf(
	    *ArgumentList[0], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "invalid argument: readTime(Register register): " + e.Message()
	);
    }

    TKinokoReadoutAction* Action = new TKinokoReadTimeAction(Register);

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("readTime(): " + e.Message());
    }

    return TParaValue((long) 0);
}

TParaValue TKinokoDaqSystemMessenger::Echo(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    string Message;
    if (ArgumentList.size() > 0) {
	Message = ArgumentList[0]->AsString();
    }

    int* Register = 0;
    if (ArgumentList.size() > 1) {
	try {
	    Register = TKinokoRegisterMessenger::GetRegisterOf(
		*ArgumentList[1], _DaqBuilder
	    );
	}
	catch (TScriptException &e) {
	    throw TScriptException(
		"invalid argument: "
		"echo(string message, Register register): " +
		e.Message()
	    );
	}
    }

    TKinokoReadoutAction* Action = new TKinokoEchoAction(Message, Register);

    try {
	_DaqBuilder->AddReadoutAction(Action);
    }
    catch (TKinokoException &e) {
	throw TScriptException("echo(): " + e.Message());
    }

    return TParaValue((long) 0);
}

TParaValue TKinokoDaqSystemMessenger::ScriptFileName(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    return TParaValue(_ScriptFileName);
}
