/* KinokoMetaReadoutAction.cc */
/* Created by Enomoto Sanshiro on 4 July 2001. */
/* Last updated by Enomoto Sanshiro on 9 March 2003. */


#include <string>
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KcomRegistry.hh"
#include "KinokoReadoutAction.hh"
#include "KinokoDataRecord.hh"
#include "KinokoPlatform.hh"
#include "KinokoTrigger.hh"
#include "KinokoMetaReadoutAction.hh"

using namespace std;


TKinokoRegisterAction::TKinokoRegisterAction(int* Register, int OperationId, int* ParameterRegister)
{
    _Register = Register;
    _OperationId = OperationId;
    _ParameterRegister = ParameterRegister;
}

TKinokoRegisterAction::~TKinokoRegisterAction()
{
}

void TKinokoRegisterAction::Initialize(void)
{
}

int TKinokoRegisterAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoRegisterAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    switch (_OperationId) {
      case OperationId_Load:
	*_Register = *_ParameterRegister;
	break;
      case OperationId_Add:
	*_Register += *_ParameterRegister;
	break;
      case OperationId_Subtract:
	*_Register -= *_ParameterRegister;
	break;
      case OperationId_Multiply:
	*_Register *= *_ParameterRegister;
	break;
      case OperationId_Divide:
	*_Register /= *_ParameterRegister;
	break;
      case OperationId_Modulo:
	*_Register %= *_ParameterRegister;
	break;
      default:
	;
    }

    return 0;
}

void TKinokoRegisterAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "OperateRegister" << hex << "\t";
    os << "[" << _Register << "]" << ", ";
    os << "[" << _ParameterRegister << "](" << *_ParameterRegister << ")";

    os << ", ";
    switch (_OperationId) {
      case OperationId_Load:
        os << "LOAD";
	break;
      case OperationId_Add:
        os << "ADD";
	break;
      case OperationId_Subtract:
        os << "SUBTRACT";
	break;
      case OperationId_Multiply:
        os << "MULTIPLY";
	break;
      case OperationId_Divide:
        os << "DIVIDE";
	break;
      case OperationId_Modulo:
        os << "MODULO";
	break;
      default:
        os << "UNKNOWN";
	;
    }
    os << dec << endl;
}



TKinokoRegisterDumpAction::TKinokoRegisterDumpAction(int* Register, const string& Name)
{
    _Register = Register;
    _Name = Name;
}

TKinokoRegisterDumpAction::~TKinokoRegisterDumpAction()
{
}

void TKinokoRegisterDumpAction::Initialize(void)
{
}

int TKinokoRegisterDumpAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoRegisterDumpAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    cout << _Name << ": " << *_Register << endl;

    return 0;
}

void TKinokoRegisterDumpAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "DumpRegister" << "\t";
    os << _Name << "[" << hex << _Register << dec << "]";
    os << endl;
}



TKinokoDataRecordFillAction::TKinokoDataRecordFillAction(TKinokoDataRecord* DataRecord, int Index, int* Register)
{
    _DataRecord = DataRecord;
    _Index = Index;
    _Register = Register;
}

TKinokoDataRecordFillAction::~TKinokoDataRecordFillAction()
{
}

void TKinokoDataRecordFillAction::Initialize(void)
{
    _DataRecord->Initialize();
}

int TKinokoDataRecordFillAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoDataRecordFillAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    _DataRecord->FillField(_Index, *_Register);

    return 0;
}

void TKinokoDataRecordFillAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "FillDataRecord" << hex << "\t";
    os << _DataRecord->Name() << ", " << _Index << ", ";
    os << "[" << _Register << "](" << *_Register << ")";
    os << dec << endl;
}



TKinokoDataRecordSendAction::TKinokoDataRecordSendAction(TKinokoDataRecord* DataRecord)
{
    _DataRecord = DataRecord;
}

TKinokoDataRecordSendAction::~TKinokoDataRecordSendAction()
{
}

void TKinokoDataRecordSendAction::Initialize(void)
{
    _DataRecord->Initialize();
    _DataRecord->Clear();
}

int TKinokoDataRecordSendAction::NextPacketSize(void)
{
    return _DataRecord->PacketSize();
}

int TKinokoDataRecordSendAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    int WrittenSize = _DataRecord->WriteTo(DataBuffer);
    _DataRecord->Clear();

    return WrittenSize;
}

void TKinokoDataRecordSendAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "SendDataRecord" << "\t";
    os << _DataRecord->Name();
    os << endl;
}



TKinokoDataRecordDumpAction::TKinokoDataRecordDumpAction(TKinokoDataRecord* DataRecord)
{
    _DataRecord = DataRecord;
}

TKinokoDataRecordDumpAction::~TKinokoDataRecordDumpAction()
{
}

void TKinokoDataRecordDumpAction::Initialize(void)
{
    _DataRecord->Initialize();
}

int TKinokoDataRecordDumpAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoDataRecordDumpAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    _DataRecord->Dump(cout);
    return 0;
}

void TKinokoDataRecordDumpAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "DumpDataRecord" << "\t";
    os << _DataRecord->Name();
    os << endl;
}



TKinokoDaqInvokeAction::TKinokoDaqInvokeAction(TKinokoEventEmitter* EventEmitter, const string& EventName, const vector<string>& EventArgumentList)
{
    _EventEmitter = EventEmitter;
    _EventName = EventName;
    _EventArgumentList = EventArgumentList;
}

TKinokoDaqInvokeAction::~TKinokoDaqInvokeAction()
{
}

void TKinokoDaqInvokeAction::Initialize(void)
{
}

int TKinokoDaqInvokeAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoDaqInvokeAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    _EventEmitter->EmitEvent(_EventName, _EventArgumentList);

    return 0;
}

void TKinokoDaqInvokeAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "Invoke  " << "\t" << _EventName;
    for (unsigned int i = 0; i < _EventArgumentList.size(); i++) {
	os << ", \"" << _EventArgumentList[i] << "\"";
    }
    os << dec << endl;
}



TKinokoReadRegistryAction::TKinokoReadRegistryAction(TKinokoRegistry* Registry, const string& RegistryPath, int* Register)
{
    _Registry = Registry;
    _RegistryPath = RegistryPath;
    _Register = Register;
}

TKinokoReadRegistryAction::~TKinokoReadRegistryAction()
{
}

void TKinokoReadRegistryAction::Initialize(void)
{
}

int TKinokoReadRegistryAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoReadRegistryAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    *_Register = _Registry->GetLongValue(_RegistryPath);

    return 0;
}

void TKinokoReadRegistryAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "ReadRegistry" << hex << "\t";
    os << _RegistryPath << ", " << "[" << _Register << "]";
    os << dec << endl;
}



TKinokoWriteRegistryAction::TKinokoWriteRegistryAction(TKinokoRegistry* Registry, const string& RegistryPath, int* Register)
{
    _Registry = Registry;
    _RegistryPath = RegistryPath;
    _Register = Register;
}

TKinokoWriteRegistryAction::~TKinokoWriteRegistryAction()
{
}

void TKinokoWriteRegistryAction::Initialize(void)
{
}

int TKinokoWriteRegistryAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoWriteRegistryAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    _Registry->SetLongValue(_RegistryPath, *_Register);

    return 0;
}

void TKinokoWriteRegistryAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "WriteRegistry" << hex << "\t";
    os << _RegistryPath << ", " << "[" << _Register << "]";
    os << dec << endl;
}



TKinokoReadTimeAction::TKinokoReadTimeAction(int* Register)
{
    _Register = Register;
}

TKinokoReadTimeAction::~TKinokoReadTimeAction()
{
}

void TKinokoReadTimeAction::Initialize(void)
{
}

int TKinokoReadTimeAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoReadTimeAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    *_Register = TMushDateTime::SecSinceEpoch();

    return 0;
}

void TKinokoReadTimeAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "ReadTime" << hex << "\t";
    os << "[" << _Register << "]";
    os << dec << endl;
}



TKinokoTerminateAction::TKinokoTerminateAction(TKinokoTrigger* Trigger)
{
    _Trigger = Trigger;
}

TKinokoTerminateAction::~TKinokoTerminateAction()
{
}

void TKinokoTerminateAction::Initialize(void)
{
}

int TKinokoTerminateAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoTerminateAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    _Trigger->Stop();
    return 0;
}

void TKinokoTerminateAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "Terminate " << endl;
}



TKinokoSuspendAction::TKinokoSuspendAction(int Time_sec, int Time_usec)
{
    _Time_sec = Time_sec;
    _Time_usec = Time_usec;
}

TKinokoSuspendAction::~TKinokoSuspendAction()
{
}

void TKinokoSuspendAction::Initialize(void)
{
}

int TKinokoSuspendAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoSuspendAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    TMushRealTimeTimer(_Time_sec, _Time_usec).Suspend();
  
    return 0;
}

void TKinokoSuspendAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "Suspend " << hex << "\t";
    os << _Time_sec << ", " << _Time_usec;
    os << dec << endl;
}



TKinokoEchoAction::TKinokoEchoAction(const string& Message, int* Register)
{
    _Message = Message;
    _Register = Register;
}

TKinokoEchoAction::~TKinokoEchoAction()
{
}

void TKinokoEchoAction::Initialize(void)
{
}

int TKinokoEchoAction::NextPacketSize(void)
{
    return 0;
}

int TKinokoEchoAction::GoNext(void* DataBuffer) throw(TKinokoException)
{
    cout << _Message;
    
    if (_Register) {
	cout << *_Register;
    }
    
    cout << endl;

    return 0;
}

void TKinokoEchoAction::Dump(ostream& os, const string& Indent)
{
    os << Indent << "Echo " << "\t";

    os << "\"" << _Message << "\"";

    if (_Register != 0) {
	os << ", [" << _Register << "]";
    }

    os << endl;
}
