/* KinokoModuleMessenger.cc */
/* Created by Enomoto Sanshiro on 8 October 1999. */
/* Last updated by Enomoto Sanshiro on 23 August 2001. */


#include <strstream>
#include "RoomDeviceFactory.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoReadoutAction.hh"
#include "KinokoMetaDeviceMessenger.hh"
#include "KinokoModuleMessenger.hh"

using namespace std;


TKinokoModuleMessenger::TKinokoModuleMessenger(const string& Name, TKinokoDaqBuilder* DaqBuilder)
: TParaObjectPrototype(Name)
{
    _DaqBuilder = DaqBuilder;
    _Module = 0;

    _ChannelList = 0;
}

TKinokoModuleMessenger::~TKinokoModuleMessenger()
{
    delete _ChannelList;
}

void TKinokoModuleMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TParaObjectPrototype::Construct(ClassName, ArgumentList);
}

string TKinokoModuleMessenger::ModuleName(void)
{
    return _ModuleName;
}

int TKinokoModuleMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "assign") || (Message == "Assign")) {
        ReturnValue = Assign(ArgumentList);
    }
    else if ((Message == "read") || (Message == "Read")) {
        ReturnValue = Read(ArgumentList);
    }
    else if ((Message == "tagRead") || (Message == "TagRead")) {
        ReturnValue = TagRead(ArgumentList);
    }
    else if ((Message == "blockRead") || (Message == "BlockRead")) {
        ReturnValue = BlockRead(ArgumentList);
    }
    else if ((Message == "sequentialRead") || (Message == "SequentialRead")) {
        ReturnValue = SequentialRead(ArgumentList);
    }
    else if ((Message == "enable") || (Message == "Enable")) {
        ReturnValue = Enable(ArgumentList);
    }
    else if ((Message == "disable") || (Message == "Disable")) {
        ReturnValue = Disable(ArgumentList);
    }
    else if ((Message == "clear") || (Message == "Clear")) {
        ReturnValue = Clear(ArgumentList);
    }
    else if ((Message == "waitData") || (Message == "WaitData")) {
        ReturnValue = WaitData(ArgumentList);
    }
    else if ((Message == "writeRegister") || (Message == "WriteRegister")) {
        ReturnValue = WriteRegister(ArgumentList);
    }
    else if ((Message == "readRegister") || (Message == "ReadRegister")) {
        ReturnValue = ReadRegister(ArgumentList);
    }
    else {
	ReturnValue = MiscControl(Message, ArgumentList);
    }

    return 1;
}

TParaValue TKinokoModuleMessenger::Read(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Too few arguments: " + 
	    _ModuleName + ".read(ChannelList channel_list)"
	);
    }

    TKinokoReadoutChannelList* ChannelList = GetChannelListOf(ArgumentList[0]);
    if (ChannelList == 0) {
	throw TScriptException(
	    "invalid argument: " + 
	    _ModuleName + ".read(ChannelList channel_list)"
	);
    } 
	    
    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int AddressWidth = _Module->AddressBitLength();
    int DataWidth = _Module->DataBitLength();

    string SectionName = _ModuleName;
    TKinokoIndexedDataSection* DataSection = new TKinokoIndexedDataSection(
	_DaqBuilder->DataSource(), SectionName, _SectionId
    );
    DataSection->SetWidth(AddressWidth, DataWidth);

    TKinokoSingleReadAction* ReadoutAction = new TKinokoSingleReadAction(
	_Module, DataSection
    );

    for (unsigned i = 0; i < ChannelList->ChannelList().size(); i++) {
	int Channel = ChannelList->ChannelList()[i];
	ReadoutAction->AddChannel(Channel);
    }

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction, DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".read(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::TagRead(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few argument[s]: " + 
	    _ModuleName + ".tagRead(ChannelList channel_list)"
	);
    } 

    TKinokoReadoutChannelList* ChannelList = GetChannelListOf(ArgumentList[0]);
    if (ChannelList == 0) {
	throw TScriptException(
	    "invalid argument: " + 
	    _ModuleName + ".read(ChannelList channel_list)"
	);
    } 
	    
    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int ValueWidth = _Module->DataBitLength();

    string SectionName = _ModuleName;
    TKinokoTaggedDataSection* DataSection = new TKinokoTaggedDataSection(
	_DaqBuilder->DataSource(), SectionName, _SectionId
    );

    TKinokoTagReadAction* ReadoutAction = new TKinokoTagReadAction(
	_Module, DataSection
    );

    const vector<pair<int, string> >& TaggedChannelList = ChannelList->TaggedChannelList();
    for (unsigned i = 0; i < TaggedChannelList.size(); i++) {
	int Channel = TaggedChannelList[i].first;
	string TagName = TaggedChannelList[i].second;

	int TagIndex = DataSection->AddField(TagName, ValueWidth);
	ReadoutAction->AddChannel(Channel, TagIndex);
    }

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction, DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".tagRead(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::BlockRead(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    unsigned OffsetAddress = 0, Size = 0;
    if (ArgumentList.size() > 0) {
	OffsetAddress = ArgumentList[0]->AsLong();
    }
    if (ArgumentList.size() > 1) {
	Size = ArgumentList[1]->AsLong();
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    string SectionName = _ModuleName;
    TKinokoBlockDataSection* DataSection = new TKinokoBlockDataSection(
	_DaqBuilder->DataSource(), SectionName, _SectionId
    );
    TKinokoReadoutAction* ReadoutAction = new TKinokoBlockReadAction(
	_Module, DataSection, OffsetAddress, Size
     );

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction, DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".blockRead(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::SequentialRead(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Too few argument[s]: " + 
	    _ModuleName + ".sequentialRead(ChannelList channel_list)"
	);
    }

    TKinokoReadoutChannelList* ChannelList = GetChannelListOf(ArgumentList[0]);
    if (ChannelList == 0) {
	throw TScriptException(
	    "invalid argument: " + 
	    _ModuleName + ".sequentialRead(ChannelList channel_list)"
	);
    } 

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int AddressWidth = _Module->AddressBitLength();
    int DataWidth = _Module->DataBitLength();

    string SectionName = _ModuleName;
    TKinokoIndexedDataSection* DataSection = new TKinokoIndexedDataSection(
	_DaqBuilder->DataSource(), SectionName, _SectionId
    );
    DataSection->SetWidth(AddressWidth, DataWidth);

    TKinokoSequentialReadAction* ReadoutAction;
    ReadoutAction = new TKinokoSequentialReadAction(_Module, DataSection);

    for (unsigned i = 0; i < ChannelList->ChannelList().size(); i++) {
	int Channel = ChannelList->ChannelList()[i];
	ReadoutAction->AddChannel(Channel);
    }

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction, DataSection);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".SequencialRead(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::Enable(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoReadoutChannelList* ChannelList = 0;
    if (ArgumentList.size() > 0) {
	ChannelList = GetChannelListOf(ArgumentList[0]);
	if (ChannelList == 0) {
	    throw TScriptException(
		"invalid argument: " + 
		_ModuleName + ".enable(ChannelList channel_list)"
	    );
	}
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    TKinokoControlAction* ReadoutAction = new TKinokoControlAction(
	_Module, TKinokoControlAction::ControlId_Enable
    );

    if (ChannelList != 0) {
	for (unsigned i = 0; i < ChannelList->ChannelList().size(); i++) {
	    int Channel = ChannelList->ChannelList()[i];
	    ReadoutAction->AddChannel(Channel);
	}
    }

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".enable(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::Disable(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoReadoutChannelList* ChannelList = 0;
    if (ArgumentList.size() > 0) {
	ChannelList = GetChannelListOf(ArgumentList[0]);
	if (ChannelList == 0) {
	    throw TScriptException(
		"invalid argument: " + 
		_ModuleName + ".disable(ChannelList channel_list)"
	    );
	}
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    TKinokoControlAction* ReadoutAction = new TKinokoControlAction(
	_Module, TKinokoControlAction::ControlId_Disable
    );

    if (ChannelList != 0) {
	for (unsigned i = 0; i < ChannelList->ChannelList().size(); i++) {
	    int Channel = ChannelList->ChannelList()[i];
	    ReadoutAction->AddChannel(Channel);
	}
    }

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".disable(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::Clear(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoReadoutChannelList* ChannelList = 0;
    if (ArgumentList.size() > 0) {
	ChannelList = GetChannelListOf(ArgumentList[0]);
	if (ChannelList == 0) {
	    throw TScriptException(
		"invalid argument: " + 
		_ModuleName + ".clear(ChannelList channel_list)"
	    );
	}
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    TKinokoControlAction* ReadoutAction = new TKinokoControlAction(
	_Module, TKinokoControlAction::ControlId_Clear
    );

    if (ChannelList != 0) {
	for (unsigned i = 0; i < ChannelList->ChannelList().size(); i++) {
	    int Channel = ChannelList->ChannelList()[i];
	    ReadoutAction->AddChannel(Channel);
	}
    }

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".clear(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::WaitData(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    TKinokoReadoutAction* ReadoutAction = new TKinokoWaitDataAction(_Module);
    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(_ModuleName + ".waitData(): " + e.Message());
    }

    return TParaValue((TParaObjectPrototype*) this); 
}

TParaValue TKinokoModuleMessenger::WriteRegister(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: " + 
	    _ModuleName + ".writeRegister(long address, long value)"
	);
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int* AddressRegister;
    int* DataRegister;
    try {
	AddressRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[0], _DaqBuilder
	);
	DataRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    _ModuleName + ".writeRegister(): " + e.Message()
	);
    }

    TKinokoReadoutAction* ReadoutAction = new TKinokoWriteRegisterAction(
	_Module, AddressRegister, DataRegister
    );

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".writeRegister(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::ReadRegister(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few arguments: " + 
	    _ModuleName + ".readRegister(long address, Register register)"
	);
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int* AddressRegister;
    int* DataRegister;
    try {
	AddressRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[0], _DaqBuilder
	);
	DataRegister = TKinokoRegisterMessenger::GetRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    _ModuleName + ".writeRegister(): " + e.Message()
	);
    }

    TKinokoReadoutAction* ReadoutAction = new TKinokoReadRegisterAction(
	_Module, AddressRegister, DataRegister
    );

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".readRegister(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TParaValue TKinokoModuleMessenger::MiscControl(const string& ControlName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int ControlId;
    try {
	ControlId = _Module->MiscControlIdOf(ControlName);
    }
    catch (THardwareException &e) {
	throw TScriptException(
	    "unknown method: " + _ModuleName + "." + ControlName + "(...)"
	);
    }

    vector<int*> ControlParameterList;
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	int* ParameterRegister;
	try {
	    ParameterRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
		*ArgumentList[i], _DaqBuilder
	    );
	    ControlParameterList.push_back(ParameterRegister);
	}
	catch (TScriptException &e) {
	    throw TScriptException(
		_ModuleName + "." + ControlName + "(...): " + e.Message()
	    );
	}
    }

    TKinokoReadoutAction* ReadoutAction = new TKinokoMiscControlAction(
	_Module, ControlId, ControlParameterList
    );

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + "." + ControlName + "(...): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}

TKinokoReadoutChannelList* TKinokoModuleMessenger::GetChannelListOf(TParaValue* Argument)
{
    if (Argument->IsLong()) {
	if (_ChannelList == 0) {
	    _ChannelList = new TKinokoReadoutChannelList();
	}
	else {
	    _ChannelList->Clear();
	}
	_ChannelList->ReadChannelBits(Argument->AsLong());
	
	return _ChannelList;
    }

    if (Argument->IsObject("ChannelList")) {
	return (TKinokoReadoutChannelList*) Argument->AsObject();
    } 

    return 0;
}



TKinokoVmeModuleMessenger::TKinokoVmeModuleMessenger(TKinokoDaqBuilder* DaqBuilder)
: TKinokoModuleMessenger("VmeModule", DaqBuilder)
{
    _VmeModule = 0;
}

TKinokoVmeModuleMessenger::~TKinokoVmeModuleMessenger()
{
}

TParaObjectPrototype* TKinokoVmeModuleMessenger::Clone(void)
{
    return new TKinokoVmeModuleMessenger(_DaqBuilder);
}

void TKinokoVmeModuleMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoModuleMessenger::Construct(ClassName, ArgumentList);

    if (ArgumentList.size() > 1) {
	_ModuleName = ArgumentList[1]->AsString();
    }
    else {
	_ModuleName = this->ObjectName();
    }

    if (ArgumentList.size() > 2) {
	if (! ArgumentList[2]->IsLong()) {
	    throw TScriptException(
		"TKinokoVmeModuleMessenger::Construct()",
		"invalid SectionId argument (int value is expected)"
	    );
	}
	_SectionId = ArgumentList[2]->AsLong();
    }
    else {
	_SectionId = TKinokoDataSection::SectionId_Unknown;
    }

    if (ArgumentList.size() > 0) {
        Assign(ArgumentList);
    }
}

int TKinokoVmeModuleMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    return TKinokoModuleMessenger::DispatchMessage(
	Message, ArgumentList, ReturnValue
    );
}

TParaValue TKinokoVmeModuleMessenger::Assign(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Too few arguments: " + 
	    _ModuleName + ".assign(string device_name)"
	);
    }

    string DeviceName = ArgumentList[0]->AsString();
    TRoomDeviceFactory* DeviceFactory = TRoomDeviceFactory::GetInstance();
    try {
        _VmeModule = DeviceFactory->CreateVmeModule(DeviceName);
	_Module = _VmeModule;
	_DaqBuilder->RegisterModule(_ModuleName, _Module);
	_Module->SetModuleName(_ModuleName);
    }
    catch (THardwareException& e) {
	 throw TScriptException(
	     "unknown vme module: " + DeviceName + ": " + e.Message()
	 );
    }
    
    return TParaValue((long) 0); 
}



TKinokoCamacModuleMessenger::TKinokoCamacModuleMessenger(TKinokoDaqBuilder* DaqBuilder)
: TKinokoModuleMessenger("CamacModule", DaqBuilder)
{
    _CamacModule = 0;
}

TKinokoCamacModuleMessenger::~TKinokoCamacModuleMessenger()
{
}

TParaObjectPrototype* TKinokoCamacModuleMessenger::Clone(void)
{
    return new TKinokoCamacModuleMessenger(_DaqBuilder);
}

void TKinokoCamacModuleMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoModuleMessenger::Construct(ClassName, ArgumentList);

    if (ArgumentList.size() > 1) {
	_ModuleName = ArgumentList[1]->AsString();
    }
    else {
	_ModuleName = this->ObjectName();
    }

    if (ArgumentList.size() > 2) {
	if (! ArgumentList[2]->IsLong()) {
	    throw TScriptException(
		"TKinokoVmeModuleMessenger::Construct()",
		"invalid SectionId argument (int value is expected)"
	    );
	}
	_SectionId = ArgumentList[2]->AsLong();
    }
    else {
	_SectionId = TKinokoDataSection::SectionId_Unknown;
    }

    if (ArgumentList.size() > 0) {
        Assign(ArgumentList);
    }
}

int TKinokoCamacModuleMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "transact") || (Message == "Transact")) {
        ReturnValue = Transact(ArgumentList);
    }
    else {
	return TKinokoModuleMessenger::DispatchMessage(
	    Message, ArgumentList, ReturnValue
	);
    }

    return 1;
}

TParaValue TKinokoCamacModuleMessenger::Assign(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Too few arguments: " + 
	    _ModuleName + ".assign(string device_name)"
	);
    }

    string DeviceName = ArgumentList[0]->AsString();
    TRoomDeviceFactory* DeviceFactory = TRoomDeviceFactory::GetInstance();
    try {
        _CamacModule = DeviceFactory->CreateCamacModule(DeviceName);
	_Module = _CamacModule;
	_DaqBuilder->RegisterModule(_ModuleName, _Module);
	_Module->SetModuleName(_ModuleName);
    }
    catch (THardwareException& e) {
	 throw TScriptException(
	     "unknown camac module: " + DeviceName + ": " + e.Message()
	 );
    }
    
    return TParaValue((long) 0); 
}

TParaValue TKinokoCamacModuleMessenger::Transact(std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few argument[s]: " + _ModuleName + 
             ".transact(int function, int address, ...)"
	);
    }

    if (_Module == 0) {
	throw TScriptException("driver not assigned: " + _ModuleName);
    }

    int* FunctionRegister;
    int* AddressRegister;
    int* DataRegister = 0;
    int* QResponseRegister = 0;
    int* XResponseRegister = 0;
    try {
	FunctionRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[0], _DaqBuilder
	);
	AddressRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
	    *ArgumentList[1], _DaqBuilder
	);
	if (ArgumentList.size() > 2) {
	    DataRegister = TKinokoRegisterMessenger::GetConstRegisterOf(
		*ArgumentList[2], _DaqBuilder
	    );
	}
	if (ArgumentList.size() > 3) {
	    QResponseRegister = TKinokoRegisterMessenger::GetRegisterOf(
		*ArgumentList[3], _DaqBuilder
	    );
	}
	if (ArgumentList.size() > 4) {
	    XResponseRegister = TKinokoRegisterMessenger::GetRegisterOf(
		*ArgumentList[4], _DaqBuilder
	    );
	}
    }
    catch (TScriptException &e) {
        throw TScriptException(
            _ModuleName + ".transact(): " + e.Message()
	);
    }

    TKinokoReadoutAction* ReadoutAction = new TKinokoCamacTransactAction(
        _CamacModule, *FunctionRegister, *AddressRegister,
	DataRegister, QResponseRegister, XResponseRegister
    );

    try {
	_DaqBuilder->AddReadoutAction(ReadoutAction);
    }
    catch (TKinokoException &e) {
	throw TScriptException(
	    _ModuleName + ".transact(): " + e.Message()
	);
    }

    return TParaValue((long) 0); 
}



TKinokoSoftwareModuleMessenger::TKinokoSoftwareModuleMessenger(TKinokoDaqBuilder* DaqBuilder)
: TKinokoModuleMessenger("SoftwareModule", DaqBuilder)
{
    _SoftwareModule = 0;
}

TKinokoSoftwareModuleMessenger::~TKinokoSoftwareModuleMessenger()
{
}

TParaObjectPrototype* TKinokoSoftwareModuleMessenger::Clone(void)
{
    return new TKinokoSoftwareModuleMessenger(_DaqBuilder);
}

void TKinokoSoftwareModuleMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    TKinokoModuleMessenger::Construct(ClassName, ArgumentList);

    if (ArgumentList.size() > 1) {
	_ModuleName = ArgumentList[1]->AsString();
    }
    else {
	_ModuleName = this->ObjectName();
    }

    if (ArgumentList.size() > 2) {
	if (! ArgumentList[2]->IsLong()) {
	    throw TScriptException(
		"TKinokoVmeModuleMessenger::Construct()",
		"invalid SectionId argument (int value is expected)"
	    );
	}
	_SectionId = ArgumentList[2]->AsLong();
    }
    else {
	_SectionId = TKinokoDataSection::SectionId_Unknown;
    }

    if (ArgumentList.size() > 0) {
        Assign(ArgumentList);
    }
}

int TKinokoSoftwareModuleMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    return TKinokoModuleMessenger::DispatchMessage(
	Message, ArgumentList, ReturnValue
    );
}

TParaValue TKinokoSoftwareModuleMessenger::Assign(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "Too few arguments: " + 
	    _ModuleName + ".assign(string device_name)"
	);
    }

    string DeviceName = ArgumentList[0]->AsString();
    TRoomDeviceFactory* DeviceFactory = TRoomDeviceFactory::GetInstance();
    try {
        _SoftwareModule = DeviceFactory->CreateSoftwareModule(DeviceName);
	_Module = _SoftwareModule;
	_DaqBuilder->RegisterModule(_ModuleName, _Module);
	_Module->SetModuleName(_ModuleName);
    }
    catch (THardwareException& e) {
	 throw TScriptException(
	     "unknown software module: " + DeviceName + ": " + e.Message()
	 );
    }
    
    return TParaValue((long) 0); 
}



TKinokoReadoutChannelList::TKinokoReadoutChannelList(void)
: TParaObjectPrototype("ChannelList")
{
}

TKinokoReadoutChannelList::~TKinokoReadoutChannelList()
{
}

TParaObjectPrototype* TKinokoReadoutChannelList::Clone(void)
{
    return new TKinokoReadoutChannelList();
}

void TKinokoReadoutChannelList::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	if (! ArgumentList[i]->IsLong()) {
	    throw TScriptException("interger argument expected");
	}

	ReadChannelBits(ArgumentList[i]->AsLong());
    }
}

int TKinokoReadoutChannelList::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "add") || (Message == "Add")) {
        ReturnValue = Add(ArgumentList);
    }
    else {
	return 0;
    }

    return 1;
}

const vector<int>& TKinokoReadoutChannelList::ChannelList(void)
{
    return _ChannelList;
}

const vector<pair<int, string> >& TKinokoReadoutChannelList::TaggedChannelList(void)
{
    return _TaggedChannelList;
}

TParaValue TKinokoReadoutChannelList::Add(vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("too few arguments");
    }
    
    if (! ArgumentList[0]->IsLong()) {
	throw TScriptException("integer argument expected");
    }
    int Channel = ArgumentList[0]->AsLong();
    
    string TagName;
    if (ArgumentList.size() > 1) {
	if (! ArgumentList[1]->IsString()) {
	    throw TScriptException("string argument expected");
	}
	TagName = ArgumentList[1]->AsString();
    }

    AddChannel(Channel, TagName);

    return TParaValue((TParaObjectPrototype*) this);
}

void TKinokoReadoutChannelList::AddChannel(int Channel, string TagName)
{
    if (TagName.empty()) {
	char TagNameBuffer[16];
	ostrstream TagNameStream(TagNameBuffer, sizeof(TagNameBuffer));
	TagNameStream << Channel << ends;
	TagName = TagNameBuffer;
    }

    unsigned Index;
    for (Index = 0; Index < _ChannelList.size(); Index++) {
	if (_ChannelList[Index] == Channel) {
	    _TaggedChannelList[Index].second = TagName;
	    break;
	}
    }
    if (Index == _ChannelList.size()) {
	_ChannelList.push_back(Channel);
	_TaggedChannelList.push_back(make_pair(Channel, TagName));
    }
}

void TKinokoReadoutChannelList::ReadChannelBits(unsigned long ChannelBits)
{
    int Channel = 0;
    while (ChannelBits > 0) {
	if (ChannelBits & 0x0001) {
	    AddChannel(Channel);
	}
	Channel++;
	ChannelBits >>= 1;
    }
}

void TKinokoReadoutChannelList::Clear(void)
{
    _ChannelList.erase(_ChannelList.begin(), _ChannelList.end());
    _TaggedChannelList.erase(_TaggedChannelList.begin(), _TaggedChannelList.end());
}
