/* KinokoReadoutAction.hh */
/* Created by Enomoto Sanshiro on 25 September 1999. */
/* Last updated by Enomoto Sanshiro on 30 September 2002. */


#ifndef __KinokoReadoutAction_hh__
#define __KinokoReadoutAction_hh__


#include <iostream>
#include <string>
#include <vector>
#include "RoomModule.hh"
#include "RoomCamacAccess.hh"
#include "KinokoStream.hh"
#include "KinokoDataSection.hh"
#include "KinokoIndexedDataSection.hh"
#include "KinokoTaggedDataSection.hh"
#include "KinokoBlockDataSection.hh"
#include "KinokoNestedDataSection.hh"


class TKinokoReadoutAction {
  public:
    TKinokoReadoutAction(void);
    virtual ~TKinokoReadoutAction();
    virtual void Initialize(void);
    virtual void Go(TKinokoOutputStream* OutputStream) throw(TKinokoException);
    virtual int NextPacketSize(void) = 0;
    virtual int GoNext(void* DataBuffer) throw(TKinokoException) = 0;
  public:
    virtual void Dump(std::ostream& os, const std::string& Indent = "") = 0;
};


class TKinokoBlockReadAction: public TKinokoReadoutAction {
  public:
    TKinokoBlockReadAction(TRoomModule* Module, TKinokoBlockDataSection* DataSection, int Address, int Size);
    virtual ~TKinokoBlockReadAction();
    virtual void Initialize(void);
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    int _Address, _Size;
    int _DataBufferSize;
    TRoomModule* _Module;
    TKinokoBlockDataSection* _DataSection;
    TKinokoBlockDataSectionFormatter* _Formatter;
};


class TKinokoSingleReadAction: public TKinokoReadoutAction {
  public:
    TKinokoSingleReadAction(TRoomModule* Module, TKinokoIndexedDataSection* DataSection);
    virtual ~TKinokoSingleReadAction();
    virtual void Initialize(void);
    virtual void AddChannel(int Channel);
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    int _DataBufferSize;
    int* _ChannelList;
    int _ChannelListSize, _ChannelListCapacity;
    TRoomModule* _Module;
    TKinokoIndexedDataSection* _DataSection;
    TKinokoIndexedDataSectionFormatter* _Formatter;
};


class TKinokoSequentialReadAction: public TKinokoReadoutAction {
  public:
    TKinokoSequentialReadAction(TRoomModule* Module, TKinokoIndexedDataSection* DataSection);
    virtual ~TKinokoSequentialReadAction();
    virtual void Initialize(void);
    virtual void AddChannel(int Channel);
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    int _DataBufferSize;
    int* _ChannelList;
    int _ChannelListSize, _ChannelListCapacity;
    int _NumberOfSamples;
    TRoomModule* _Module;
    TKinokoIndexedDataSection* _DataSection;
    TKinokoIndexedDataSectionFormatter* _Formatter;
};


class TKinokoTagReadAction: public TKinokoReadoutAction {
  public:
    TKinokoTagReadAction(TRoomModule* Module, TKinokoTaggedDataSection* DataSection);
    virtual ~TKinokoTagReadAction();
    virtual void Initialize(void);
    virtual void AddChannel(int Channel, int TagIndex);
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    int _DataBufferSize;
    int* _ChannelList;
    int* _TagIndexList;
    int _ChannelListSize, _ChannelListCapacity;
    TRoomModule* _Module;
    TKinokoTaggedDataSection* _DataSection;
    TKinokoTaggedDataSectionFormatter* _Formatter;
};


class TKinokoControlAction: public TKinokoReadoutAction {
  public:
    TKinokoControlAction(TRoomModule* Module, int ControlId);
    virtual ~TKinokoControlAction();
    virtual void AddChannel(int Channel);
    virtual void Transact(int Channel) throw(TKinokoException);
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    TRoomModule* _Module;
    int _ControlId;
    int* _ChannelList;
    int _ChannelListSize, _ChannelListCapacity;
  public:
    enum TControlId {
	ControlId_Enable,
	ControlId_Disable,
	ControlId_Clear
    };
};


class TKinokoWaitDataAction: public TKinokoReadoutAction {
  public:
    TKinokoWaitDataAction(TRoomModule* Module);
    virtual ~TKinokoWaitDataAction();
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    TRoomModule* _Module;
};


class TKinokoWriteRegisterAction: public TKinokoReadoutAction {
  public:
    TKinokoWriteRegisterAction(TRoomModule* Module, int* AddressRegister, int* DataRegister);
    virtual ~TKinokoWriteRegisterAction();
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    TRoomModule* _Module;
    int* _AddressRegister;
    int* _DataRegister;
};


class TKinokoReadRegisterAction: public TKinokoReadoutAction {
  public:
    TKinokoReadRegisterAction(TRoomModule* Module, int* AddressRegister, int* DataRegister);
    virtual ~TKinokoReadRegisterAction();
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    TRoomModule* _Module;
    int* _AddressRegister;
    int* _DataRegister;
};


class TKinokoMiscControlAction: public TKinokoReadoutAction {
  public:
    TKinokoMiscControlAction(TRoomModule* Module, int ControlId, const std::vector<int*>& ParameterRegisterList);
    virtual ~TKinokoMiscControlAction();
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    TRoomModule* _Module;
    int _ControlId;
    int** _ParameterRegisterList;
    int* _ParameterList;
    int _NumberOfParameters;
};


class TKinokoCamacControllerAction: public TKinokoReadoutAction {
  public:
    TKinokoCamacControllerAction(TRoomCamacController* Controller, const std::string& ControllerName, int ActionId);
    virtual ~TKinokoCamacControllerAction();
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  public:
    enum TCamacControllerActionId {
        CamacControllerAction_Initialize,
        CamacControllerAction_Clear,
        CamacControllerAction_SetInhibition,
        CamacControllerAction_ReleaseInhibition
    };
  protected:
    TRoomCamacController* _CamacController;
    std::string _ControllerName;
    int _ActionId;
};


class TKinokoCamacTransactAction: public TKinokoReadoutAction {
  public:
    TKinokoCamacTransactAction(TRoomCamacModule* CamacModule, int Function, int Address, int* DataRegister, int* QResponseRegister, int* XResponseRegister);
    virtual ~TKinokoCamacTransactAction();
    virtual int NextPacketSize(void);
    virtual int GoNext(void* DataBuffer) throw(TKinokoException);
    virtual void Dump(std::ostream& os, const std::string& Indent = "");
  protected:
    TRoomCamacModule* _CamacModule;
    int _Function, _Address;
    int* _DataRegister;
    int* _QResponseRegister;
    int* _XResponseRegister;
};


#endif
