/* KinokoTransaction.cc */
/* Created by Enomoto Sanshiro on 16 April 1999. */
/* Last updated by Enomoto Sanshiro on 29 March 2002. */


#include <string>
#include "KinokoStream.hh"
#include "KinokoReadoutSequence.hh"
#include "KinokoDataStreamFormatter.hh"
#include "KinokoTransaction.hh"

using namespace std;


TKinokoTransaction::TKinokoTransaction(TKinokoDataSource* DataSource, TKinokoOutputStream* OutputStream)
{
    _DataSource = DataSource;
    _OutputStream = OutputStream;

    _StreamFormatter = 0;
    _CommandPacketSize = TKinokoDataStreamFormatter::CommandPacketSize();
    _TrailerPacketSize = TKinokoDataStreamFormatter::TrailerPacketSize();

    _LastTriggerSourceId = 0;
    _LastCommand = "";
    _LastTriggerSequence = 0;    
    _LastCommandSequence = 0;    
}

TKinokoTransaction::~TKinokoTransaction() 
{
    map<int, TKinokoReadoutSequence*>::iterator ReadoutSequenceIterator;
    for (
        ReadoutSequenceIterator = _ReadoutSequenceTable.begin();
        ReadoutSequenceIterator != _ReadoutSequenceTable.end();
        ReadoutSequenceIterator++
    ){
        delete (*ReadoutSequenceIterator).second;
    }

    map<string, TKinokoReadoutSequence*>::iterator DaqSequenceIterator;
    for (
        DaqSequenceIterator = _DaqEventSequenceTable.begin();
        DaqSequenceIterator != _DaqEventSequenceTable.end();
        DaqSequenceIterator++
    ){
        delete (*DaqSequenceIterator).second;
    }

    delete _StreamFormatter;
}

void TKinokoTransaction::AddReadoutSequence(int TriggerSourceId, TKinokoReadoutSequence* ReadoutSequence)
{
    _ReadoutSequenceTable[TriggerSourceId] = ReadoutSequence;
}

void TKinokoTransaction::AddDaqEventSequence(const string& EventName, TKinokoReadoutSequence* ReadoutSequence)
{
    _DaqEventSequenceTable[EventName] = ReadoutSequence;
}

void TKinokoTransaction::SendDataDescriptorPacket(void) throw(TKinokoException)
{
    if (_StreamFormatter == 0) {
	_StreamFormatter = new TKinokoDataStreamFormatter(_DataSource);
    }

    void* DescriptorArea;
    unsigned DescriptorSize = _StreamFormatter->DescriptorPacketSize();
    do {
	_OutputStream->NextEntry(DescriptorArea, DescriptorSize);
    } while (DescriptorArea == 0);
    _StreamFormatter->WriteDescriptor(DescriptorArea);
    _OutputStream->Flush(DescriptorArea, DescriptorSize);
}

void TKinokoTransaction::Initialize(void) throw(TKinokoException) 
{
    if (_StreamFormatter == 0) {
	_StreamFormatter = new TKinokoDataStreamFormatter(_DataSource);
    }

    void* CommandArea;
    int Command = TKinokoDataStreamFormatter::Command_RunBegin;
    do {
	_OutputStream->NextEntry(CommandArea, _CommandPacketSize);
    } while (CommandArea == 0);
    _StreamFormatter->WriteCommand(CommandArea, Command);
    _OutputStream->Flush(CommandArea, _CommandPacketSize);

    map<int, TKinokoReadoutSequence*>::iterator ReadoutSequenceIterator;
    for (
        ReadoutSequenceIterator = _ReadoutSequenceTable.begin();
        ReadoutSequenceIterator != _ReadoutSequenceTable.end();
        ReadoutSequenceIterator++
    ){
        (*ReadoutSequenceIterator).second->Initialize();
    }

    map<string, TKinokoReadoutSequence*>::iterator DaqEventSequenceIterator;
    for (
        DaqEventSequenceIterator = _DaqEventSequenceTable.begin();
        DaqEventSequenceIterator != _DaqEventSequenceTable.end();
        DaqEventSequenceIterator++
    ){
        (*DaqEventSequenceIterator).second->Initialize();
    }

    ReadoutSequenceIterator = _ReadoutSequenceTable.find(TrapId_RunBegin);
    if (ReadoutSequenceIterator != _ReadoutSequenceTable.end()) {
	int* ParameterList = 0;
	int NumberOfParameters = 0;
	ExecuteSequence(
	    (*ReadoutSequenceIterator).second, 
	    ParameterList, NumberOfParameters, 
	    TKinokoDataStreamFormatter::Trailer_Trap
	);
    }
}

void TKinokoTransaction::Finalize(void) throw(TKinokoException) 
{
    map<int, TKinokoReadoutSequence*>::iterator ReadoutSequenceIterator;
    ReadoutSequenceIterator = _ReadoutSequenceTable.find(TrapId_RunEnd);
    if (ReadoutSequenceIterator != _ReadoutSequenceTable.end()) {
	int* ParameterList = 0;
	int NumberOfParameters = 0;
	ExecuteSequence(
	    (*ReadoutSequenceIterator).second, 
	    ParameterList, NumberOfParameters, 
	    TKinokoDataStreamFormatter::Trailer_Trap
	);
    }

    void* CommandArea;
    int Command = TKinokoDataStreamFormatter::Command_RunEnd;
    do {
	_OutputStream->NextEntry(CommandArea, _CommandPacketSize);
    } while (CommandArea == 0);
    _StreamFormatter->WriteCommand(CommandArea, Command);
    _OutputStream->Flush(CommandArea, _CommandPacketSize);
}

int TKinokoTransaction::OnTrigger(int TriggerSourceId, int* ParameterList, int NumberOfParameters) throw(TKinokoException) 
{
    if (TriggerSourceId != _LastTriggerSourceId) {
	map<int, TKinokoReadoutSequence*>::iterator SequenceIterator;    
	SequenceIterator = _ReadoutSequenceTable.find(TriggerSourceId);
	_LastTriggerSourceId = TriggerSourceId;

	if (SequenceIterator != _ReadoutSequenceTable.end()) {
	    _LastTriggerSequence = (*SequenceIterator).second;
	}
	else {
	    _LastTriggerSequence = 0;
	}
    }

    if (_LastTriggerSequence == 0) {
	return 0;
    }

    ExecuteSequence(
	_LastTriggerSequence, ParameterList, NumberOfParameters, 
	TKinokoDataStreamFormatter::Trailer_Event
    );

    return 1;
}

int TKinokoTransaction::OnCommand(const string& Command, int* ParameterList, int NumberOfParameters) throw(TKinokoException)
{
    if (Command != _LastCommand) {
	map<string, TKinokoReadoutSequence*>::iterator SequenceIterator;    
	SequenceIterator = _DaqEventSequenceTable.find(Command);
	_LastCommand = Command;

	if (SequenceIterator != _DaqEventSequenceTable.end()) {
	    _LastCommandSequence = (*SequenceIterator).second;
	}
	else {
	    _LastCommandSequence = 0;
	}
    }

    if (_LastCommandSequence == 0) {
	return 0;
    }

    ExecuteSequence(
	_LastCommandSequence, ParameterList, NumberOfParameters, 
	TKinokoDataStreamFormatter::Trailer_Command
    );

    return 1;
}

void TKinokoTransaction::ExecuteSequence(TKinokoReadoutSequence* Sequence, int* ParameterList, int NumberOfParameters, int TrailerValue) throw(TKinokoException)
{
    if (NumberOfParameters > 0) {
	Sequence->LoadArguments(ParameterList, NumberOfParameters);
    }

    Sequence->Go(_OutputStream);

    void* TrailerArea;
    do {
	_OutputStream->NextEntry(TrailerArea, _TrailerPacketSize);
    } while (TrailerArea == 0);
    _StreamFormatter->WriteTrailer(TrailerArea, TrailerValue);
    _OutputStream->Flush(TrailerArea, _TrailerPacketSize);
}

void TKinokoTransaction::Dump(ostream& os)
{
    map<int, TKinokoReadoutSequence*>::iterator ReadoutSequenceIterator;
    for (
        ReadoutSequenceIterator = _ReadoutSequenceTable.begin();
        ReadoutSequenceIterator != _ReadoutSequenceTable.end();
        ReadoutSequenceIterator++
    ){
	os << "trigger ";

	int TriggerId = (*ReadoutSequenceIterator).first;
	switch (TriggerId) {
	case TrapId_RunBegin:
	    os << "RUN_BEGIN:" << endl;
	    break;
	case TrapId_RunEnd:
	    os << "RUN_END:" << endl;
	    break;
	case TrapId_RunSuspend:
	    os << "RUN_SUSPEND:" << endl;
	    break;
	case TrapId_RunResume:
	    os << "RUN_RESUME:" << endl;
	    break;
	default:
	    os << hex << TriggerId << dec << ":" << endl;
	}

        (*ReadoutSequenceIterator).second->Dump(os, "\t");
	os << endl;
    }

    map<string, TKinokoReadoutSequence*>::iterator DaqSequenceIterator;
    for (
        DaqSequenceIterator = _DaqEventSequenceTable.begin();
        DaqSequenceIterator != _DaqEventSequenceTable.end();
        DaqSequenceIterator++
    ){
	os << "command \"" << (*DaqSequenceIterator).first << "\":" << endl;
        (*DaqSequenceIterator).second->Dump(os, "\t");
	os << endl;
    }
}
