/* KinokoKdfStorage.cc */
/* Created by Enomoto Sanshiro on 9 January 2000. */
/* Last updated by Enomoto Sanshiro on 6 October 2002. */


#include <strstream>
#include <string>
#include <cstring>
#include "MushDecoratedFile.hh"
#include "MushFramedFile.hh"
#include "MushChainedFile.hh"
#include "MushDataEncoder.hh"
#include "MushFileSystem.hh"
#include "ParaTokenizer.hh"
#include "KinokoDataSource.hh"
#include "KinokoDataSection.hh"
#include "KinokoFileStream.hh"
#include "KinokoStorage.hh"
#include "KinokoKdfStorage.hh"


using namespace std;


#define FullVersion(Major, Minor) ((Major << 16) | Minor)
#define MajorVersion(FullVersion) ((FullVersion >> 16) & 0xffff)
#define MinorVersion(FullVersion) ((FullVersion >> 0) & 0xffff)

static const int MagicNumber = ('.' << 24) | ('k' << 16) | ('d' << 8) | 'f';
static const int PreambleVersion = FullVersion(1, 2);
static const int HeaderVersion = FullVersion(1, 1);
static const int DataAreaVersion = FullVersion(2, 1);


//* Chained File Parameters *//
// If data size becomes more than MaxFileSize, data file will
// be divided into small files like "DataFileName.1", "DataFileName.2", ...
// Packets smaller than MinimumTailPacketSize may not be the last packet of 
// each divided file (except the very last packet).
static const int MaxFileSize = 0x10000000;  // 256MB
static const int MinimumTailPacketSize = 0x0005; // FramedFile header is 4byte


// Free Area //
static const int FreeAreaSize = 256;
static char FreeArea[FreeAreaSize];
static const char* FreeAreaMessage = (
    "0...1...2...\n\n"
    "KDF: Kinoko Data File\n"
    "---------------------\n"
    "tools to see the contents:\n"
    "     kdfdump: shows data in text format\n"
    "     kdfprofile: displays some basic information and statistics\n"
    "     kdfcheck: dumps all data packets\n"
    "\n\f\n"
);

// Preamble //
enum PreambleOffsets {
    MagicNumberOffset,
    PreambleVersionOffset,
    HeaderVersionOffset,
    HeaderAreaOffsetOffset,
    DataAreaVersionOffset,
    DataAreaOffsetOffset,
    DataAreaFormatFlagsOffset,
    PreambleTrailerOffset,
    NumberOfPreambleWords
};
static U32bit PreambleBuffer[NumberOfPreambleWords];



TKinokoKdfStorage::TKinokoKdfStorage(const string& FileName, bool IsRawStorage)
{
    _FileName = FileName;
    _IsRawStorage = IsRawStorage;

    _InputRawFile = 0;
    _OutputRawFile = 0;

    _InputFile = 0;
    _OutputFile = 0;

    _InputStream = 0;
    _OutputStream = 0;
    
    _HeaderString = 0;
    _DataAreaOffset = -1;
}

TKinokoKdfStorage::~TKinokoKdfStorage()
{
    delete[] _HeaderString;

    delete _InputStream;
    delete _OutputStream;

    if (_InputFile) {
	delete _InputFile;
    }
    else {
	delete _InputRawFile;
    }

    if (_OutputFile) {
	delete _OutputFile;
    }
    else {
	delete _OutputRawFile;
    }
}

void TKinokoKdfStorage::OpenOutputRawFile(void) throw(TKinokoException)
{
    if (_OutputRawFile) {
	return;
    }

    if (! _IsOverWriteAllowed) {
	if (TMushFileAttribute(_FileName).IsReadable()) {
	    throw TKinokoException(
		"TKinokoKdfStorage::OpenOutputRawFile()",
		"file already exists: " + _FileName
	    );
	}
    }

    try {
        _OutputRawFile = new TMushChainedFile(
	    _FileName, MaxFileSize, MinimumTailPacketSize
	);
	_OutputRawFile->OpenWriteMode(_OutputFileAccessMode);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::OpenOutputRawFile()",
	    "system call exception: " + e.Message()
        );
    }
}

void TKinokoKdfStorage::OpenInputRawFile(void) throw(TKinokoException)
{
    if (_InputRawFile) {
	return;
    }

    try {
        _InputRawFile = new TMushChainedFile(_FileName);
	_InputRawFile->OpenReadMode();
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::OpenInputRawFile()",
	    "system call exception: " + e.Message()
        );
    }
}

int TKinokoKdfStorage::WritePreamble(void) throw(TKinokoException)
{
    if (_OutputRawFile == 0) {
        OpenOutputRawFile();
    }

    if (_IsRawStorage) {
	_DataAreaOffset = 0;
        return 0;
    }

    int PreambleSize = sizeof(PreambleBuffer);
    _PreambleVersion = PreambleVersion;
    _HeaderVersion = HeaderVersion;
    _HeaderAreaOffset = FreeAreaSize + PreambleSize;
    _DataAreaVersion = DataAreaVersion;
    _DataAreaOffset = _HeaderAreaOffset + _HeaderAreaSize;

    _DataAreaFormatFlags = 0;
    if (_IsDataCompressionEnabled) {
	_DataAreaFormatFlags |= FormatFlag_DataCompressionEnabled;
    }
#if 1  // for temporary backward compatibility ...
    else {
	_DataAreaVersion = FullVersion(1, 1);
    }
#endif

    if (_IsIndexEnabled) {
	if (_IsDataCompressionEnabled) {
	    throw TKinokoException(
		"TKinokoKdfStorage::WritePreamble()",
		"indexing for compressed files is not supported yet"
	    );
	}
	_DataAreaFormatFlags |= FormatFlag_IndexEnabled;
    }
    
    strcpy(FreeArea, FreeAreaMessage);

    // make the free area a fake data packet
    // so that raw-kdf tools can skip this area
    int FreeAreaPacketSize = _DataAreaOffset - 4;
    for (int Byte = 0; Byte < 4; Byte++){
        FreeArea[Byte] = (FreeAreaPacketSize >> (8 * Byte)) & 0x000000ff;
    }
    ((U32bit*) FreeArea)[1] = TKinokoDataSource::DataSourceId_Null;
    ((U32bit*) FreeArea)[2] = TKinokoDataStreamFormatter::PacketType_Null;

    PreambleBuffer[MagicNumberOffset] = MagicNumber;
    PreambleBuffer[PreambleVersionOffset] = _PreambleVersion;
    PreambleBuffer[HeaderVersionOffset] = _HeaderVersion;
    PreambleBuffer[HeaderAreaOffsetOffset] = _HeaderAreaOffset;
    PreambleBuffer[DataAreaVersionOffset] = _DataAreaVersion;
    PreambleBuffer[DataAreaOffsetOffset] = _DataAreaOffset;
    PreambleBuffer[DataAreaFormatFlagsOffset] = _DataAreaFormatFlags;
    PreambleBuffer[PreambleTrailerOffset] = 0;

    try {
	_OutputRawFile->SeekTo(0);
        _OutputRawFile->Write(FreeArea, FreeAreaSize);
	_OutputRawFile->SeekTo(FreeAreaSize);
	_OutputRawFile->Write(PreambleBuffer, sizeof(PreambleBuffer));
	if (_HeaderString != 0) {
	    _OutputRawFile->SeekTo(_HeaderAreaOffset);
	    _OutputRawFile->Write(_HeaderString, _HeaderAreaSize);
	}
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::WritePreamble()",
	    "system call exception: " + e.Message()
        );
    }

    return 0;
}

int TKinokoKdfStorage::ReadPreamble(void) throw(TKinokoException)
{
    if (_InputRawFile == 0) {
        OpenInputRawFile();
    }

    if (_IsRawStorage) {
	_DataAreaOffset = 0;
        return 0;
    }

    try {
	_InputRawFile->SeekTo(FreeAreaSize);
	_InputRawFile->Read(PreambleBuffer, sizeof(PreambleBuffer));    
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::ReadPreamble()",
	    "system call exception: " + e.Message()
        );
    }

    long DataFileMagic = PreambleBuffer[MagicNumberOffset];
    bool ReverseBytes = false;
    if (DataFileMagic != MagicNumber) {
        // byte order may be reversed //
        ReverseBytes = true;
        TMushBinaryDataEncoder DataEncoder(sizeof(U32bit), ReverseBytes);
	DataEncoder.DecodeOn(PreambleBuffer, sizeof(PreambleBuffer));

	DataFileMagic = PreambleBuffer[MagicNumberOffset];
    }

    if (DataFileMagic != MagicNumber) {
        throw TKinokoException(
	    "TKinokoKdfStorage::ReadPreamble()",
	    "inconsistent magic number"
	);
    }

    _PreambleVersion = PreambleBuffer[PreambleVersionOffset];
    if (MajorVersion(_PreambleVersion) > MajorVersion(PreambleVersion)) {
        throw TKinokoException(
	    "TKinokoKdfStorage::ReadPreamble()",
	    "version number mismatch (preamble): "
	    "use newer version of the Kinoko library"
	);
    }

    _HeaderVersion = PreambleBuffer[HeaderVersionOffset];
    _HeaderAreaOffset = PreambleBuffer[HeaderAreaOffsetOffset];
    _DataAreaVersion = PreambleBuffer[DataAreaVersionOffset];
    _DataAreaOffset = PreambleBuffer[DataAreaOffsetOffset];
    _HeaderAreaSize = _DataAreaOffset - _HeaderAreaOffset;

    if (_PreambleVersion >= FullVersion(1, 2)) {
	_DataAreaFormatFlags = PreambleBuffer[DataAreaFormatFlagsOffset];
    }
    else {
	_DataAreaFormatFlags = 0;
    }

    if (_DataAreaFormatFlags & FormatFlag_DataCompressionEnabled) {
	_IsDataCompressionEnabled = true;
    }
    if (_DataAreaFormatFlags & FormatFlag_IndexEnabled) {
	_IsIndexEnabled = true;
    }

    if (_HeaderAreaSize <= 0) {
        throw TKinokoException(
	    "TKinokoKdfStorage::ReadPreamble()",
	    "unable to find storage header"
        );
    }

    _HeaderString = new char[_HeaderAreaSize];
    try {
	_InputRawFile->SeekTo(_HeaderAreaOffset);
	_InputRawFile->Read(_HeaderString, _HeaderAreaSize);    
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::ReadPreamble()",
	    "system call exception: " + e.Message()
        );
    }

    return 0;
}

int TKinokoKdfStorage::WriteHeader(const TKinokoStorageHeader& Header) throw(TKinokoException)
{
    ostrstream BufferStream;
    BufferStream << "[Header]\n{\n";
    for (unsigned i = 0; i < Header.NumberOfEntries(); i++) {
	string Value = Header.ValueAt(i);
	for (unsigned j = 0; j < Value.size(); j++) {
	    if (Value[j] == '\"') {
		Value[j] = '\'';
	    }
	}

        BufferStream << "\t" << Header.EntryNameAt(i) << "=";
	BufferStream << "\"" << Value << "\";\n";
    }
    BufferStream << "}\n\n" << ends;

    size_t HeaderStringLength = strlen(BufferStream.str());
    _HeaderString = new char[HeaderStringLength];
    _HeaderAreaSize = (HeaderStringLength / 4 + 1) * 4;

    strcpy(_HeaderString, BufferStream.str());
    // BufferStream.freeze(0); //...

    return 0;
}

int TKinokoKdfStorage::ReadHeader(TKinokoStorageHeader& Header) throw(TKinokoException)
{
    if (_HeaderString == 0) {
        ReadPreamble();
    }

    if (MajorVersion(_HeaderVersion) > MajorVersion(HeaderVersion)) {
	throw TKinokoException(
	    "TKinokoKdfStorage::ReadHeader()",
	    "version number mismatch (storage header): "
	    "use newer version of the Kinoko library"
	);
    }

    if (_HeaderString[0] != '[') {
        // byte order is reversed //
        bool ReverseBytes = true;
        TMushBinaryDataEncoder DataEncoder(sizeof(U32bit), ReverseBytes);
	DataEncoder.DecodeOn(_HeaderString, _HeaderAreaSize);
    }

    istrstream BufferStream(_HeaderString);
    TParaCxxTokenTable TokenTable;
    TParaTokenizer Tokenizer(BufferStream, &TokenTable);
    try {
        Tokenizer.Next().MustBe("[");
        Tokenizer.Next();
        Tokenizer.Next().MustBe("]");

        Tokenizer.Next().MustBe("{");
	while (Tokenizer.LookAhead().IsNot("}")) {
	    string Name = Tokenizer.Next().AsString();
	    Tokenizer.Next().MustBe("=");
	    string Value = Tokenizer.Next().RemoveQuotation().AsString();
	    Tokenizer.Next().MustBe(";");

	    Header.AddEntry(Name, Value);
	}
    }
    catch (TScriptException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::ReadHeader()",
	    "badly formed data header: " + e.Message()
	);
    }

    return 0;
}

TKinokoOutputFileStream* TKinokoKdfStorage::GetOutputStream(void) throw(TKinokoException)
{
    if (_OutputStream != 0) {
	return _OutputStream;
    }

    if (_DataAreaOffset < 0) {
        WritePreamble();
    }

    try {
        if (! _IsRawStorage) {
	    _OutputRawFile->SeekTo(_DataAreaOffset);
	}

	if (_IsDataCompressionEnabled) {
	    TMushFile* CompressedFile = new TMushCompressedFile(_OutputRawFile);
	    _OutputFile = new TMushFramedFile(CompressedFile, _IsIndexEnabled);
	}
	else {
	    _OutputFile = new TMushFramedFile(_OutputRawFile, _IsIndexEnabled);
	}

	_OutputFile->OpenWriteMode(_OutputFileAccessMode);
	_OutputStream = new TKinokoOutputFileStream(_OutputFile);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::GetOutputStream()",
	    "system call exception: " + e.Message()
        );
    }
    catch (TKinokoException &e) {
	throw;
    }

    return _OutputStream;
}

TKinokoInputFileStream* TKinokoKdfStorage::GetInputStream(void) throw(TKinokoException)
{
    if (_InputStream != 0) {
	return _InputStream;
    }

    if (_DataAreaOffset < 0) {
        ReadPreamble();
    }

    if (! _IsRawStorage) {
        if (MajorVersion(_DataAreaVersion) > MajorVersion(DataAreaVersion)) {
	    throw TKinokoException(
	        "TKinokoKdfStorage::GetInputStream()",
	        "version number mismatch (data area): "
		"use newer version of the Kinoko library"
	    );
	}
    }

    try {
        if (! _IsRawStorage) {
	    _InputRawFile->SeekTo(_DataAreaOffset);
	}

	if (_IsDataCompressionEnabled) {
	    TMushFile* CompressedFile = new TMushCompressedFile(_InputRawFile);
	    _InputFile = new TMushFramedFile(CompressedFile);
	}
	else {
	    _InputFile = new TMushFramedFile(_InputRawFile, _IsIndexEnabled);
	}

	_InputFile->OpenReadMode();
	_InputStream = new TKinokoInputFileStream(_InputFile);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoKdfStorage::GetInputStream()",
	    "system call exception: " + e.Message()
        );
    }
    catch (TKinokoException &e) {
	throw;
    }

    return _InputStream;
}

long TKinokoKdfStorage::DataFilePreambleVersion(void) const
{
    return _PreambleVersion;
}

long TKinokoKdfStorage::DataFileHeaderVersion(void) const
{
    return _HeaderVersion;
}

long TKinokoKdfStorage::DataFileDataAreaVersion(void) const
{
    return _DataAreaVersion;
}

long TKinokoKdfStorage::DataFileFormatFlags(void) const
{
    return _DataAreaFormatFlags;
}

