/* KinokoFileStream.cc */
/* Created by Enomoto Sanshiro on 24 February 1998. */
/* Last updated by Enomoto Sanshiro on 21 February 2002. */


#include "MushFramedFile.hh"
#include "KinokoStream.hh"
#include "KinokoFileStream.hh"

using namespace std;


static const int MaxNumberOfEntries = 2048;


TKinokoInputFileStream::TKinokoInputFileStream(TMushFramedFile *File, int LocalBufferSize, int LocalBufferEntryTableSize)
: TKinokoInputStream(LocalBufferSize, LocalBufferEntryTableSize)
{
    _File = File;

    if (! _File->IsOpened()) {
	try {
	    _File->OpenReadMode();
	}
	catch (TSystemCallException &e) {
            throw TKinokoException(
	        "TKinokoInputFileStream::TKinokoInputFileStream()",
	        "system call exception: " + e.Message()
            );
	}
    }
}

TKinokoInputFileStream::~TKinokoInputFileStream() 
{
}

void TKinokoInputFileStream::Destroy(void)
{
    _File->Close();

    delete _File;
    _File = 0;
}

bool TKinokoInputFileStream::HasData(void)
{
    return true;
}

size_t TKinokoInputFileStream::NextDataSize(void)
{
    size_t Result = _File->NextPacketSize();
    if (Result < 0) {
	Result = 0;
    }

    return Result;
}

int TKinokoInputFileStream::Read(void *Address, size_t MaxSize) throw(TKinokoException)
{
    int ReadLength;

    try {
	ReadLength = _File->Read(Address, MaxSize);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoInputFileStream::Read()",
	    "system call exception: " + e.Message()
        );
    }
    if (ReadLength < 0) {
	// interrupted system call
	ReadLength = 0;
    }

    return ReadLength;
}

int TKinokoInputFileStream::NextEntry(void *&Address) throw(TKinokoException)
{
    size_t NextPacketSize = (size_t) _File->NextPacketSize();

    if (NextPacketSize <= 0) {
	Address = 0;
	return 0;
    }

    if (NextPacketSize > _WorkAreaSize) {
	ResizeWorkArea(NextPacketSize);
    }

    int ReadLength;
    try {
	ReadLength = _File->Read(_WorkArea, NextPacketSize);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TInputFileStream::NextEntry()",
	    "system call exception: " + e.Message()
	);
    }
    if (ReadLength <= 0) {
	// interrupted system call
	ReadLength = 0;
	Address = 0;
    }
    else {
	Address = _WorkArea;
    }

    return ReadLength;
}

void TKinokoInputFileStream::Flush(void* Address) throw(TKinokoException)
{
}

int TKinokoInputFileStream::GetEntry(void *&Address) throw(TKinokoException)
{
    if (_LocalBuffer == 0) {
	CreateLocalBuffer();
    }

    int PacketSize = _File->NextPacketSize();
    if (PacketSize <= 0) {
	Address = 0;
	return 0;
    }

    if (! _LocalBuffer->Allocate(Address, PacketSize)) {
	_IsLocalBufferFull = true;
	Address = 0;
	return 0;
    }

    int ReadLength;
    try {
	ReadLength = _File->Read(Address, PacketSize);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoInputFileStream::GetEntry()",
	    "system call exception: " + e.Message()
	);
    }
    if (ReadLength < 0) {
	// interrupted system call
	ReadLength = 0;
    }

    return ReadLength;
}

void TKinokoInputFileStream::ReleaseEntry(void* Address) throw(TKinokoException)
{
    if (Address != 0) {
	_LocalBuffer->Release(Address);
	_IsLocalBufferFull = false;
    }
}

long TKinokoInputFileStream::NumberOfPackets(void) throw(TKinokoException)
{
    try {
	return _File->Size();
    }
    catch (TSystemCallException &e) {
	throw TKinokoException(
	    "TKinokoInputFileStream::NumberOfPackets()", e.Message()
	);
    }
}

long TKinokoInputFileStream::PacketCount(void) throw(TKinokoException)
{
    try {
	return _File->Position();
    }
    catch (TSystemCallException &e) {
	throw TKinokoException(
	    "TKinokoInputFileStream::NextPacketIndex()", e.Message()
	);
    }
}

void TKinokoInputFileStream::JumpTo(long PacketIndex) throw(TKinokoException)
{
    try {
	_File->SeekTo(PacketIndex);
    }
    catch (TSystemCallException &e) {
	throw TKinokoException(
	    "TKinokoInputFileStream::JumpTo()", e.Message()
	);
    }
}


TKinokoOutputFileStream::TKinokoOutputFileStream(TMushFramedFile *File, int LocalBufferSize, int LocalBufferEntryTableSize)
: TKinokoOutputStream(LocalBufferSize, LocalBufferEntryTableSize)
{
    _File = File;

    if (! _File->IsOpened()) {
	try {
	    _File->OpenWriteMode();
	}
	catch (TSystemCallException &e) {
	    throw TKinokoException(
		"TKinokoOutputFileStream::TKinokoOutputFileStream()",
		"system call exception: " + e.Message()
	    );
	}
    }
}

TKinokoOutputFileStream::~TKinokoOutputFileStream() 
{
}

void TKinokoOutputFileStream::Destroy(void)
{
    _File->Close();

    delete _File;
    _File = 0;
}

int TKinokoOutputFileStream::Write(void *Address, size_t Size) throw(TKinokoException)
{
    int WrittenLength;

    try {
	WrittenLength = _File->Write(Address, Size);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoOutputFileStream::Write()",
	    "system call exception: " + e.Message()
        );
    }

    return WrittenLength;
}

int TKinokoOutputFileStream::NextEntry(void *&Address, size_t MaxSize) throw(TKinokoException)
{
    if (MaxSize > _WorkAreaSize) {
	ResizeWorkArea(MaxSize);
    }
    Address = _WorkArea;

    return MaxSize;
}

int TKinokoOutputFileStream::Flush(void* Address, size_t Size) throw(TKinokoException)
{
    int WrittenLength;

    try {
        WrittenLength = _File->Write(_WorkArea, Size);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TOutputFileStream::Flush()",
	    "system call exception: " + e.Message()
        );
    }

    return WrittenLength;
}

int TKinokoOutputFileStream::GetEntry(void *&Address, size_t MaxSize) throw(TKinokoException)
{
    if (_LocalBuffer == 0) {
	CreateLocalBuffer();
    }

    if (! _LocalBuffer->Allocate(Address, MaxSize)) {
	_IsLocalBufferFull = true;
	Address = 0;
	return 0;
    }

    return MaxSize;
}

int TKinokoOutputFileStream::ReleaseEntry(void* Address, size_t Size) throw(TKinokoException)
{
    if (Address == 0) {
	return 0;
    }

    int WrittenLength;

    try {
        WrittenLength = _File->Write(Address, Size);
    }
    catch (TSystemCallException &e) {
        throw TKinokoException(
	    "TKinokoOutputFileStream::Flush()",
	    "system call exception: " + e.Message()
        );
    } 

    _LocalBuffer->Release(Address);
    _IsLocalBufferFull = false;

    return WrittenLength;
}
