/* Kinoko2dHistogramView.cc */
/* Created by Enomoto Sanshiro on 13 January 2001. */
/* Last updated by Enomoto Sanshiro on 13 January 2001. */


#include <strstream>
#include <cmath>
#include "Kasp2dHistogram.hh"
#include "Kinoko2dHistogramView.hh"

using namespace std;


TKinoko2dHistogramView::TKinoko2dHistogramView(TKasp2dHistogram* Histogram)
{
    _Histogram = Histogram;

    _PlotType = PlotType_Color;
    _IsZScaleEnabled = true;
    _IsZScaleLog = false;

    _ZMin = 0;
    _ZMax = 1.0;
}

TKinoko2dHistogramView::~TKinoko2dHistogramView()
{
}

void TKinoko2dHistogramView::SetAxisTitle(const std::string& XTitle, const std::string& YTitle)
{
    _XTitle = XTitle;
    _YTitle = YTitle;
}

void TKinoko2dHistogramView::DisableZScale(void)
{
    _IsZScaleEnabled = false;
}

void TKinoko2dHistogramView::SetZScaleLog(void)
{
    _IsZScaleLog = true;
    _ZMin = 0.5;
}

void TKinoko2dHistogramView::SetZScaleLinear(void)
{
    _IsZScaleLog = false;
    _ZMin = 0;
}

bool TKinoko2dHistogramView::SetDrawOption(const string& OptionString)
{
    bool Result = true;

    if (OptionString == "scatter") {
	_PlotType = PlotType_Scatter;
    }
    else if (OptionString == "box") {
	_PlotType = PlotType_Box;
    }
    else if (OptionString == "color") {
	_PlotType = PlotType_Color;
    }
    else if (OptionString == "gray") {
	_PlotType = PlotType_Gray;
    }
    else if (OptionString == "contour") {
	_PlotType = PlotType_Contour;
    }
    else {
	Result = TKinokoView::SetDrawOption(OptionString);
    }

    return Result;
}

void TKinoko2dHistogramView::SaveThis(TKaspRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->Save2dHistogram(_Name, _Histogram);
    }
    catch (TKaspException &e) {
	throw TKinokoException(
	    "TKinoko2dHistogramView::SaveThis()", e.Message()
	);
    }
}


void TKinoko2dHistogramView::ClearThis(void)
{
    _Histogram->Clear();
}



TKinokoKoap2dHistogramView::TKinokoKoap2dHistogramView(TKasp2dHistogram* Histogram, ostream* OutputStream)
: TKinoko2dHistogramView(Histogram)
{
    _OutputStream = OutputStream;
}

TKinokoKoap2dHistogramView::~TKinokoKoap2dHistogramView()
{
}

void TKinokoKoap2dHistogramView::DeployThis(void) 
{
    if (_IsZScaleEnabled) {
	switch (_PlotType) {
          case PlotType_Scatter:
	    (*_OutputStream) << ".create plot3dmonochrome " << _Name << " ";
	    break;
	  case PlotType_Box:
	    (*_OutputStream) << ".create plot3dmonochrome " << _Name << " ";
	    break;
	  case PlotType_Color:
	    (*_OutputStream) << ".create plot3d " << _Name << " ";
	    break;
	  case PlotType_Gray:
	    (*_OutputStream) << ".create plot3dmonochrome " << _Name << " ";
	    break;
	  case PlotType_Contour:
	    (*_OutputStream) << ".create plot3dmonochrome " << _Name << " ";
	    break;
	}
    }
    else {
	(*_OutputStream) << ".create plot3dnoscale " << _Name << " ";
    }

    (*_OutputStream) << 100 * Left() << " " << 100 * Top() << " ";
    (*_OutputStream) << 100 * Width() << " " << 100 * Height() << ";" << endl;

    (*_OutputStream) << _Name << " set title " << _Histogram->Title() << ";" << endl;
    if (! _XTitle.empty()) {
	(*_OutputStream) << _Name << " set xtitle " << _XTitle << ";" << endl;
    }
    if (! _YTitle.empty()) {
	(*_OutputStream) << _Name << " set ytitle " << _YTitle << ";" << endl;
    }

    _XMin = _Histogram->XScale().Min();
    _XMax = _Histogram->XScale().Max();
    _NumberOfXBins = _Histogram->XScale().NumberOfBins();
    _XBinStep = (_XMax - _XMin) / _NumberOfXBins;

    _YMin = _Histogram->YScale().Min();
    _YMax = _Histogram->YScale().Max();
    _NumberOfYBins = _Histogram->YScale().NumberOfBins();
    _YBinStep = (_YMax - _YMin) / _NumberOfYBins;

    if (_IsZScaleLog) {
	(*_OutputStream) << _Name << " set zscale log;" << endl;
    }

    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << "0 1;" << endl;
}

void TKinokoKoap2dHistogramView::DrawThis(void)
{
    int Order = (int) log(_Histogram->PeakCounts() + 1);
    if (! _IsZScaleLog) {
	_ZMax = 1.05 * exp((float) (Order + 1));
    }
    else {
	_ZMax = 2.0 * exp((float) (Order + 1));
    }

    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;

    if (! _Histogram->HasData()) {
        (*_OutputStream) << _Name << " drawtext ";
        (*_OutputStream) << (7 * _XMin + _XMax) / 8 << " ";
        (*_OutputStream) << (_YMin + _YMax) / 2 << " ";
        (*_OutputStream) << "no data;" << endl;
	return;
    }

    switch (_PlotType) {
      case PlotType_Scatter:
	(*_OutputStream) << _Name << " scatterhist ";
	break;
      case PlotType_Box:
	(*_OutputStream) << _Name << " boxhist ";
	break;
      case PlotType_Color:
	(*_OutputStream) << _Name << " colorhist ";
	break;
      case PlotType_Gray:
	(*_OutputStream) << _Name << " colorhist ";
	break;
      case PlotType_Contour:
	(*_OutputStream) << _Name << " contour ";
	break;
      default:
	(*_OutputStream) << _Name << " colorhist ";
	break;
    }
    (*_OutputStream) << _XMin << " " << _XBinStep << " ";
    (*_OutputStream) <<_NumberOfXBins << " ";
    (*_OutputStream) << _YMin << " " << _YBinStep << " ";
    for (int YBin = 0; YBin < _NumberOfYBins; YBin++) {
	for (int XBin = 0; XBin < _NumberOfXBins; XBin++) {
	    (*_OutputStream) << " " << _Histogram->CountsOf(XBin, YBin);
	}
    }
    (*_OutputStream) << ";" << endl;

    (*_OutputStream) << _Name << " drawframe;" << endl;
}

void TKinokoKoap2dHistogramView::ClearThis(void)
{
    TKinoko2dHistogramView::ClearThis();
    
    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << "0 1;" << endl;
}
