/* KinokoHistogramView.cc */
/* Created by Enomoto Sanshiro on 4 November 2000. */
/* Last updated by Enomoto Sanshiro on 4 November 2000. */


#include <strstream>
#include <cmath>
#include "KaspHistogram.hh"
#include "KinokoHistogramView.hh"

using namespace std;


TKinokoHistogramView::TKinokoHistogramView(TKaspHistogram* Histogram)
{
    _Histogram = Histogram;
    _IsYScaleLog = false;

    _YMin = 0;
    _YMax = 1.0;
}

TKinokoHistogramView::~TKinokoHistogramView()
{
}

void TKinokoHistogramView::SetYScaleLog(void)
{
    _IsYScaleLog = true;
    _YMin = 0.5;
}

void TKinokoHistogramView::SetYScaleLinear(void)
{
    _IsYScaleLog = false;
    _YMin = 0;
}

void TKinokoHistogramView::SetAxisTitle(const std::string& XTitle, const std::string& YTitle)
{
    _XTitle = XTitle;
    _YTitle = YTitle;
}

void TKinokoHistogramView::SaveThis(TKaspRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->SaveHistogram(_Name, _Histogram);
    }
    catch (TKaspException &e) {
	throw TKinokoException(
	    "TKinokoHistogramView::SaveThis()", e.Message()
	);
    }
}

void TKinokoHistogramView::ClearThis(void)
{
    _Histogram->Clear();
}



TKinokoKoapHistogramView::TKinokoKoapHistogramView(TKaspHistogram* Histogram, ostream* OutputStream)
: TKinokoHistogramView(Histogram)
{
    _OutputStream = OutputStream;
}

TKinokoKoapHistogramView::~TKinokoKoapHistogramView()
{
}

void TKinokoKoapHistogramView::DeployThis(void) 
{
    (*_OutputStream) << ".create plot " << _Name << " ";
    (*_OutputStream) << 100 * Left() << " " << 100 * Top() << " ";
    (*_OutputStream) << 100 * Width() << " " << 100 * Height() << ";" << endl;

    (*_OutputStream) << _Name << " set title " << _Histogram->Title() << ";" << endl;
    if (! _XTitle.empty()) {
	(*_OutputStream) << _Name << " set xtitle " << _XTitle << ";" << endl;
    }
    if (! _YTitle.empty()) {
	(*_OutputStream) << _Name << " set ytitle " << _YTitle << ";" << endl;
    }
    
    _Min = _Histogram->Scale().Min();
    _Max = _Histogram->Scale().Max();
    _NumberOfBins = _Histogram->Scale().NumberOfBins();
    _BinStep = (_Max - _Min) / _NumberOfBins;

    if (_IsYScaleLog) {
	(*_OutputStream) << _Name << " set yscale log;" << endl;
    }

    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _Min << " " << _Max << " ";
    (*_OutputStream) << _YMin << " " << _YMax << ";" << endl;
}

void TKinokoKoapHistogramView::DrawThis(void)
{
    int Order = (int) log(_Histogram->PeakCounts() + 1);
    if (! _IsYScaleLog) {
	_YMax = 1.1 * exp((float) (Order + 1));
    }
    else {
	_YMax = 2.0 * exp((float) (Order + 1));
    }

    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _Min << " " << _Max << " ";
    (*_OutputStream) << _YMin << " " << _YMax << ";" << endl;

    if (! _Histogram->HasData()) {
        (*_OutputStream) << _Name << " drawtext ";
        (*_OutputStream) << (7 * _Min + _Max) / 8 << " ";
        (*_OutputStream) << (_YMin + _YMax) / 2 << " ";
        (*_OutputStream) << "no data;" << endl;
	return;
    }

    (*_OutputStream) << _Name << " hist " << _Min << " " << _BinStep;
    for (int BinIndex = 0; BinIndex < _NumberOfBins; BinIndex++) {
	(*_OutputStream) << " " << _Histogram->CountsOf(BinIndex);
    }
    (*_OutputStream) << ";" << endl;

    (*_OutputStream) << _Name << " comment -100 0 ";
    (*_OutputStream) << "entries: " << _Histogram->NumberOfEntries() << endl;
    (*_OutputStream) << "underflow: " << _Histogram->UnderflowCounts() << endl;
    (*_OutputStream) << "overflow: " << _Histogram->OverflowCounts() << endl;
    (*_OutputStream) << "mean: " << _Histogram->Mean() << endl;
    (*_OutputStream) << ";" << endl;

    (*_OutputStream) << _Name << " drawframe;" << endl;
}

void TKinokoKoapHistogramView::ClearThis(void)
{
    TKinokoHistogramView::ClearThis();
    
    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _Min << " " << _Max << " ";
    (*_OutputStream) << _YMin << " " << _YMax << ";" << endl;
}
