/* KinokoMapView.cc */
/* Created by Enomoto Sanshiro on 2 April 2001. */
/* Last updated by Enomoto Sanshiro on 8 April 2001. */


#include <strstream>
#include <cmath>
#include <map>
#include "KaspMap.hh"
#include "KinokoMapView.hh"

using namespace std;


TKinokoMapView::TKinokoMapView(TKaspMap* Map, float XMin, float XMax, float YMin, float YMax, float ZMin, float ZMax)
{
    _Map = Map;

    _LastAddress = -1;

    _XMin = XMin; _XMax = XMax;
    _YMin = YMin; _YMax = YMax;
    _ZMin = ZMin; _ZMax = ZMax;

    _PointRadius = 1.0;
}

TKinokoMapView::~TKinokoMapView()
{
}

void TKinokoMapView::AddPoint(int Address, float PositionX, float PositionY)
{
    _PositionTable[Address] = make_pair(PositionX, PositionY);
}

void TKinokoMapView::SetPointSize(float Radius)
{
    _PointRadius = Radius;
}

bool TKinokoMapView::IsValidAddress(int Address)
{
    if (Address != _LastAddress) {
	_LastAddress = Address;
	_LastPosition = _PositionTable.find(Address);
    }

    return _LastPosition != _PositionTable.end();
}

float TKinokoMapView::PositionXOf(int Address)
{
    if (Address != _LastAddress) {
	_LastAddress = Address;
	_LastPosition = _PositionTable.find(Address);
    }

    if (_LastPosition == _PositionTable.end()) {
	return 0;
    }

    return (*_LastPosition).second.first;
}

float TKinokoMapView::PositionYOf(int Address)
{
    if (Address != _LastAddress) {
	_LastAddress = Address;
	_LastPosition = _PositionTable.find(Address);
    }

    if (_LastPosition == _PositionTable.end()) {
	return 0;
    }

    return (*_LastPosition).second.second;
}

float TKinokoMapView::PointRadius(void)
{
    return _PointRadius;
}

void TKinokoMapView::SaveThis(TKaspRepository* Repository)throw(TKinokoException)
{
    try {
	Repository->SaveMap(_Name, _Map);
    }
    catch (TKaspException &e) {
	throw TKinokoException(
	    "TKinokoMapView::SaveThis()", e.Message()
	);
    }
}

void TKinokoMapView::ClearThis(void)
{
    _Map->Clear();
}


TKinokoKoapMapView::TKinokoKoapMapView(TKaspMap* Map, float XMin, float XMax, float YMin, float YMax, float ZMin, float ZMax, ostream* OutputStream)
: TKinokoMapView(Map, XMin, XMax, YMin, YMax, ZMin, ZMax)
{
    _OutputStream = OutputStream;
}

TKinokoKoapMapView::~TKinokoKoapMapView()
{
}

void TKinokoKoapMapView::DeployThis(void) 
{
    (*_OutputStream) << ".create mapplot " << _Name << " ";
    (*_OutputStream) << 100 * Left() << " " << 100 * Top() << " ";
    (*_OutputStream) << 100 * Width() << " " << 100 * Height() << ";" << endl;

    (*_OutputStream) << _Name << " set title " << _Map->Title() << ";" << endl;

    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;
}

void TKinokoKoapMapView::DrawThis(void)
{
    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;

    if (! _Map->HasData()) {
	return;
    }

    (*_OutputStream) << _Name << " plot " << _PointRadius;
    int Address;
    int NumberOfPoints = _Map->NumberOfPoints();
    for (int Index = 0; Index < NumberOfPoints; Index++) {
	Address = _Map->AddressOf(Index);
	if (! IsValidAddress(Address)) {
	    continue;
	}

	(*_OutputStream) << " " << PositionXOf(Address);
	(*_OutputStream) << " " << PositionYOf(Address);
	(*_OutputStream) << " " << _Map->ValueOf(Index);
    }
    (*_OutputStream) << ";" << endl;
}

void TKinokoKoapMapView::ClearThis(void)
{
    TKinokoMapView::ClearThis();
    
    (*_OutputStream) << _Name << " clear;" << endl;
    (*_OutputStream) << _Name << " frame ";
    (*_OutputStream) << _XMin << " " << _XMax << " ";
    (*_OutputStream) << _YMin << " " << _YMax << " ";
    (*_OutputStream) << _ZMin << " " << _ZMax << ";" << endl;
}
