/* KinokoViewMessenger.cc */
/* Created by Enomoto Sanshiro on 31 December 2000. */
/* Last updated by Enomoto Sanshiro on 29 September 2002. */


#include <string>
#include "ParaPackage.hh"
#include "KaspRepository.hh"
#include "KaspRepositoryFactory.hh"
#include "KinokoAnalysisMessenger.hh"
#include "KinokoAnalysisSequenceBuilder.hh"
#include "KinokoHistogramWriter.hh"
#include "Kinoko2dHistogramWriter.hh"
#include "KinokoHistoryWriter.hh"
#include "KinokoWaveWriter.hh"
#include "KinokoMapWriter.hh"
#include "KinokoTabularWriter.hh"
#include "KinokoView.hh"
#include "KinokoViewBuilder.hh"
#include "KinokoViewFrame.hh"
#include "KinokoViewSequence.hh"
#include "KinokoViewFactory.hh"
#include "KinokoViewMessenger.hh"


using namespace std;


TKinokoViewSystemMessenger::TKinokoViewSystemMessenger(TKinokoViewBuilder* ViewBuilder, const string& ScriptFileName)
: TParaObjectPrototype("ViewSystem")
{
    _ViewBuilder = ViewBuilder;
    _ScriptFileName = ScriptFileName;
}

TKinokoViewSystemMessenger::~TKinokoViewSystemMessenger()
{
}

TParaObjectPrototype* TKinokoViewSystemMessenger::Clone(void)
{
    return new TKinokoViewSystemMessenger(_ViewBuilder, _ScriptFileName);
}

int TKinokoViewSystemMessenger::DispatchMessage(const std::string& Message, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "scriptFileName") || (Message == "ScriptFileName")) {
	return ScriptFileName(ArgumentList, ReturnValue);
    }
    else if ((Message == "getRegistry") || (Message == "GetRegistry")) {
	return GetRegistry(ArgumentList, ReturnValue);
    }
    else if ((Message == "setRegistry") || (Message == "SetRegistry")) {
	return SetRegistry(ArgumentList, ReturnValue);
    }

    return 0;
}

int TKinokoViewSystemMessenger::ScriptFileName(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    ReturnValue = TParaValue(_ScriptFileName);
    return 1;
}

int TKinokoViewSystemMessenger::GetRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "too few arguments: getRegistry(string registry_path)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument:  getRegistry(string registry_path)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    TKinokoRegistry* Registry = _ViewBuilder->Registry();
    ReturnValue = TParaValue(Registry->GetValue(RegistryPath));

    return 1;
}

int TKinokoViewSystemMessenger::SetRegistry(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "too few argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }
    if (! ArgumentList[0]->IsString()) {
	throw TScriptException(
	    "invalid argument[s]: "
	    "setRegistry(string registry_path, string value)"
	);
    }

    string RegistryPath = ArgumentList[0]->AsString();
    string Value = ArgumentList[1]->AsString();

    TKinokoRegistry* Registry = _ViewBuilder->Registry();
    Registry->SetValue(RegistryPath, Value);
    ReturnValue = TParaValue(Value);

    return 1;
}



TKinokoViewRepositoryMessenger::TKinokoViewRepositoryMessenger(TKinokoViewBuilder* ViewBuilder)
: TParaObjectPrototype("ViewRepository")
{
    _ViewBuilder = ViewBuilder;
    _Repository = 0;
}

TKinokoViewRepositoryMessenger::~TKinokoViewRepositoryMessenger()
{
}

TParaObjectPrototype* TKinokoViewRepositoryMessenger::Clone(void)
{
    return new TKinokoViewRepositoryMessenger(_ViewBuilder);
}

int TKinokoViewRepositoryMessenger::DispatchMessage(const std::string& Message, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    return 0;
}

void TKinokoViewRepositoryMessenger::Construct(const std::string& ClassName, std::vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "ViewRepository::ViewRepository(string file_name)",
	    "too few argument[s]"
	);
    }
    string FileName = ArgumentList[0]->AsString();
    _Repository = 0;

    TKaspRepositoryFactory* Factory = 0;
    if (ClassName == "PlainTextViewRepository") {
	Factory = TKaspRepositoryFactory::GetInstance("PlainText");
    }
    else if (ClassName == "XmlViewRepository") {
	Factory = TKaspRepositoryFactory::GetInstance("Xml");
    }
    else if (ClassName == "RootViewRepository") {
	Factory = TKaspRepositoryFactory::GetInstance("Root");
    }
    else {
	Factory = TKaspRepositoryFactory::GetInstance("PlainText");
    }

    if (Factory != 0) {
	_Repository = Factory->CreateRepository(FileName);
    }
    
    if (_Repository == 0) {
	throw TScriptException(
	    "ViewRepository::ViewRepository(string file_name)",
	    "repository not available: " + ClassName
	);
    }

    _ViewBuilder->AddViewRepository(_Repository);
}

TKaspRepository* TKinokoViewRepositoryMessenger::Repository(void)
{
    return _Repository;
}



TKinokoViewMessenger::TKinokoViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TParaObjectPrototype("View")
{
    _ViewBuilder = ViewBuilder;
    _AnalysisSequenceBuilder = AnalysisSequenceBuilder;

    _ViewFactory = _ViewBuilder->ViewFactory();
}

TKinokoViewMessenger::~TKinokoViewMessenger()
{
}

int TKinokoViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "fill") || (Message == "Fill")) {
	return Fill(ArgumentList, ReturnValue);
    }
    else if ((Message == "fillOne") || (Message == "FillOne")) {
	bool IsSingleEventAction;
	return Fill(ArgumentList, ReturnValue, IsSingleEventAction = true);
    }
    else if ((Message == "draw") || (Message == "Draw")) {
	return Draw(ArgumentList, ReturnValue);
    }
    else if ((Message == "clear") || (Message == "Clear")) {
	return Clear(ArgumentList, ReturnValue);
    }
    else if ((Message == "save") || (Message == "Save")) {
	return Save(ArgumentList, ReturnValue);
    }
    else if ((Message == "putLine") || (Message == "PutLine")) {
	return PutLine(ArgumentList, ReturnValue);
    }
    else if ((Message == "putRectangle") || (Message == "PutRectangle")) {
	return PutRectangle(ArgumentList, ReturnValue);
    }
    else if ((Message == "putCircle") || (Message == "PutCircle")) {
	return PutCircle(ArgumentList, ReturnValue);
    }
    else if ((Message == "putText") || (Message == "PutText")) {
	return PutText(ArgumentList, ReturnValue);
    }
    else if ((Message == "putImage") || (Message == "PutImage")) {
	return PutImage(ArgumentList, ReturnValue);
    }
    else if ((Message == "setColor") || (Message == "SetColor")) {
	return SetColor(ArgumentList, ReturnValue);
    }
    else if ((Message == "setTextAdjustment") || (Message == "SetTextAdjustment")) {
	return SetTextAdjustment(ArgumentList, ReturnValue);
    }
    else if ((Message == "setFont") || (Message == "SetFont")) {
	return SetFont(ArgumentList, ReturnValue);
    }

    //... for backward compatibility ...//
    else if ((Message == "take") || (Message == "Take")) {
	return Fill(ArgumentList, ReturnValue);
    }
    else if ((Message == "takeOne") || (Message == "TakeOne")) {
	bool IsSingleEventAction;
	return Fill(ArgumentList, ReturnValue, IsSingleEventAction = true);
    }

    return 0;
}

int TKinokoViewMessenger::Fill(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue, bool IsSingleEventAction) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "View::fill(DataElement data_element): too few argument[s]"
	);
    }

    vector<TKinokoDataElementMessenger*> DataElementMessengerList;
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	if (! ArgumentList[i]->IsObject("DataElement")) {
	    throw TScriptException(
		"View::fill(DataElement data_element): invalid argument[s]"
	    );
	}

	DataElementMessengerList.push_back(
	    (TKinokoDataElementMessenger*) ArgumentList[i]->AsObject()
	);
    }

    TKinokoDataAnalyzer* Analyzer;
    Analyzer = CreateAnalysisAction(DataElementMessengerList);
    if (Analyzer == 0) {
	throw TScriptException(
	    "View::fill(DataElement data_element): invalid data element"
	);
    }

    try {
	if (! IsSingleEventAction) {
	    _AnalysisSequenceBuilder->AddAction(Analyzer);
	}
	else {
	    _AnalysisSequenceBuilder->AddSingleEventAction(Analyzer);
	}
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoViewMessenger::Draw(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoViewDrawAction(View()));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    if (ArgumentList.size() > 0) {
	string OptionString = ArgumentList[0]->AsString();
	if (! View()->SetDrawOption(OptionString)) {
	    throw TScriptException(
		"View::draw(string option)", "invalid option: " + OptionString
	    );
	}
    }

    return 1;
}

int TKinokoViewMessenger::Clear(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoViewClearAction(View()));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoViewMessenger::Save(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 1) ||
	(! ArgumentList[0]->IsObject("ViewRepository"))
    ){
	throw TScriptException(
	    "View::Save(ViewRepository repository)", "invalid argument[s]"
	);
    }

    TKaspRepository* Repository = (
	((TKinokoViewRepositoryMessenger*) ArgumentList[0]->AsObject())->Repository()
    );

    try {
	_ViewBuilder->AddAction(new TKinokoViewSaveAction(View(), Repository));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoViewMessenger::PutLine(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "View::putLine(double x0, double y0, double x1, double y1)",
	    "too few argument[s]"
	);
    }

    double x0 = ArgumentList[0]->AsDouble();
    double y0 = ArgumentList[1]->AsDouble();
    double x1 = ArgumentList[2]->AsDouble();
    double y1 = ArgumentList[3]->AsDouble();

    View()->AddViewTrimming(
	_ViewFactory->CreateLineTrimming(View(), x0, y0, x1, y1)
    );

    return 1;
}

int TKinokoViewMessenger::PutRectangle(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "View::putRectangle(double x0, double y0, double x1, double y1)",
	    "too few argument[s]"
	);
    }

    double x0 = ArgumentList[0]->AsDouble();
    double y0 = ArgumentList[1]->AsDouble();
    double x1 = ArgumentList[2]->AsDouble();
    double y1 = ArgumentList[3]->AsDouble();

    View()->AddViewTrimming(
	_ViewFactory->CreateRectangleTrimming(View(), x0, y0, x1, y1)
    );

    return 1;
}

int TKinokoViewMessenger::PutCircle(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "View::putCircle(double x, double y, double r)",
	    "too few argument[s]"
	);
    }

    double x = ArgumentList[0]->AsDouble();
    double y = ArgumentList[1]->AsDouble();
    double r = ArgumentList[2]->AsDouble();

    View()->AddViewTrimming(
	_ViewFactory->CreateCircleTrimming(View(), x, y, r)
    );

    return 1;
}

int TKinokoViewMessenger::PutText(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "View::putText(double x, double y, string text)",
	    "too few argument[s]"
	);
    }

    double x = ArgumentList[0]->AsDouble();
    double y = ArgumentList[1]->AsDouble();
    string Text = ArgumentList[2]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateTextTrimming(View(), x, y, Text)
    );

    return 1;
}

int TKinokoViewMessenger::PutImage(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "View::putImage(double x, double y, string file_name)",
	    "too few argument[s]"
	);
    }

    double x = ArgumentList[0]->AsDouble();
    double y = ArgumentList[1]->AsDouble();
    string FileName = ArgumentList[2]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateImageTrimming(View(), x, y, FileName)
    );

    return 1;
}

int TKinokoViewMessenger::SetColor(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "View::setColor(string color_name)", "too few argument[s]"
	);
    }

    string ColorName = ArgumentList[0]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateColorTrimming(View(), ColorName)
    );

    return 1;
}

int TKinokoViewMessenger::SetTextAdjustment(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException(
	    "View::setTextAdjustment(string text_adjustment)", 
	    "too few argument[s]"
	);
    }

    string TextAdjustmentString = ArgumentList[0]->AsString();

    View()->AddViewTrimming(
	_ViewFactory->CreateTextAdjustmentTrimming(View(), TextAdjustmentString)
    );

    return 1;
}

int TKinokoViewMessenger::SetFont(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException(
	    "View::setFont(string font_name, int font_size)", 
	    "too few argument[s]"
	);
    }

    string FontName = ArgumentList[0]->AsString();
    int FontSize = ArgumentList[1]->AsLong();

    View()->AddViewTrimming(
	_ViewFactory->CreateFontTrimming(View(), FontName, FontSize)
    );

    return 1;
}



TKinokoViewGridMessenger::TKinokoViewGridMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
}

TKinokoViewGridMessenger::~TKinokoViewGridMessenger()
{
}

TParaObjectPrototype* TKinokoViewGridMessenger::Clone(void)
{
    return new TKinokoViewGridMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoViewGridMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    int NumberOfColumns = 0;
    if (ArgumentList.size() >= 1) {
	if (! ArgumentList[0]->IsLong()) {
	    throw TScriptException(
		"Grid::Grid(int number_of_columns)", "invalid argument[s]"
	    );
	}
	NumberOfColumns = ArgumentList[0]->AsLong();
    }

    _Grid = new TKinokoViewGridFrame(NumberOfColumns);
    _ViewBuilder->AddView(_Grid);
}

int TKinokoViewGridMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (Message == "put") {
	return PutView(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

int TKinokoViewGridMessenger::PutView(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 1) ||
	! ArgumentList[0]->IsObject("View")
    ){	
	throw TScriptException(
	    "Grid::put(View child_view)", "invalid argument[s]"
	);
    }

    _Grid->AddView(
	((TKinokoViewMessenger*) ArgumentList[0]->AsObject())->View()
    );
    
    return 1;
}

TKinokoView* TKinokoViewGridMessenger::View(void)
{
    return _Grid;
}

TKinokoDataAnalyzer* TKinokoViewGridMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    return 0;
}



TKinokoViewPlacerMessenger::TKinokoViewPlacerMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
}

TKinokoViewPlacerMessenger::~TKinokoViewPlacerMessenger()
{
}

TParaObjectPrototype* TKinokoViewPlacerMessenger::Clone(void)
{
    return new TKinokoViewPlacerMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

int TKinokoViewPlacerMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (Message == "put") {
	return PutView(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

void TKinokoViewPlacerMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() == 0) {
	_Placer = new TKinokoViewPlacerFrame();
    }
    else if (
	(ArgumentList.size() < 4) ||
	(! ArgumentList[0]->IsNumeric()) ||
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric())
    ){
	throw TScriptException(
	    "Placer::Placer(float x0, float x1, float y0, float y1)", 
	    "invalid argument[s]"
	);
    }
    else {
	float X0 = ArgumentList[0]->AsDouble();
	float X1 = ArgumentList[1]->AsDouble();
	float Y0 = ArgumentList[2]->AsDouble();
	float Y1 = ArgumentList[3]->AsDouble();
	_Placer = new TKinokoViewPlacerFrame(X0, X1, Y0, Y1);
    }

    _ViewBuilder->AddView(_Placer);
}

int TKinokoViewPlacerMessenger::PutView(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 5) ||
	(! ArgumentList[0]->IsObject("View"))
    ){	
	throw TScriptException(
	    "Placer::put"
            "(View child_view, float x0, float x1, float y0, float y1)", 
	    "invalid argument[s]"
	);
    }

    TKinokoViewPlacement Placement;
    try {
	Placement.x0 = ArgumentList[1]->AsDouble();
	Placement.x1 = ArgumentList[2]->AsDouble();
	Placement.y0 = ArgumentList[3]->AsDouble();
	Placement.y1 = ArgumentList[4]->AsDouble();
    }
    catch (TScriptException &e) {
	throw TScriptException(
	    "Placer::put"
            "(View child_view, float x0, float x1, float y0, float y1)", 
	    "invalid argument[s]"
	);
    }

    _Placer->AddViewAt(
	((TKinokoViewMessenger*) ArgumentList[0]->AsObject())->View(),
	Placement
    );
    
    return 1;
}

TKinokoView* TKinokoViewPlacerMessenger::View(void)
{
    return _Placer;
}

TKinokoDataAnalyzer* TKinokoViewPlacerMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    return 0;
}



TKinokoHistogramViewMessenger::TKinokoHistogramViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _HistogramView = 0;
    _Histogram = 0;
}

TKinokoHistogramViewMessenger::~TKinokoHistogramViewMessenger()
{
}

TParaObjectPrototype* TKinokoHistogramViewMessenger::Clone(void)
{
    return new TKinokoHistogramViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoHistogramViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException("Histogram::Histogram(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsLong()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric())
    ){
	throw TScriptException("Histogram::Histogram(): invalid argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    int NumberOfBins = ArgumentList[1]->AsLong();
    double Min = ArgumentList[2]->AsDouble();
    double Max = ArgumentList[3]->AsDouble();

    if ((NumberOfBins <= 0) || (Max <= Min)) {
	throw TScriptException("Histogram::Histogram(): incorrect bin range");
    }

    _Histogram = new TKaspHistogram(Title, NumberOfBins, Min, Max);
    _HistogramView = _ViewFactory->CreateHistogramView(_Histogram);
    _HistogramView->SetName(_Name);

    _ViewBuilder->AddView(_HistogramView);
}

int TKinokoHistogramViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "setYScaleLog") || (Message == "setLogY")) {
	return SetYScaleLog(ArgumentList, ReturnValue);
    }
    else if ((Message == "setYScaleLinear") || (Message == "setLinearY")) {
	return SetYScaleLinear(ArgumentList, ReturnValue);
    }
    else if (Message == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoHistogramViewMessenger::View(void)
{
    return _HistogramView;
}

TKinokoDataAnalyzer* TKinokoHistogramViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoHistogramWriter* AnalysisAction = new TKinokoHistogramWriter(
	_Histogram, 
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoHistogramViewMessenger::SetYScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetYScaleLog();
    return 1;
}

int TKinokoHistogramViewMessenger::SetYScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetYScaleLinear();
    return 1;
}

int TKinokoHistogramViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _Histogram->Scale().SetTitle(XTitle);
    _HistogramView->SetAxisTitle(XTitle, YTitle);

    return 1;
}



TKinoko2dHistogramViewMessenger::TKinoko2dHistogramViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _HistogramView = 0;
    _Histogram = 0;
}

TKinoko2dHistogramViewMessenger::~TKinoko2dHistogramViewMessenger()
{
}

TParaObjectPrototype* TKinoko2dHistogramViewMessenger::Clone(void)
{
    return new TKinoko2dHistogramViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinoko2dHistogramViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 7) {
	throw TScriptException("Histogram2d::Histogram2d(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsLong()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsLong()) ||
	(! ArgumentList[5]->IsNumeric()) ||
	(! ArgumentList[6]->IsNumeric())
    ){
	throw TScriptException("Histogram2d::Histogram2d(): invalid argument");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    int NumberOfXBins = ArgumentList[1]->AsLong();
    double XMin = ArgumentList[2]->AsDouble();
    double XMax = ArgumentList[3]->AsDouble();
    int NumberOfYBins = ArgumentList[4]->AsLong();
    double YMin = ArgumentList[5]->AsDouble();
    double YMax = ArgumentList[6]->AsDouble();

    if (
	(NumberOfXBins <= 0) || (XMax <= XMin) || 
	(NumberOfYBins <= 0) || (YMax <= YMin)
    ){
	throw TScriptException(
	    "Histogram2d::Histogram2d(): incorrect bin range"
	);
    }
    
    _Histogram = new TKasp2dHistogram(
	Title,
	TKaspHistogramScale(NumberOfXBins, XMin, XMax),
	TKaspHistogramScale(NumberOfYBins, YMin, YMax)
    );
    _HistogramView = _ViewFactory->Create2dHistogramView(_Histogram);
    _HistogramView->SetName(_Name);

    _ViewBuilder->AddView(_HistogramView);
}

int TKinoko2dHistogramViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (Message == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }
    else if (Message == "disableZScale") {
	return DisableZScale(ArgumentList, ReturnValue);
    }
    else if ((Message == "setZScaleLog") || (Message == "setLogZ")) {
	return SetZScaleLog(ArgumentList, ReturnValue);
    }
    else if ((Message == "setZScaleLinear") || (Message == "setLinearZ")) {
	return SetZScaleLinear(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

TKinokoView* TKinoko2dHistogramViewMessenger::View(void)
{
    return _HistogramView;
}

TKinokoDataAnalyzer* TKinoko2dHistogramViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinoko2dHistogramWriter* AnalysisAction = new TKinoko2dHistogramWriter(
	_Histogram,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier(),
	DataElementMessengerList[1]->SectionPath(),
	DataElementMessengerList[1]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinoko2dHistogramViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _HistogramView->SetAxisTitle(XTitle, YTitle);

    return 1;
}

int TKinoko2dHistogramViewMessenger::DisableZScale(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->DisableZScale();

    return 1;
}

int TKinoko2dHistogramViewMessenger::SetZScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetZScaleLog();
    return 1;
}

int TKinoko2dHistogramViewMessenger::SetZScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistogramView->SetZScaleLinear();
    return 1;
}



TKinokoHistoryViewMessenger::TKinokoHistoryViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _HistoryView = 0;
    _History = 0;
}

TKinokoHistoryViewMessenger::~TKinokoHistoryViewMessenger()
{
}

TParaObjectPrototype* TKinokoHistoryViewMessenger::Clone(void)
{
    return new TKinokoHistoryViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoHistoryViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException("History::History(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsLong())
    ){
	throw TScriptException("History::History(): invalid argument");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double MinValue = ArgumentList[1]->AsDouble();
    double MaxValue = ArgumentList[2]->AsDouble();
    int NumberOfSamples = ArgumentList[3]->AsLong();

    double WindowWidth = NumberOfSamples;
    if (ArgumentList.size() > 4) {
        if (! ArgumentList[4]->IsNumeric()) {
	    throw TScriptException(
	        "History::History(string name, double range_min, double range_max, int number_of_samples, double window_width): invalid argument"
	    );
	}
	WindowWidth = ArgumentList[4]->AsLong();
    }

    _History = new TKaspHistory(Title, NumberOfSamples);
    _HistoryView = _ViewFactory->CreateHistoryView(
	_History, MinValue, MaxValue, WindowWidth
    );
    _HistoryView->SetName(_Name);

    _ViewBuilder->AddView(_HistoryView);

    if (ClassName == "MeanValueHistory") {
        _HistoryView->SetDisplayValue(TKinokoHistoryView::FieldId_Mean);
    }
    else if (ClassName == "TotalValueHistory") {
        _HistoryView->SetDisplayValue(TKinokoHistoryView::FieldId_Sum);
    }
    else if (ClassName == "EntryCountHistory") {
        _HistoryView->SetDisplayValue(TKinokoHistoryView::FieldId_Counts);
    }
}

int TKinokoHistoryViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "drawCounts") || (Message == "drawCounts")) {
	return DrawCounts(ArgumentList, ReturnValue);
    }
    else if ((Message == "drawSum") || (Message == "drawSum")) {
	return DrawSum(ArgumentList, ReturnValue);
    }
    else if ((Message == "drawMean") || (Message == "drawMean")) {
	return DrawMean(ArgumentList, ReturnValue);
    }
    else if ((Message == "drawDeviation") || (Message == "drawDeviation")) {
	return DrawDeviation(ArgumentList, ReturnValue);
    }
    else if ((Message == "setYScaleLog") || (Message == "setLogY")) {
	return SetYScaleLog(ArgumentList, ReturnValue);
    }
    else if ((Message == "setYScaleLinear") || (Message == "setLinearY")) {
	return SetYScaleLinear(ArgumentList, ReturnValue);
    }
    else if (Message == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }
    else if (Message == "hold") {
	return Hold(ArgumentList, ReturnValue);
    }
    else if (Message == "setOperationRange") {
	return SetOperationRange(ArgumentList, ReturnValue);
    }
    else if (Message == "enableAlarm") {
	return EnableAlarm(ArgumentList, ReturnValue);
    }
    else if (Message == "disableAlarm") {
	return DisableAlarm(ArgumentList, ReturnValue);
    }
    else if ((Message == "fillOne") || (Message == "FillOne")) {
	throw TScriptException(
	    "History::fillOne()",
	    "interface not available (use fill() instead)"
	);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoHistoryViewMessenger::View(void)
{
    return _HistoryView;
}

TKinokoDataAnalyzer* TKinokoHistoryViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoHistoryWriter* AnalysisAction = new TKinokoHistoryWriter(
	_History,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoHistoryViewMessenger::DrawCounts(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Counts
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DrawSum(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Sum
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DrawMean(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Mean
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DrawDeviation(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(new TKinokoHistoryViewDrawAction(
            _HistoryView, TKinokoHistoryView::FieldId_Deviation
	));
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int  TKinokoHistoryViewMessenger::SetYScaleLog(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistoryView->SetYScaleLog();
    return 1;
}

int  TKinokoHistoryViewMessenger::SetYScaleLinear(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _HistoryView->SetYScaleLinear();
    return 1;
}

int TKinokoHistoryViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _HistoryView->SetAxisTitle(XTitle, YTitle);

    return 1;
}

int TKinokoHistoryViewMessenger::Hold(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
        _ViewBuilder->AddAction(
	    new TKinokoHistoryViewHoldAction(_HistoryView, _History)
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    _HistoryView->DisableAutoHold();

    return 1;
}

int TKinokoHistoryViewMessenger::SetOperationRange(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(ArgumentList.size() < 2) ||
	(! ArgumentList[0]->IsNumeric()) ||
	(! ArgumentList[1]->IsNumeric())
    ){
	throw TScriptException(
	    "History::setOperationRange(float lower_bound, float upper_bound)"
	    "invalid argument[s]"
	);
    }

    double LowerBound = ArgumentList[0]->AsDouble();
    double UpperBound = ArgumentList[1]->AsDouble();

    try {
	_ViewBuilder->AddAction(
	    new TKinokoHistoryViewSetOperationRangeAction(
		_HistoryView, LowerBound, UpperBound
	    )
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::EnableAlarm(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string Message;
    if (ArgumentList.size() > 0) {
	Message = ArgumentList[0]->AsString();
    }

    try {
	_ViewBuilder->AddAction(
	    new TKinokoHistoryViewEnableAlarmAction(
		_HistoryView, _ViewBuilder->EventEmitter(), Message
	    )
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}

int TKinokoHistoryViewMessenger::DisableAlarm(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    try {
	_ViewBuilder->AddAction(
	    new TKinokoHistoryViewDisableAlarmAction(_HistoryView)
	);
    }
    catch (TKinokoException &e) {
	throw TScriptException(e.Message());
    }

    return 1;
}



TKinokoWaveViewMessenger::TKinokoWaveViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _WaveView = 0;
    _Wave = 0;
}

TKinokoWaveViewMessenger::~TKinokoWaveViewMessenger()
{
}

TParaObjectPrototype* TKinokoWaveViewMessenger::Clone(void)
{
    return new TKinokoWaveViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoWaveViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 5) {
	throw TScriptException("Wave::Wave(): too few argument[s]");
    }
    if (
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsNumeric())
    ){
	throw TScriptException("Wave::Wave(): invalid argument");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double XMin = ArgumentList[1]->AsDouble();
    double XMax = ArgumentList[2]->AsDouble();
    double YMin = ArgumentList[3]->AsDouble();
    double YMax = ArgumentList[4]->AsDouble();
    int InitialCapacity = (int) (XMax - XMin) + 1;
    
    _Wave = new TKaspWave(Title, InitialCapacity);
    _WaveView = _ViewFactory->CreateWaveView(_Wave, XMin, XMax, YMin, YMax);
    _WaveView->SetName(_Name);

    _ViewBuilder->AddView(_WaveView);
}

int TKinokoWaveViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (Message == "setAxisTitle") {
	return SetAxisTitle(ArgumentList, ReturnValue);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoWaveViewMessenger::View(void)
{
    return _WaveView;
}

TKinokoDataAnalyzer* TKinokoWaveViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoWaveWriter* AnalysisAction = new TKinokoWaveWriter(
	_Wave,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoWaveViewMessenger::SetAxisTitle(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string XTitle, YTitle;
    if (ArgumentList.size() > 0) {
	XTitle = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	YTitle = ArgumentList[1]->AsString();
    }

    _WaveView->SetAxisTitle(XTitle, YTitle);

    return 1;
}



TKinokoMapViewMessenger::TKinokoMapViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _MapView = 0;
    _Map = 0;
}

TKinokoMapViewMessenger::~TKinokoMapViewMessenger()
{
}

TParaObjectPrototype* TKinokoMapViewMessenger::Clone(void)
{
    return new TKinokoMapViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoMapViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 7) {
	throw TScriptException("Map::Map(): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsString()) ||
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsNumeric()) ||
	(! ArgumentList[5]->IsNumeric()) ||
	(! ArgumentList[6]->IsNumeric())
    ){
	throw TScriptException("Map::Map(): invalid argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double XMin = ArgumentList[1]->AsDouble();
    double XMax = ArgumentList[2]->AsDouble();
    double YMin = ArgumentList[3]->AsDouble();
    double YMax = ArgumentList[4]->AsDouble();
    double ZMin = ArgumentList[5]->AsDouble();
    double ZMax = ArgumentList[6]->AsDouble();
    
    _Map = new TKaspMap(Title);
    _MapView = _ViewFactory->CreateMapView(
	_Map, XMin, XMax, YMin, YMax, ZMin, ZMax
    );
    _MapView->SetName(_Name);

    _ViewBuilder->AddView(_MapView);
}

int TKinokoMapViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "addPoint") || (Message == "AddPoint")) {
	return AddPoint(ArgumentList, ReturnValue);
    }
    else if ((Message == "setPointSize") || (Message == "SetPointSize")) {
	return SetPointSize(ArgumentList, ReturnValue);
    }
    else if ((Message == "fill") || (Message == "Fill")) {
	throw TScriptException(
	    "Map::fill()",
	    "interface not available (use fillOne() instead)"
	);
    }
    else {
	return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
    }
}

TKinokoView* TKinokoMapViewMessenger::View(void)
{
    return _MapView;
}

TKinokoDataAnalyzer* TKinokoMapViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoMapWriter* AnalysisAction = new TKinokoMapWriter(
	_Map,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}

int TKinokoMapViewMessenger::AddPoint(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException("Map::addPoint(): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsLong()) || 
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric())
    ){
	throw TScriptException("Map::addPoint(): invalid argument");
    }

    long Address = ArgumentList[0]->AsLong();
    double PositionX = ArgumentList[1]->AsDouble();
    double PositionY = ArgumentList[2]->AsDouble();

    _MapView->AddPoint(Address, PositionX, PositionY);

    return 1;
}

int TKinokoMapViewMessenger::SetPointSize(vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("Map::setPointSize(): too few argument[s]");
    }
    if (! ArgumentList[0]->IsNumeric()) {
	throw TScriptException("Map::setPointSize(): invalid argument");
    }

    double PointSize = ArgumentList[0]->AsDouble();
    _MapView->SetPointSize(PointSize);

    return 1;
}



TKinokoTabularViewMessenger::TKinokoTabularViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _TabularView = 0;
    _Tabular = 0;
}

TKinokoTabularViewMessenger::~TKinokoTabularViewMessenger()
{
}

TParaObjectPrototype* TKinokoTabularViewMessenger::Clone(void)
{
    return new TKinokoTabularViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoTabularViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("Tabular::Tabular(): too few argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();

    int NumberOfColumns = 1;
    if (ArgumentList.size() >= 2) {
	if (! ArgumentList[1]->IsLong()) {
	    throw TScriptException("Tabular::Tabular(): invalid argument");
	}
	NumberOfColumns = ArgumentList[1]->AsLong();
    }

    _Tabular = new TKaspTabular(Title);
    _TabularView = _ViewFactory->CreateTabularView(_Tabular, NumberOfColumns);
    _TabularView->SetName(_Name);

    _ViewBuilder->AddView(_TabularView);
}

int TKinokoTabularViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if ((Message == "fill") || (Message == "Fill")) {
	throw TScriptException(
	    "Tabular::fill()",
	    "interface not available (use fillOne() instead)"
	);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoTabularViewMessenger::View(void)
{
    return _TabularView;
}

TKinokoDataAnalyzer* TKinokoTabularViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    TKinokoTabularWriter* AnalysisAction = new TKinokoTabularWriter(
	_Tabular,
	DataElementMessengerList[0]->SectionPath(),
	DataElementMessengerList[0]->DataElementSpecifier()
    );
    AnalysisAction->SetTargetName(_Name);

    return AnalysisAction;
}



TKinokoPictureViewMessenger::TKinokoPictureViewMessenger(TKinokoViewBuilder* ViewBuilder, TKinokoAnalysisSequenceBuilder* AnalysisSequenceBuilder)
: TKinokoViewMessenger(ViewBuilder, AnalysisSequenceBuilder)
{
    _PictureView = 0;
}

TKinokoPictureViewMessenger::~TKinokoPictureViewMessenger()
{
}

TParaObjectPrototype* TKinokoPictureViewMessenger::Clone(void)
{
    return new TKinokoPictureViewMessenger(_ViewBuilder, _AnalysisSequenceBuilder);
}

void TKinokoPictureViewMessenger::Construct(const string& ClassName, vector<TParaValue*>& ArgumentList) throw(TScriptException)
{
    if (ArgumentList.size() < 5) {
	throw TScriptException("Picture::Picture(): too few argument[s]");
    }
    if (
	(! ArgumentList[0]->IsString()) ||
	(! ArgumentList[1]->IsNumeric()) ||
	(! ArgumentList[2]->IsNumeric()) ||
	(! ArgumentList[3]->IsNumeric()) ||
	(! ArgumentList[4]->IsNumeric())
    ){
	throw TScriptException("Picture::Picture(): invalid argument[s]");
    }

    _Name = this->ObjectName();
    string Title = ArgumentList[0]->AsString();
    double XMin = ArgumentList[1]->AsDouble();
    double XMax = ArgumentList[2]->AsDouble();
    double YMin = ArgumentList[3]->AsDouble();
    double YMax = ArgumentList[4]->AsDouble();
    
    _PictureView = _ViewFactory->CreatePictureView(
	Title, XMin, XMax, YMin, YMax
    );
    _PictureView->SetName(_Name);

    _ViewBuilder->AddView(_PictureView);
}

int TKinokoPictureViewMessenger::DispatchMessage(const string& Message, vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (
	(Message == "fill") || (Message == "Fill") ||
	(Message == "fillOne") || (Message == "FillOne")
    ){
	throw TScriptException(
	    "Picture::fill()", "interface not available"
	);
    }

    return TKinokoViewMessenger::DispatchMessage(Message, ArgumentList, ReturnValue);
}

TKinokoView* TKinokoPictureViewMessenger::View(void)
{
    return _PictureView;
}

TKinokoDataAnalyzer* TKinokoPictureViewMessenger::CreateAnalysisAction(vector<TKinokoDataElementMessenger*> DataElementMessengerList)
{
    return 0;
}
