/* KinokoViewScriptStatement.cc */
/* Created by Enomoto Sanshiro on 26 December 2000. */
/* Last updated by Enomoto Sanshiro on 23 September 2001. */


#include <string>
#include <vector>
#include "ParaParser.hh"
#include "KinokoViewBuilder.hh"
#include "KinokoViewMessenger.hh"
#include "KinokoViewScriptStatement.hh"

using namespace std;


TKinokoViewScriptOnStatement::TKinokoViewScriptOnStatement(TKinokoViewBuilder* ViewBuilder)
{
    _ViewBuilder = ViewBuilder;
    _ConcreteStatement = 0;
}

TKinokoViewScriptOnStatement::~TKinokoViewScriptOnStatement()
{
    delete _ConcreteStatement;
}

TParaStatement* TKinokoViewScriptOnStatement::Clone(void)
{
    return new TKinokoViewScriptOnStatement(_ViewBuilder);
}

string TKinokoViewScriptOnStatement::FirstToken(void) const
{
    return string("on");
}

void TKinokoViewScriptOnStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->LookAhead(1).MustBe(FirstToken());
    if (Tokenizer->LookAhead(2).Is("every")) {
	_ConcreteStatement = new TKinokoViewScriptOnEveryStatement(_ViewBuilder);
    }
    else if (Tokenizer->LookAhead(2).Is("command")) {
	_ConcreteStatement = new TKinokoViewScriptOnCommandStatement(_ViewBuilder);
    }
    else {
	_ConcreteStatement = new TKinokoViewScriptOnTrapStatement(_ViewBuilder);
    }

    _ConcreteStatement->Parse(Tokenizer, StatementParser, SymbolTable);
}

TParaStatement::TExecResult TKinokoViewScriptOnStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    return _ConcreteStatement->Execute(SymbolTable);
}



TKinokoViewScriptOnEveryStatement::TKinokoViewScriptOnEveryStatement(TKinokoViewBuilder* ViewBuilder)
: TKinokoViewScriptOnStatement(ViewBuilder)
{
    _Statement = 0;
}

TKinokoViewScriptOnEveryStatement::~TKinokoViewScriptOnEveryStatement()
{
    delete _Statement;
}

TParaStatement* TKinokoViewScriptOnEveryStatement::Clone(void)
{
    return new TKinokoViewScriptOnEveryStatement(_ViewBuilder);
}

void TKinokoViewScriptOnEveryStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    TParaExpressionParser* ExpressionParser;
    ExpressionParser = StatementParser->ExpressionParser();

    Tokenizer->Next().MustBe(FirstToken());
    Tokenizer->Next().MustBe("every");
    Tokenizer->Next().MustBe("(");

    _IntervalExpression = ExpressionParser->Parse(Tokenizer, SymbolTable);

    if (Tokenizer->LookAhead().Is("sec")) {
	Tokenizer->Next();
    }
    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoViewScriptOnEveryStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    long Interval;
    try {
	Interval = _IntervalExpression->Evaluate(SymbolTable).AsLong();
    }
    catch (TScriptException &e) {
	throw TScriptException(_IntervalExpression->Position() + e.Message());
    }

    _ViewBuilder->OpenTimeSequence(Interval);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _ViewBuilder->CloseSequence();

    return Result;
}



TKinokoViewScriptOnCommandStatement::TKinokoViewScriptOnCommandStatement(TKinokoViewBuilder* ViewBuilder)
: TKinokoViewScriptOnStatement(ViewBuilder)
{
    _Statement = 0;
}

TKinokoViewScriptOnCommandStatement::~TKinokoViewScriptOnCommandStatement()
{
    delete _Statement;
}

TParaStatement* TKinokoViewScriptOnCommandStatement::Clone(void)
{
    return new TKinokoViewScriptOnCommandStatement(_ViewBuilder);
}

void TKinokoViewScriptOnCommandStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    Tokenizer->Next().MustBe("command");

    Tokenizer->Next().MustBe("(");
    _CommandName = Tokenizer->Next().RemoveQuotation('"').AsString();
    Tokenizer->Next().MustBe(")");

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoViewScriptOnCommandStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _ViewBuilder->OpenCommandSequence(_CommandName);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _ViewBuilder->CloseSequence();

    return Result;
}



TKinokoViewScriptOnTrapStatement::TKinokoViewScriptOnTrapStatement(TKinokoViewBuilder* ViewBuilder)
: TKinokoViewScriptOnStatement(ViewBuilder)
{
    _Statement = 0;
}

TKinokoViewScriptOnTrapStatement::~TKinokoViewScriptOnTrapStatement()
{
    delete _Statement;
}

TParaStatement* TKinokoViewScriptOnTrapStatement::Clone(void)
{
    return new TKinokoViewScriptOnTrapStatement(_ViewBuilder);
}

void TKinokoViewScriptOnTrapStatement::Parse(TParaTokenizer* Tokenizer, TParaStatementParser* StatementParser, TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    Tokenizer->Next().MustBe(FirstToken());
    string TrapName = Tokenizer->Next().RemoveQuotation('"').AsString();

    if (TrapName == "construct") {
	_TrapId = TKinokoViewSequenceTable::TrapId_Construct;
    }
    else if (TrapName == "destruct") {
	_TrapId = TKinokoViewSequenceTable::TrapId_Destruct;
    }
    else if (TrapName == "run_begin") {
	_TrapId = TKinokoViewSequenceTable::TrapId_RunBegin;
    }
    else if (TrapName == "run_end") {
	_TrapId = TKinokoViewSequenceTable::TrapId_RunEnd;
    }
    else if (TrapName == "run_suspend") {
	_TrapId = TKinokoViewSequenceTable::TrapId_RunSuspend;
    }
    else if (TrapName == "run_resume") {
	_TrapId = TKinokoViewSequenceTable::TrapId_RunResume;
    }
    else if (TrapName == "pre_clear") {
	_TrapId = TKinokoViewSequenceTable::TrapId_PreClear;
    }
    else if ((TrapName == "post_clear") || (TrapName == "clear")) {
	_TrapId = TKinokoViewSequenceTable::TrapId_PostClear;
    }
    else {
	throw TScriptException("unknown on-statement tag: " + TrapName);
    }

    _Statement = StatementParser->Parse(Tokenizer, SymbolTable);
}

TParaStatement::TExecResult TKinokoViewScriptOnTrapStatement::Execute(TParaSymbolTable* SymbolTable) throw(TScriptException)
{
    _ViewBuilder->OpenTrapSequence(_TrapId);
    TExecResult Result = _Statement->Execute(SymbolTable);
    _ViewBuilder->CloseSequence();

    return Result;
}
