/* KinokoViewer.cc */
/* Created by Enomoto Sanshiro on 31 October 2000. */
/* Last updated by Enomoto Sanshiro on 20 September 2001. */


#include <strstream>
#include <string>
#include <vector>
#include "MushTimer.hh"
#include "KinokoPlatform.hh"
#include "KinokoDataAnalyzer.hh"
#include "KinokoAnalysisSequenceBuilder.hh"
#include "KinokoView.hh"
#include "KinokoViewFactory.hh"
#include "KinokoViewCanvas.hh"
#include "KinokoViewBuilder.hh"
#include "KinokoViewScript.hh"
#include "KinokoViewer.hh"

using namespace std;


TKinokoViewer::TKinokoViewer(TKinokoEventEmitter* EventEmitter, TKinokoRegistry* Registry)
{
    _EventEmitter = EventEmitter;
    _Registry = Registry;

    _AnalysisSequenceBuilder = 0;
    _ViewBuilder = 0;
    _ViewFactory = 0;
    _ViewScript = 0;

    _ViewCanvas = 0;
    _ViewSequenceTable = 0;
    _DataDistributor = 0;

    _IsFirstRun = true;
    _IsRunning = false;
}

TKinokoViewer::~TKinokoViewer()
{
    delete _ViewBuilder;
    delete _AnalysisSequenceBuilder;

    delete _ViewScript;
}

void TKinokoViewer::SetScript(const string& ScriptFileName, const string& DisplayEntryName)
{
    _ScriptFileName = ScriptFileName;
    _DisplayEntryName = DisplayEntryName;
}

void TKinokoViewer::OnConstruct(void) throw(TKinokoException)
{
    if (_ScriptFileName.empty()) {
  	throw TKinokoException("script file is not specified");
    }	

    ifstream ScriptFile(_ScriptFileName.c_str());
    if (! ScriptFile) {
  	throw TKinokoException("unable to open file: " + _ScriptFileName);
    }	

    _AnalysisSequenceBuilder = new TKinokoAnalysisSequenceBuilder(
	_EventEmitter, _Registry
    );
    _ViewFactory = new TKinokoKoapViewFactory(_OutputIoStream);
    _ViewBuilder = new TKinokoViewBuilder(
	_ViewFactory, _EventEmitter, _Registry
    );
    _ViewScript = new TKinokoViewScript(
	_ViewBuilder, _AnalysisSequenceBuilder, _ScriptFileName
    );

    try {
	_ViewScript->Parse(ScriptFile);

	if (_DisplayEntryName.empty()) {
	    _DisplayEntryName = _ViewScript->DisplayNameList()[0];
	}

	_ViewScript->Execute(_DisplayEntryName);
    }
    catch (TScriptException &e) {
	throw TKinokoException("script exception: " + e.Message());
    }

    _DataDistributor = _AnalysisSequenceBuilder->DataDistributor();
    _AnalysisSequenceList = _AnalysisSequenceBuilder->RootSequenceList();
    _ViewCanvas = _ViewBuilder->ViewCanvas();
    _ViewSequenceTable = _ViewBuilder->SequenceTable();

    _ViewCanvas->DeployViews();
    _ViewCanvas->Clear();
    _ViewSequenceTable->ExecuteTrapSequence(
        TKinokoViewSequenceTable::TrapId_Construct
    );
    _ViewSequenceTable->ExecuteTrapSequence(
	TKinokoViewSequenceTable::TrapId_PostClear
    );
    _ViewCanvas->Update();

    _IsFirstRun = true;
    _IsRunning = false;
}

void TKinokoViewer::OnDestruct(void) throw(TKinokoException)
{
    if (_ViewSequenceTable != 0) {
        _ViewSequenceTable->ExecuteTrapSequence(
            TKinokoViewSequenceTable::TrapId_Destruct
	);
    }

    delete _AnalysisSequenceBuilder;
    delete _ViewFactory;
    delete _ViewBuilder;
    delete _ViewScript;

    _AnalysisSequenceBuilder = 0;
    _ViewFactory = 0;
    _ViewBuilder = 0;
    _ViewScript = 0;

    _ViewCanvas = 0;
    _ViewSequenceTable = 0;
    _DataDistributor = 0;
}

void TKinokoViewer::OnRunBegin(void) throw(TKinokoException) 
{
    if ((_DataDistributor != 0) && (_IsFirstRun)) {
	_DataDistributor->ProcessDataDescriptor(_InputDataDescriptor);
	_IsFirstRun = false;
    }

    if (_ViewSequenceTable != 0) {
	_ViewSequenceTable->ExecuteTrapSequence(
	    TKinokoViewSequenceTable::TrapId_RunBegin
	);
	_ViewCanvas->Update();
    }

    _IsRunning = true;
}

void TKinokoViewer::OnRunEnd(void) throw(TKinokoException) 
{
    if (_ViewSequenceTable != 0) {
	_ViewSequenceTable->ExecuteTrapSequence(
	    TKinokoViewSequenceTable::TrapId_RunEnd
	);
	_ViewCanvas->Update();
    }

    _IsRunning = false;
}

int TKinokoViewer::ProcessData(void) throw(TKinokoException)
{
    int Result = 0;

    if (_InputStream->HasData()) {
	Result = TKinokoDataConsumer::ProcessData();
    }

    if (_IsRunning && _ViewSequenceTable != 0) {
	if (_ViewSequenceTable->ExecuteTimeSequence() > 0) {
	    _ViewCanvas->Update();

	    //...
	    for (unsigned i = 0; i < _AnalysisSequenceList.size(); i++) {
		_AnalysisSequenceList[i]->Flush();
	    }
	}
    }

    return Result;
}

void TKinokoViewer::OnReceiveDataPacket(void* DataPacket, long PacketSize) throw(TKinokoException) 
{
    if (_DataDistributor != 0) {
	_DataDistributor->ProcessDataPacket(DataPacket);
    }
}

void TKinokoViewer::OnReceiveTrailerPacket(void* DataPacket, long PacketSize) throw(TKinokoException) 
{
    if (_DataDistributor != 0) {
	_DataDistributor->ProcessTrailerPacket(DataPacket);
    }
}

int TKinokoViewer::ExecuteCommand(const std::string& Command) throw(TKinokoException)
{
    int Result = 0;

    if (_ViewSequenceTable != 0) {
	if (_ViewSequenceTable->ExecuteCommandSequence(Command)) {
	    _ViewCanvas->Update();
	    for (unsigned i = 0; i < _AnalysisSequenceList.size(); i++) {
		_AnalysisSequenceList[i]->Flush();
	    }
	    Result = 1;
	}
    }

    return Result;
}

void TKinokoViewer::Reset(void)
{
    for (unsigned i = 0; i < _AnalysisSequenceList.size(); i++) {
	_AnalysisSequenceList[i]->Flush();
	_AnalysisSequenceList[i]->Reset();
    }

    if (_ViewSequenceTable != 0) {
	_ViewSequenceTable->ExecuteTrapSequence(
	    TKinokoViewSequenceTable::TrapId_PreClear
	);
    }

    if (_ViewCanvas != 0) {
	_ViewCanvas->Clear();
    }

    if (_ViewSequenceTable != 0) {
	_ViewSequenceTable->ExecuteTrapSequence(
	    TKinokoViewSequenceTable::TrapId_PostClear
	);
	_ViewCanvas->Update();
    }
}

void TKinokoViewer::SaveImage(const string& ImageFileName)
{
    if (_ViewCanvas != 0) {
	_ViewCanvas->SaveImage(ImageFileName);
    }
}
