/* kdfcheck.cc */
/* Created by Enomoto Sanshiro on 28 December 2001. */
/* Last updated by Enomoto Sanshiro on 6 October 2002. */


#include <cstdlib>
#include <iostream>
#include <strstream>
#include <iomanip>
#include <string>
#include "MushFile.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataProcessor.hh"

using namespace std;


class TKinokoKdfCheck {
  public:
    TKinokoKdfCheck(const string& DataFileName, bool IsQuiet, bool IsRaw, long StartIndex);
    virtual ~TKinokoKdfCheck();
    virtual void Start(void) throw(TKinokoException);
  protected:
    virtual void Construct(void) throw(TKinokoException);
    virtual void Destruct(void) throw(TKinokoException);
    virtual void ShowPreamble(void);
    virtual void ShowHeader(TKinokoStorageHeader& Header);
    virtual void ProcessPacket(void* DataPacket, long PacketSize) throw(TKinokoException);
  protected:
    virtual void ProcessDataDescriptorPacket(void* DataPacket, long PacketSize);
    virtual void ProcessCommandPacket(void* DataPacket, long PacketSize);
    virtual void ProcessTrailerPacket(void* DataPacket, long PacketSize);
    virtual void ProcessDataPacket(void* DataPacket, long PacketSize);
  protected:
    string _DataFileName;
    bool _IsQuiet;
    bool _IsRawStorage;
    long _StartIndex;
    TKinokoKdfStorage* _Storage;
    TKinokoInputFileStream* _InputStream;
    TKinokoDataStreamScanner* _StreamScanner;
    long _PacketCount;
};



TKinokoKdfCheck::TKinokoKdfCheck(const string& DataFileName, bool IsQuiet, bool IsRaw, long StartIndex)
{
    _DataFileName = DataFileName;
    _IsQuiet = IsQuiet;
    _IsRawStorage = IsRaw;
    _StartIndex = StartIndex;

    _Storage = 0;
    _InputStream = 0;
    _StreamScanner = 0;
    
    _PacketCount = 0;
}

TKinokoKdfCheck::~TKinokoKdfCheck()
{
    delete _StreamScanner;
    delete _Storage;
}

void TKinokoKdfCheck::Start(void) throw(TKinokoException)
{
    Construct();

    if (_StartIndex > 0) {
	_InputStream->JumpTo(_StartIndex);
	_PacketCount = _StartIndex;
    }

    void* Packet;
    int PacketSize;
    while ((PacketSize = _InputStream->NextEntry(Packet)) > 0) {
	ProcessPacket(Packet, PacketSize);
	_InputStream->Flush(Packet);
    }

    Destruct();
}

void TKinokoKdfCheck::Construct(void) throw(TKinokoException)
{
    _Storage = new TKinokoKdfStorage(_DataFileName, _IsRawStorage);

    if (! _IsRawStorage) {
	TKinokoStorageHeader StorageHeader;
	try {
	    _Storage->ReadHeader(StorageHeader);
	    ShowPreamble();
	    if (! _IsQuiet) {
	        ShowHeader(StorageHeader);
	    }
	}
	catch (TKinokoException &e) {
	    /* version number mismatch */
	    cout << endl; 
	    cout << "WARNING: " << e << endl << endl;
	}
    }

    try {
        _InputStream = _Storage->GetInputStream();
    }
    catch (TKinokoException &e) {
	throw;
    }

    _StreamScanner = new TKinokoDataStreamScanner();
}

void TKinokoKdfCheck::Destruct(void) throw(TKinokoException)
{
}

void TKinokoKdfCheck::ProcessPacket(void* Packet, long PacketSize) throw(TKinokoException)
{
    if (_StreamScanner->IsByteOrderReversed(Packet)) {
	_StreamScanner->CorrectByteOrder(Packet, PacketSize);
	cout << "BYTE ORDER REVERSED" << endl;
    }

    //...
    off_t Offset = _Storage->InputRawFile()->Position() - (PacketSize + 4);

    long DataSourceId = _StreamScanner->DataSourceIdOf(Packet);
    cout << "[" << DataSourceId << "]--- ";
    cout << _PacketCount++ << ", ";
    cout << PacketSize << " bytes ";
    cout << "@" << hex << Offset << dec << endl;
    
    if (_StreamScanner->IsDataDescriptorPacket(Packet)) {
	ProcessDataDescriptorPacket(Packet, PacketSize);
    }
    else if (_StreamScanner->IsCommandPacket(Packet)) {
	ProcessCommandPacket(Packet, PacketSize);
    }
    else if (_StreamScanner->IsTrailerPacket(Packet)) {
	ProcessTrailerPacket(Packet, PacketSize);
    }
    else if (_StreamScanner->IsDataPacket(Packet)) {
	ProcessDataPacket(Packet, PacketSize);
    }
    else if (_StreamScanner->IsNullPacket(Packet)) {
	cout << "Null" << endl;
    }
    else {
	cout << "UNKNOWN PACKET TYPE" << endl;
    }

    cout << endl;
}

void TKinokoKdfCheck::ProcessDataDescriptorPacket(void* DataPacket, long PacketSize)
{
    cout << "Data Descriptor" << endl;

    if (! _IsQuiet) {
	char* DescriptorText = _StreamScanner->DataDescriptorOf(DataPacket);
	istrstream DescriptorTextStream(DescriptorText);

	string Line;
	while (getline(DescriptorTextStream, Line)) {
	    cout << "    " << Line << endl;
	}
    }
}

void TKinokoKdfCheck::ProcessCommandPacket(void* DataPacket, long PacketSize)
{
    int CommandValue = _StreamScanner->CommandValueOf(DataPacket);
    cout << "Command: " << CommandValue << " ";

    switch (CommandValue) {
      case TKinokoDataStreamScanner::Command_RunBegin:
	cout << "(RunBegin)";
	break;
      case TKinokoDataStreamScanner::Command_RunEnd:
	cout << "(RunEnd)";
	break;
      case TKinokoDataStreamScanner::Command_RunSuspend:
	cout << "(RunSuspend)";
	break;
      case TKinokoDataStreamScanner::Command_RunResume:
	cout << "(RunResume)";
	break;
      default:
	cout << "(UNKNOWN)";
	break;
    }
    
    cout << endl;
}

void TKinokoKdfCheck::ProcessTrailerPacket(void* DataPacket, long PacketSize)
{
    int TrailerValue = _StreamScanner->TrailerValueOf(DataPacket);
    cout << "Trailer: " << TrailerValue << " ";

    switch (TrailerValue) {
      case TKinokoDataStreamScanner::Trailer_Event:
	cout << "(Event)";
	break;
      case TKinokoDataStreamScanner::Trailer_Command:
	cout << "(Command)";
	break;
      case TKinokoDataStreamScanner::Trailer_Trap:
	cout << "(Trap)";
	break;
      default:
	cout << "(UNKNOWN)";
	break;
    }
    
    cout << endl;
}

void TKinokoKdfCheck::ProcessDataPacket(void* DataPacket, long PacketSize)
{
    int SectionId = TKinokoDataSectionScanner::SectionIdOf(DataPacket);
    int DataSize = TKinokoDataSectionScanner::DataSizeOf(DataPacket);
    void* DataArea = TKinokoDataSectionScanner::DataAreaOf(DataPacket);

    cout << "Data<" << SectionId << ">: " << DataSize << " bytes";

    if (! _IsQuiet) {
	cout << hex << setfill('0');
	for (int Offset = 0; Offset < DataSize; Offset++) {
	    if (Offset % 16 == 0) {
		cout << endl << "    ";
		cout << setw(4) << (Offset >> 16) << " ";
		cout << setw(4) << (Offset & 0xffff) << ":  ";
	    }
	    else if (Offset % 8 == 0) {
		cout << "  ";
	    } 
	    cout << setw(2) << (int) (((U8bit*) DataArea)[Offset]) << " ";
	}
    }

    cout << dec << endl;
}

void TKinokoKdfCheck::ShowPreamble(void)
{
    cout << hex << setfill('0');
    cout << "% Preamble Version: " << setw(8) << _Storage->DataFilePreambleVersion() << endl;
    cout << "% Header Version: " << setw(8) << _Storage->DataFileHeaderVersion() << endl;
    cout << "% Data Area Version: " << setw(8) << _Storage->DataFileDataAreaVersion() << endl;
    cout << "% Data Format Flags: " << setw(8) << _Storage->DataFileFormatFlags() << endl;
    cout << dec << endl;
}

void TKinokoKdfCheck::ShowHeader(TKinokoStorageHeader& Header)
{
    for (unsigned i = 0; i < Header.NumberOfEntries(); i++) {
	string EntryName = Header.EntryNameAt(i);
	string Value = Header.ValueAt(i);

	cout << "# " << EntryName << ": " << Value << endl;
    }
    cout << endl;
}



int main(int argc, char** argv)
{
    if (argc < 2) {
	cerr << "Usage: " << argv[0];
	cerr << " [Options] DataFileName [DataSourceName]" << endl;
	cerr << "Options:" << endl;
	cerr << "  --quiet  eliminate data dump" << endl;
	cerr << "  --raw    reads in raw-mode" << endl;
	cerr << "  --skip=START_INDEX " << endl;
        return EXIT_FAILURE;
    }

    TMushArgumentList ArgumentList(argc, argv);
    string DataFileName = ArgumentList[0];
    bool IsQuiet = ArgumentList.IsOptionSpecified("--quiet");
    bool IsRaw = ArgumentList.IsOptionSpecified("--raw");

    //... for backward compatibility ...
    IsQuiet |= ArgumentList.IsOptionSpecified("--silent");

    long StartIndex = 0;
    if (ArgumentList.IsOptionSpecified("--skip")) {
	try {
	    StartIndex = ArgumentList.IntOptionValueOf("--skip");
	}
	catch (TSystemCallException &e) {
	    cerr << "ERROR: invalid --skip option" << endl;
	    return EXIT_FAILURE;
	}
    }

    try {
	TKinokoKdfCheck(DataFileName, IsQuiet, IsRaw, StartIndex).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
