/* kdfextract.cc */
/* Created by Enomoto Sanshiro on 4 October 2002. */
/* Last updated by Enomoto Sanshiro on 4 October 2002. */


#include <cstdlib>
#include <cstring>
#include <iostream>
#include <strstream>
#include <string>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataDistributor.hh"
#include "KinokoDataProcessor.hh"

using namespace std;


class TKinokoKdfExtract {
  public:
    TKinokoKdfExtract(const string& InputFileName);
    virtual ~TKinokoKdfExtract();
    virtual void Start(void) throw(TKinokoException);
  protected:
    string _InputFileName;
    TKinokoStorage* _InputStorage;
    TKinokoStorage* _OutputStorage;
    TKinokoStorageHeader* _InputStorageHeader;
    TKinokoInputStream* _InputStream;
    TKinokoOutputStream* _OutputStream;
    map<int, int> _DescriptorCounts;
};



TKinokoKdfExtract::TKinokoKdfExtract(const string& InputFileName)
{
    _InputFileName = InputFileName;
    _InputStorage = 0;

    _OutputStream = 0;
    _OutputStorage = 0;
}

TKinokoKdfExtract::~TKinokoKdfExtract()
{
    delete _InputStorageHeader;
    delete _InputStorage;
    delete _OutputStorage;
}

void TKinokoKdfExtract::Start(void) throw(TKinokoException)
{
    _InputStorage = new TKinokoKdfStorage(_InputFileName);
    _InputStorageHeader = new TKinokoStorageHeader();

    try {
        _InputStorage->ReadHeader(*_InputStorageHeader);
    }
    catch (TKinokoException &e) {
        /* version number mismatch ? */
	cout << "WARNING: " << e << endl << endl;
    }

    _InputStream = _InputStorage->GetInputStream();
    TMushFileAttribute InputFileAttr(_InputFileName);

    string OutputFileName = InputFileAttr.FileRootName() + "-RunBegin." + InputFileAttr.Extension();
    cout << "making " << OutputFileName << "..." << endl;
    _OutputStorage = new TKinokoKdfStorage(OutputFileName);
    _OutputStorage->WriteHeader(*_InputStorageHeader);
    _OutputStream = _OutputStorage->GetOutputStream();

    void* Packet;
    int PacketSize;
    while ((PacketSize = _InputStream->NextEntry(Packet)) > 0) {
	//... correct byte order ...
	long DataSourceId = TKinokoDataStreamScanner::DataSourceIdOf(Packet);

	if (TKinokoDataStreamScanner::IsDataDescriptorPacket(Packet)) {
	    _OutputStream->Write(Packet, PacketSize);
	    _DescriptorCounts[DataSourceId] += 1;
	}
	else if (TKinokoDataStreamScanner::IsCommandPacket(Packet)) {
	    int CommandValue = TKinokoDataStreamScanner::CommandValueOf(Packet);
	    if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
               if (_DescriptorCounts[DataSourceId] == 1) {
	            _OutputStream->Write(Packet, PacketSize);
                    _DescriptorCounts[DataSourceId] -= 1;
	       }
	    }
	}

	_InputStream->Flush(Packet);
    }
}


int main(int argc, char** argv)
{
    if (argc < 2) {
        cerr << argv[0] << ": extracts DataDescriptor and RunBegin packets from a kdf file" << endl;
	cerr << "Usage: " << argv[0] << " SoureFileName" << endl;
        return EXIT_FAILURE;
    }

    string InputFileName = argv[1];

    try {
	TKinokoKdfExtract(InputFileName).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
