/* kdfseparate.cc */
/* Created by Enomoto Sanshiro on 26 April 2002. */
/* Last updated by Enomoto Sanshiro on 26 April 2002. */


#include <cstdlib>
#include <cstring>
#include <iostream>
#include <strstream>
#include <string>
#include "MushFile.hh"
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "KinokoKdfStorage.hh"
#include "KinokoDataDistributor.hh"
#include "KinokoDataProcessor.hh"

using namespace std;


class TKinokoKdfSeparate {
  public:
    TKinokoKdfSeparate(const string& InputFileName);
    virtual ~TKinokoKdfSeparate();
    virtual void Start(void) throw(TKinokoException);
  protected:
    string _InputFileName;
    TKinokoStorage* _InputStorage;
    TKinokoStorage* _OutputStorage;
    TKinokoStorageHeader* _InputStorageHeader;
    TKinokoInputStream* _InputStream;
    TKinokoOutputStream* _OutputStream;
    vector<pair<char*, int> > _DescriptorPacketList;
    int _NumberOfDataSources;
    int _RunCount;
};



TKinokoKdfSeparate::TKinokoKdfSeparate(const string& InputFileName)
{
    _InputFileName = InputFileName;
    _InputStorage = 0;

    _OutputStream = 0;
    _OutputStorage = 0;

    _NumberOfDataSources = 0;
    _RunCount = 0;
}

TKinokoKdfSeparate::~TKinokoKdfSeparate()
{
    for (unsigned i = 0; i < _DescriptorPacketList.size(); i++) {
	delete[] _DescriptorPacketList[i].first;
    }

    delete _InputStorageHeader;
    delete _InputStorage;
    delete _OutputStorage;
}

void TKinokoKdfSeparate::Start(void) throw(TKinokoException)
{
    _InputStorage = new TKinokoKdfStorage(_InputFileName);
    _InputStorageHeader = new TKinokoStorageHeader();

    try {
        _InputStorage->ReadHeader(*_InputStorageHeader);
    }
    catch (TKinokoException &e) {
        /* version number mismatch ? */
	cout << "WARNING: " << e << endl << endl;
    }

    _InputStream = _InputStorage->GetInputStream();

    void* Packet;
    int PacketSize;
    while ((PacketSize = _InputStream->NextEntry(Packet)) > 0) {
	//... correct byte order ...

	if (TKinokoDataStreamScanner::IsDataDescriptorPacket(Packet)) {
	    char* DescriptorPacket = new char[PacketSize];
	    memcpy(DescriptorPacket, Packet, PacketSize);
	    _DescriptorPacketList.push_back(
		make_pair(DescriptorPacket, PacketSize)
	    );
	    continue;
	}

	if (_OutputStream == 0) {
	    char RunCountString[32];
	    ostrstream RunCountStream(RunCountString, sizeof(RunCountString));
	    RunCountStream << ++_RunCount << ends;

	    TMushFileAttribute InputFileAttr(_InputFileName);
	    string OutputFileName = InputFileAttr.FileRootName() + "-" + RunCountString + "." + InputFileAttr.Extension();

	    cout << "making " << OutputFileName << "..." << endl;

	    _OutputStorage = new TKinokoKdfStorage(OutputFileName);
	    _OutputStorage->WriteHeader(*_InputStorageHeader);
	    _OutputStream = _OutputStorage->GetOutputStream();

	    for (unsigned i = 0; i < _DescriptorPacketList.size(); i++) {
		_OutputStream->Write(
		    _DescriptorPacketList[i].first,
		    _DescriptorPacketList[i].second
		);
	    }
	}
	
	_OutputStream->Write(Packet, PacketSize);
	_InputStream->Flush(Packet);

	if (TKinokoDataStreamScanner::IsCommandPacket(Packet)) {
	    int CommandValue = TKinokoDataStreamScanner::CommandValueOf(Packet);
	    if (CommandValue == TKinokoDataStreamScanner::Command_RunBegin) {
		_NumberOfDataSources++;
	    }
	    if (CommandValue == TKinokoDataStreamScanner::Command_RunEnd) {
		_NumberOfDataSources--;
	    }
	}

	if (_NumberOfDataSources <= 0) {
	    delete _OutputStorage;
	    _OutputStorage = 0;
	    _OutputStream = 0;
	}
    }
}


int main(int argc, char** argv)
{
    if (argc < 2) {
	cerr << "Usage: " << argv[0] << " SoureFileName" << endl;
        return EXIT_FAILURE;
    }

    string InputFileName = argv[1];

    try {
	TKinokoKdfSeparate(InputFileName).Start();
    }
    catch (TKinokoException& e) {
	cerr << "ERROR: " << e << endl;
        return EXIT_FAILURE;
    }

    return 0;
}
