/* tinykinoko.cc */
/* Created by Enomoto Sanshiro on 23 April 1999. */
/* Last updated by Enomoto Sanshiro on 8 July 2001. */


#include <iostream>
#include <strstream>
#include <string>
#include "MushFileSystem.hh"
#include "MushArgumentList.hh"
#include "MushSignal.hh"
#include "MushDecoratedFile.hh"
#include "MushProcess.hh"
#include "MushMisc.hh"
#include "MushJunks.hh"
#include "KinokoStream.hh"
#include "KinokoComponentPlatform.hh"
#include "KinokoDaqFrontend.hh"
#include "KinokoKdfStorage.hh"

using namespace std;


static const char* DefaultDataSourceName = "";


class TTinyKinoko {
  public:
    TTinyKinoko(const string& ScriptFileName, const string& OutputFileName, long NumberOfEvents, TMushArgumentList& ArgumentList);
    virtual ~TTinyKinoko();
    virtual void Start(void) throw(TKinokoException);
  protected:
    virtual void Construct(void) throw(TKinokoException);
    virtual void Destruct(void) throw(TKinokoException);
    virtual void OnStart(void) throw(TKinokoException);
    virtual void OnStop(void) throw(TKinokoException);
    virtual int ProcessData(void) throw(TKinokoException);
  protected:
    string _ScriptFileName;
    string _DataSourceName;
    string _OutputFileName;
    long _NumberOfEvents;
    string _RegistryFileName;
    TKcomRegistry* _KcomRegistry;
    TKinokoRegistry* _Registry;
    TKinokoDaqFrontend* _DaqFrontend;
    TKinokoEventEmitter* _EventEmitter;
    TKinokoOutputStream* _OutputStream;
    TKinokoStorage* _Storage;
    long _EventCount;
    bool _IsStopRequested;
    bool _IsDataCompressionEnabled;
    bool _IsIndexEnabled;
    bool _IsQuiet;
};



TTinyKinoko::TTinyKinoko(const string& ScriptFileName, const string& OutputFileName, long NumberOfEvents, TMushArgumentList& ArgumentList)
{
    _ScriptFileName = ScriptFileName;
    _OutputFileName = OutputFileName;
    _NumberOfEvents = NumberOfEvents;

    _IsQuiet = ArgumentList.IsOptionSpecified("--quiet", 'q');

    if (ArgumentList.IsOptionSpecified("--enable-compress", 'c')) {
	_IsDataCompressionEnabled = true;
    }
    else {
	_IsDataCompressionEnabled = false;
    }

    if (ArgumentList.IsOptionSpecified("--enable-index", 'i')) {
	_IsIndexEnabled = true;
    }
    else {
	_IsIndexEnabled = false;
    }

    if (ArgumentList.IsOptionSpecified("--datasource")) {
	_DataSourceName = ArgumentList.OptionValueOf("--datasource");
    }
    else {
	_DataSourceName = DefaultDataSourceName;
    }

    if (ArgumentList.IsOptionSpecified("--registry")) {
	_RegistryFileName = ArgumentList.OptionValueOf("--registry");
    }

    _DaqFrontend = 0;
    _EventEmitter = 0;
    _Registry = 0;
    _KcomRegistry = 0;

    _Storage = 0;
}

TTinyKinoko::~TTinyKinoko()
{
    delete _Storage;

    delete _EventEmitter;
    delete _Registry;
    delete _KcomRegistry;

    delete _DaqFrontend;
}

void TTinyKinoko::Start(void) throw(TKinokoException)
{
    TMushSignalHandler SignalHandler;
    TMushSignalCounter SignalCounter;
    SignalHandler.RegisterClient(SIGINT, &SignalCounter);
    SignalHandler.RegisterClient(SIGTERM, &SignalCounter);
    SignalHandler.StartHandling();

    TMushProgressBar* ProgressBar = 0;
    if (! _IsQuiet) {
	cerr << "Press Ctrl-c to stop." << endl;
	int ProgressStep = 0;
	if (_NumberOfEvents > 0) {
	    ProgressStep = _NumberOfEvents / 10;
	}
	ProgressBar = new TMushProgressBar(cerr, ProgressStep);
	ProgressBar->Start();
    }


    Construct();
    OnStart();
    while (SignalCounter.SignalCount() == 0) {
	if (
	    _IsStopRequested || 
	    ((_NumberOfEvents > 0) && (_EventCount >= _NumberOfEvents))
	){
	    break;
        }
	if ((ProcessData() > 0) && ! _IsQuiet) {
	    ProgressBar->Next();
	}
    }
    OnStop();
    Destruct();

    if (! _IsQuiet) {
	ProgressBar->Clear();
	delete ProgressBar;

	if (SignalCounter.SignalCount() > 0) {
	    cerr << "INTERRUPTED" << endl;
	}
	cerr << _EventCount << " events have been taken." << endl;
    }
}

void TTinyKinoko::Construct(void) throw(TKinokoException)
{
    try {
	if (_RegistryFileName.empty()) {
	    _Registry = new TKinokoNullRegistry();
	}
	else {
	    if (TMushFileAttribute(_RegistryFileName).IsReadable()) {
		TKcomRegistryRepository Repository(_RegistryFileName);
		_KcomRegistry = Repository.Load();
	    }
	    else {
		_KcomRegistry = new TKcomRegistry();
	    }
	    _Registry = new TKinokoComponentRegistry(_KcomRegistry);
	}
    }
    catch (TKcomException &e) {
	throw TKinokoException(e.Message());
    }

    // We should use 'unique' DataSourceId for each datasource
    // because data taken by tinykinoko independently might be merged.
    // DataSourceId must be greater than 1023 and less than 0x10000.
    //... Is it better to use a hash value of the DataSourceName string?
    int DataSourceId = (TMushProcess().ProcessId() & 0x7fff) + 1024;

    string Date = TMushDateTime().AsString("%Y-%m-%d %H:%M:%S %Z");

    TKinokoStorageHeader StorageHeader;
    StorageHeader.AddEntry("Creator", "tinykinoko");
    StorageHeader.AddEntry("DateTime", Date);
    StorageHeader.AddEntry("UserName", TMushUser().Name());
    StorageHeader.AddEntry("Host", TMushLocalHost::HostName());
    StorageHeader.AddEntry("Directory", TMushFileSystem::CurrentDirectory());
    StorageHeader.AddEntry("ScriptFile", _ScriptFileName);
    
    _Storage = new TKinokoKdfStorage(_OutputFileName);
    _Storage->AllowOverWrite();

    if (_IsDataCompressionEnabled) {
	_Storage->EnableDataCompression();
    }
    if (_IsIndexEnabled) {
	_Storage->EnableIndex();
    }

    _Storage->WriteHeader(StorageHeader);
    _OutputStream = _Storage->GetOutputStream();
    
    _EventEmitter = new TKinokoEventEmitter();
    _DaqFrontend = new TKinokoDaqFrontend("tinykinoko");
    _DaqFrontend->Construct(
	_ScriptFileName, _DataSourceName, DataSourceId, 
	_OutputStream, _EventEmitter, _Registry
    );

    _IsStopRequested = false;
}

void TTinyKinoko::Destruct(void) throw(TKinokoException)
{
    try {
	if (! _RegistryFileName.empty()) {
	    TKcomRegistryRepository Repository(_RegistryFileName);
	    Repository.Save(_KcomRegistry);
	}
    }
    catch (TKcomException &e) {
	throw TKinokoException(e.Message());
    }

    delete _EventEmitter;
    delete _Registry;
    delete _KcomRegistry;
    delete _Storage;

    _EventEmitter = 0;
    _Registry = 0;
    _KcomRegistry = 0;
    _Storage = 0;
}

void TTinyKinoko::OnStart(void) throw(TKinokoException)
{
    _EventCount = 0;
    _DaqFrontend->OnStart();
}

void TTinyKinoko::OnStop(void) throw(TKinokoException)
{
    _DaqFrontend->OnStop();
}

int TTinyKinoko::ProcessData(void) throw(TKinokoException)
{
    if (! _DaqFrontend->DoTransaction()) {
	_IsStopRequested = _DaqFrontend->IsStopRequested();
	return 0;
    }
    else {
	_EventCount++;
	return 1;
    }
}



int main(int argc, char **argv)
{
    TMushSignalHandler SignalHandler;
    SignalHandler.StartDefaultAction(SIGSEGV);
    SignalHandler.StartDefaultAction(SIGPIPE);

    TMushArgumentList ArgumentList(argc, argv);

    if (ArgumentList.NumberOfParameters() < 2) {
	cerr << "Usage: " << argv[0];
	cerr << " [Options] ScriptFileName DataFileName [NumberOfEvents]" << endl;
	cerr << "Options:" << endl;
	cerr << "  -f, --force                     overwrite data file if exists" << endl;
	cerr << "  -q, --quiet                     do not display messages" << endl;
	cerr << "  -i, --enable-index              make index file" << endl;
	cerr << "  -c, --enable-compress           enable data compression" << endl;
	cerr << "  --datasource=DataSourceName     specifies datasource name to use" << endl;
	cerr << "  --registry=RegistryFileName     specifies registry file" << endl;
	return EXIT_FAILURE;
    }
    
    string ScriptFileName = ArgumentList[0];
    string OutputFileName = ArgumentList[1];
    long NumberOfEvents = -1;
    if (ArgumentList.NumberOfParameters() > 2) {
	try {
	    NumberOfEvents = ArgumentList.IntParameterOf(2);
	}
	catch (TSystemCallException &e) {
	    cerr << "ERROR: NumberOfEvents must be an integer" << endl;
	    return EXIT_FAILURE;
	}
    }

    if (TMushFileAttribute(OutputFileName).IsReadable()) {
	if (! ArgumentList.IsOptionSpecified("--force", 'f')) {
	    cerr << "ERROR: file exists: " << OutputFileName << endl;
	    cerr << "use -f to overwrite" << endl;
	    return EXIT_FAILURE;
	}
    }

    try {
	TTinyKinoko(ScriptFileName, OutputFileName, NumberOfEvents, ArgumentList).Start();
    }
    catch (TKinokoException &e) {
	cerr << "ERROR: " << argv[0] << ": " << e << endl;
	return EXIT_FAILURE;
    }

    return 0;
}
