/* KinokoCanvas.cc */
/* Created by Enomoto Sanshiro on 9 July 2000. */
/* Last updated by Enomoto Sanshiro on 27 March 2002. */


#include <iostream>
#include <strstream>
#include <vector>
#include <map>
#include "KinokoShellConnector.hh"
#include "KinokoCanvas.hh"
#include "KinokoCanvasObject.hh"
#include "KinokoCanvasPictureObject.hh"
#include "KinokoCanvasPlotObject.hh"
#include "KinokoCanvas3dPlotObject.hh"
#include "KinokoCanvasTextBoxObject.hh"
#include "KinokoCanvasImageArea.hh"
#include "KinokoCanvasImageAreaEps.hh"

using namespace std;


TKinokoCanvas::TKinokoCanvas(TKinokoShellConnector* ShellConnector)
: TKinokoShell(ShellConnector)
{
    _ImageArea = 0;
    _EpsImageArea = 0;

    _IsAutoRedrawEnabled = true;
}

TKinokoCanvas::~TKinokoCanvas()
{
    delete _EpsImageArea;
}

void TKinokoCanvas::Initialize(TKinokoCanvasImageArea* ImageArea, bool IsEpsEnabled)
{
    if (IsEpsEnabled) {
	_EpsImageArea = new TKinokoCanvasImageAreaEps(
	    ImageArea->Width(), ImageArea->Height(), ImageArea
	);
	_ImageArea = _EpsImageArea;
    }
    else {
	_ImageArea = ImageArea;
    }

    RegisterObjectPrototype(
	"picture", new TKinokoCanvasPictureObject(this, _ImageArea)
    );
    RegisterObjectPrototype(
	"textbox", new TKinokoCanvasTextBoxObject(this, _ImageArea)
    );
    RegisterObjectPrototype(
	"plot", new TKinokoCanvasPlotObject(this, _ImageArea)
    );
    RegisterObjectPrototype(
	"plot3dnoscale", new TKinokoCanvas3dPlotObject(this, _ImageArea)
    );
    RegisterObjectPrototype(
	"plot3dmonochrome", new TKinokoCanvasZScaled3dPlotObject(this, _ImageArea, true)
    );
    RegisterObjectPrototype(
	"plot3d", new TKinokoCanvasZScaled3dPlotObject(this, _ImageArea)
    );
    RegisterObjectPrototype(
	"mapplot", new TKinokoCanvasMapPlotObject(this, _ImageArea)
    );
}

void TKinokoCanvas::AddCanvasObject(TKinokoCanvasObject* CanvasObject)
{
    _CanvasObjectList.push_back(CanvasObject);
}

void TKinokoCanvas::OnClick(int ButtonNumber, int X, int Y)
{
    if (ButtonNumber == 3) {
	for (int i = _CanvasObjectList.size() - 1; i >= 0; i--) {
	    if (_CanvasObjectList[i]->Includes(X, Y)) {
		OpenContextMenu(_CanvasObjectList[i]);
		break;
	    }
	}
    }
}

int TKinokoCanvas::Clear(void)
{
    _ImageArea->Clear();
    return 1;
}

int TKinokoCanvas::Redraw(void)
{
    _ImageArea->Redraw();
    return 1;
}

int TKinokoCanvas::SetTitle(const string& Title)
{
    return 1;
}

int TKinokoCanvas::SaveImageTo(const string& FileName)
{
    return _ImageArea->SaveImageTo(FileName);
}

int TKinokoCanvas::BeginEps(const std::string& FileName)
{
    if (_EpsImageArea) {
	return _EpsImageArea->Open(FileName);
    }

    return 1;
}

int TKinokoCanvas::EndEps(void)
{
    if (_EpsImageArea) {
	return _EpsImageArea->Close();
    }

    return 1;
}

int TKinokoCanvas::ProcessCommand(std::istream& InputStream)
{
    int Result = TKinokoShell::ProcessCommand(InputStream);

    if ((Result > 0) && _IsAutoRedrawEnabled) {
	Redraw();
    }

    return Result;
}

int TKinokoCanvas::ProcessSystemCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == ".set") {
	Result = ProcessSetCommand(InputStream);
    }
    else if (Command == ".redraw") {
	Result = ProcessRedrawCommand(InputStream);
    }
    else if (Command == ".clear") {
	Result = ProcessClearCommand(InputStream);
    }
    else if (Command == ".saveImage") {
	Result = ProcessSaveImageCommand(InputStream);
    }
    else if (Command == ".beginEps") {
	Result = ProcessBeginEpsCommand(InputStream);
    }
    else if (Command == ".endEps") {
	Result = ProcessEndEpsCommand(InputStream);
    }
    else {
	Result = TKinokoShell::ProcessSystemCommand(Command, InputStream);
    }
    
    return Result;
}

int TKinokoCanvas::ProcessSetCommand(istream& InputStream)
{
    int Result = 0;

    string Name;
    if (InputStream >> Name) {
	if (Name == "redraw") {
	    string Value;
	    if (InputStream >> Value) {
		if (Value == "auto") {
		    _IsAutoRedrawEnabled = true;
		    Result = 1;
		}
		else if (Value == "manual") {
		    _IsAutoRedrawEnabled = false;
		    Result = 1;
		}
	    }
	}
	else if (Name == "title") {
	    char Title[128];
	    InputStream >> ws;
	    if (InputStream.getline(Title, sizeof(Title), '\n')) {
		Result = SetTitle(Title);
	    }
	}
    }

    return Result;
}

int TKinokoCanvas::ProcessRedrawCommand(istream& InputStream)
{
    return Redraw();
}

int TKinokoCanvas::ProcessClearCommand(std::istream& InputStream)
{
    return Clear();
}

int TKinokoCanvas::ProcessSaveImageCommand(istream& InputStream)
{
    int Result = 0;
    string FileName;
    if (InputStream >> FileName) {
	Result = _ImageArea->SaveImageTo(FileName);
    }

    return Result;
}

int TKinokoCanvas::ProcessBeginEpsCommand(istream& InputStream)
{
    int Result = 0;
    string FileName;
    if (InputStream >> FileName) {
	Result = BeginEps(FileName);
    }

    return Result;
}

int TKinokoCanvas::ProcessEndEpsCommand(istream& InputStream)
{
    return EndEps();
}
