/* KinokoCanvas3dPlotObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 4 November 2002. */


#include <strstream>
#include <algorithm>
#include <cmath>
#include "KinokoCanvasPlotObject.hh"
#include "KinokoCanvas3dPlotObject.hh"

using namespace std;


#define clip(x, lower, upper) (min(max((lower), (x)), (upper)))


TKinokoCanvas3dPlotObject::TKinokoCanvas3dPlotObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea, bool IsMonochrome)
: TKinokoCanvasPlotObject(Canvas, ImageArea), TKinokoCanvasFramedObject(Canvas, ImageArea)
{
    _ZMin = 0;
    _ZMax = 1.0;
    _ZWidth = (_ZMax - _ZMin);

    _LogZMin = (_ZMin > 0) ? log10(_ZMin) : log10(0.5);
    _LogZMax = (_ZMax > 0) ? log10(_ZMax) : 0;
    _LogZWidth = _LogZMax - _LogZMin;

    _IsMonochrome = IsMonochrome;

    _IsZScaleLog = false;
    SetZCoordinate(0, 1);

    if (IsMonochrome) {
	_ColorScale = new TKinokoCanvasGrayColorScale(_ImageArea);
    }
    else {
	_ColorScale = new TKinokoCanvasWhiteBackgroundRainbowColorScale(_ImageArea);
    }
}

TKinokoCanvas3dPlotObject::~TKinokoCanvas3dPlotObject()
{
    delete _ColorScale;
}

TKinokoShellObject* TKinokoCanvas3dPlotObject::Clone(void)
{
    return new TKinokoCanvas3dPlotObject(_Canvas, _ImageArea, _IsMonochrome);
}

vector<string> TKinokoCanvas3dPlotObject::ActionList(void)
{
    vector<string> ActionList = TKinokoCanvasPlotObject::ActionList();

    ActionList.push_back("setZScaleLinear");
    ActionList.push_back("setZScaleLog");

    return ActionList;
}

int TKinokoCanvas3dPlotObject::ProcessAction(const string& ActionName)
{
    int Result = 1;
    if (ActionName == "setZScaleLinear") {
	SetZScaleLinear();
    }
    else if (ActionName == "setZScaleLog") {
	SetZScaleLog();
    }
    else {
	Result = TKinokoCanvasPlotObject::ProcessAction(ActionName);
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::SetZCoordinate(float ZMin, float ZWidth)
{
    _ZMinOrg = ZMin;
    _ZWidthOrg = ZWidth;

    if (_IsZScaleLog) {
	if (ZMin <= 0) {
	    float ZMax = ZMin + ZWidth;
	    float ZMaxForLog = (ZMax <= 0) ? 1.0 : ZMax;
	    float ZMinForLog = min(ZMaxForLog / 100.0, 0.5);
	    ZMin = ZMinForLog;
	    ZWidth = ZMaxForLog - ZMinForLog;
	}

	_LogZMin = log10(ZMin);
	_LogZMax = log10(ZMin + ZWidth);
	_LogZWidth = _LogZMax - _LogZMin;
    }

    _ZMin = ZMin;
    _ZWidth = ZWidth;
    _ZMax = ZMin + ZWidth;

    return 1;
}

int TKinokoCanvas3dPlotObject::SetZScaleLinear(void)
{
    if (_IsZScaleLog) {
	_IsZScaleLog = false;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
    }

    return 1;
}

int TKinokoCanvas3dPlotObject::SetZScaleLog(void)
{
    if (! _IsZScaleLog) {
	_IsZScaleLog = true;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
    }

    return 1;
}

int TKinokoCanvas3dPlotObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "zscale") {
	if (Value == "log") {
	    SetZScaleLog();
	}
	else {
	    SetZScaleLinear();
	}
    }
    else {
	Result = TKinokoCanvasPlotObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "scatterhist") {
	Result = Process2dHistCommand(InputStream, Type_Scatter);
    }
    else if (Command == "boxhist") {
	Result = Process2dHistCommand(InputStream, Type_Box);
    }
    else if (Command == "colorhist") {
	Result = Process2dHistCommand(InputStream, Type_Color);
    }
    else if (Command == "contour") {
	Result = ProcessContourCommand(InputStream);
    }
    else {
	Result = TKinokoCanvasPlotObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = TKinokoCanvasPlotObject::ProcessFrameCommand(InputStream);
    if (Result == 0) {
	return Result;
    }

    Result = 0;

    float ZMin, ZMax;
    if (InputStream >> ZMin >> ZMax) {
	if (ZMax > ZMin) {
	    Result = SetZCoordinate(ZMin, ZMax - ZMin);
	}
    }

    return Result;
}

int TKinokoCanvas3dPlotObject::Process2dHistCommand(istream& InputStream, int Type)
{
    // syntax: 
    //   'hist' x_min x_step x_bins y_min y_step z00 z01 ...

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    float xmin, xstep, ymin, ystep;
    int xbins;
    if (! (InputStream >> xmin >> xstep >> xbins >> ymin >> ystep)) {
	return 0;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);

    float cx0, cy0, cx1, cy1;
    float x = xmin, y = ymin, z;
    int DataCount = 0;
    while (InputStream >> z) {
	if (DataCount++ % xbins == 0) {
	    x = xmin;
	    cx0 = CanvasXOf(x);
	    cy0 = CanvasYOf(y);

	    y += ystep;
	    cy1 = CanvasYOf(y);
	}
	x += xstep;
	cx1 = CanvasXOf(x);

	if ((x >= _XMin) && (x <= _XMax) && (y >= _YMin) && (y <= _YMax)) {
	    float NormalizedZ;
	    if (_IsZScaleLog) {
		if (z > 0) {
		    float LogZ = clip((float) log10(z), _LogZMin, _LogZMax);
		    NormalizedZ = (LogZ - _LogZMin) / _LogZWidth;
		}
		else {
		    NormalizedZ = 0;
		}
	    }
	    else {
		z = clip(z, _ZMin, _ZMax);
		NormalizedZ = (z - _ZMin) / _ZWidth;
	    }

	    if (Type == Type_Scatter) {
		float Area = (cx1 - cx0 + 1) * (cy0 - cy1 + 1);
		int NumberOfPoints = (int) (Area * NormalizedZ);
		for (int i = 0; i < NumberOfPoints; i++) {
		    _ImageArea->DrawPoint(
			cx0  + ((cx1 - cx0) * drand48()),
			cy0  + ((cy1 - cy0) * drand48())
		    );
		}
	    }
	    else if (Type == Type_Box) {
		float Ratio = sqrt(fabs(NormalizedZ));
		if (Ratio > 0) {
		    float HalfWidth = Ratio * (cx1 - cx0) / 2;
		    float HalfHeight = Ratio * (cy0 - cy1) / 2;
		    
		    float cx = (cx0 + cx1) / 2;
		    float cy = (cy0 + cy1) / 2;
		    _ImageArea->DrawRect(
			(cx - HalfWidth), (cy - HalfHeight),
			(cx + HalfWidth), (cy + HalfHeight) 
		    ); 
		}
	    }
	    else if (Type == Type_Color) {
		_ImageArea->SetColor(_ColorScale->ColorIndexOf(NormalizedZ));
		_ImageArea->DrawRectFill(cx0, cy1, cx1, cy0);
	    }
	}

	cx0 = cx1;
    }

    _ImageArea->SetColor(OldColorIndex);
    TKinokoCanvasPlotObject::DrawFrame();

    return Result = 1;
}

int TKinokoCanvas3dPlotObject::ProcessContourCommand(istream& InputStream)
{
    // syntax: 
    //   'contour' thresh x_min x_step x_bins y_min y_step y_bins z00 z01 ...

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    float Threshold;
#if 0
    if (! (InputStream >> Threshold)) {
	return 0;
    }
#else
    //... for test
    Threshold = (_ZMin + _ZMax) / 4.0;
#endif

    float xmin, xstep, ymin, ystep;
    int xbins, ybins;
    if (! (InputStream >> xmin >> xstep >> xbins >> ymin >> ystep >> ybins)) {
	return 0;
    }

    float LeftZ;
    float* LowerZ = new float[xbins];

    float X = xmin, Y = ymin, Z;
    int DataCount = 0, XBinCount = 0;

    float cx0, cy0, cx1, cy1;
    bool DrawsLeft, DrawsLower;
    while (InputStream >> Z) {
	if (XBinCount == 0) {
	    DrawsLeft = false;
	}
	else {
	    DrawsLeft = (
		((LeftZ < Threshold) && (Z >= Threshold)) ||
		((LeftZ >= Threshold) && (Z < Threshold))
	    );
	}
	LeftZ = Z;

	if (DataCount < xbins) {
	    DrawsLower = false;
	}
	else {
	    DrawsLower = (
		((LowerZ[XBinCount] < Threshold) && (Z >= Threshold)) ||
		((LowerZ[XBinCount] >= Threshold) && (Z < Threshold))
	    );
	}
	LowerZ[XBinCount] = Z;

	if (DrawsLeft || DrawsLower) {
	    if ((X >= _XMin) && (X <= _XMax) && (Y >= _YMin) && (Y <= _YMax)) {
		cx0 = CanvasXOf(X);
		cx1 = CanvasXOf(X + xstep);
		cy0 = CanvasYOf(Y);
		cy1 = CanvasYOf(Y + ystep);
		if (DrawsLeft) {
		    _ImageArea->DrawLine(cx0, cy0, cx0, cy1);
		}
		if (DrawsLower) {
		    _ImageArea->DrawLine(cx0, cy0, cx1, cy0);
		}
	    }
	}

	if (++DataCount % xbins == 0) {
	    X = xmin;
	    Y += ystep;
	    XBinCount = 0;
	}
	else {
	    X += xstep;
	    XBinCount++;
	}
    }

    delete[] LowerZ;
    TKinokoCanvasPlotObject::DrawFrame();

    return Result = 1;
}



TKinokoCanvasZScaled3dPlotObject::TKinokoCanvasZScaled3dPlotObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea, bool IsMonochrome)
: TKinokoCanvas3dPlotObject(Canvas, ImageArea, IsMonochrome), 
  TKinokoCanvasColorScaledObject(
      Canvas, ImageArea, 
      (IsMonochrome ? (TKinokoCanvasColorScale*) new TKinokoCanvasGrayColorScale(_ImageArea) : (TKinokoCanvasColorScale*) new TKinokoCanvasWhiteBackgroundRainbowColorScale(_ImageArea))
  ), 
  TKinokoCanvasFramedObject(Canvas, ImageArea)
{
}

TKinokoCanvasZScaled3dPlotObject::~TKinokoCanvasZScaled3dPlotObject()
{
}

TKinokoShellObject* TKinokoCanvasZScaled3dPlotObject::Clone(void)
{
    return new TKinokoCanvasZScaled3dPlotObject(_Canvas, _ImageArea, _IsMonochrome);
}

int TKinokoCanvasZScaled3dPlotObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = TKinokoCanvas3dPlotObject::ProcessFrameCommand(InputStream);
    if (Result) {
	float ZMin = TKinokoCanvas3dPlotObject::_ZMin;
	float ZWidth = TKinokoCanvas3dPlotObject::_ZWidth;
	TKinokoCanvasColorScaledObject::SetZCoordinate(ZMin, ZWidth);
    }

    return Result;
}

int TKinokoCanvasZScaled3dPlotObject::SetZScaleLinear(void)
{
    TKinokoCanvas3dPlotObject::SetZScaleLinear();
    TKinokoCanvasColorScaledObject::SetZScaleLinear();

    return 1;
}

int TKinokoCanvasZScaled3dPlotObject::SetZScaleLog(void)
{
    TKinokoCanvas3dPlotObject::SetZScaleLog();
    TKinokoCanvasColorScaledObject::SetZScaleLog();

    return 1;
}

int TKinokoCanvasZScaled3dPlotObject::DrawFrame(void)
{
    int Result = TKinokoCanvas3dPlotObject::DrawFrame();
    if (Result > 0) {
	TKinokoCanvasColorScaledObject::DrawFrame();
    }

    return Result;
}
