/* KinokoCanvasColorScale.cc */
/* Created by Enomoto Sanshiro on 14 January 2001. */
/* Last updated by Enomoto Sanshiro on 14 January 2001. */


#include <cmath>
#include "KinokoCanvasImageArea.hh"
#include "KinokoCanvasColorScale.hh"

using namespace std;



TKinokoCanvasColorScale::TKinokoCanvasColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
{
    _ImageArea = ImageArea;
    _NumberOfColors = NumberOfColors;
    _ColorTable = 0;
}

TKinokoCanvasColorScale::~TKinokoCanvasColorScale()
{
    delete[] _ColorTable;
}

int TKinokoCanvasColorScale::ColorIndexOf(float Value)
{
    if (_ColorTable == 0) {
	_ColorTable = new int[_NumberOfColors];
	CreateColorTable();
    }
    
    int ScaleIndex;
    if (Value < 0) {
	ScaleIndex = 0;
    }
    else if (Value >= 1) {
	ScaleIndex = _NumberOfColors - 1;
    }
    else {
	ScaleIndex = (int) (Value * _NumberOfColors);
    }

    return _ColorTable[ScaleIndex];
}



TKinokoCanvasGrayColorScale::TKinokoCanvasGrayColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
: TKinokoCanvasColorScale(ImageArea, NumberOfColors)
{
}

TKinokoCanvasGrayColorScale::~TKinokoCanvasGrayColorScale()
{
}

void TKinokoCanvasGrayColorScale::CreateColorTable(void)
{
    for (int i = 0; i < _NumberOfColors; i++) {
	float v = 1.0 - (float) i / (_NumberOfColors - 1);
	float r = v;
	float g = v;
	float b = v;

	int Color = _ImageArea->AllocateColorRgb(r, g, b);
	_ColorTable[i] = Color;
    }
}



TKinokoCanvasRainbowColorScale::TKinokoCanvasRainbowColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
: TKinokoCanvasColorScale(ImageArea, NumberOfColors)
{
}

TKinokoCanvasRainbowColorScale::~TKinokoCanvasRainbowColorScale()
{
}

void TKinokoCanvasRainbowColorScale::CreateColorTable(void)
{
    static const float HueCutOff = 0.7;
    static const float Saturation = 1.0;
    static const float Brightness = 0.5;
    
    for (int i = 0; i < _NumberOfColors; i++) {
	float Value = (float) i / (_NumberOfColors - 1);
	float h = HueCutOff * (1.0 - Value);
	float s = Saturation;
	float v = Brightness;

	float r, g, b;
	HSV2RGB(h, s, v, r, g, b);

	int Color = _ImageArea->AllocateColorRgb(r, g, b);
	_ColorTable[i] = Color;
    }
}

static float HSV2RGB_aux(float rm1, float rm2, float h);

void TKinokoCanvasRainbowColorScale::HSV2RGB(float h, float s, float v, float& r, float& g, float& b)
{
    // see HIGZ routine IGHTOR

    // normalization
    h = ((h >= 0) && (h <= 1.0)) ? h : fmod(h + abs((int) h) + 1, 1.0f);
    s = ((s >= 0) && (s <= 1.0)) ? s : fmod(s + abs((int) s) + 1, 1.0f);
    v = ((v >= 0) && (v <= 1.0)) ? v : fmod(v + abs((int) v) + 1, 1.0f);

    // use degree unit for convenience...
    h *= 360;

    float rm1, rm2;
    if (v < 0.5) {
	rm2 = v * (1.0 + s);
    }
    else {
	rm2 = v + s - v * s;
    }
    rm1 = 2.0 * v - rm2;

    if (s == 0) {
	r = g = b = v;
    }
    else {
	r = HSV2RGB_aux(rm1, rm2, h + 120);
	g = HSV2RGB_aux(rm1, rm2, h);
	b = HSV2RGB_aux(rm1, rm2, h - 120);
    }
}

static float HSV2RGB_aux(float rm1, float rm2, float h)
{
    // see HIGZ routine IGHR01
    h = (h > 360) ? (h - 360) : ((h < 0) ? (h + 360) : h);

    if (h < 60) {
	return rm1 + (rm2 - rm1) * h / 60;
    }
    else if (h < 180) {
	return rm2;
    }
    else if (h < 240) {
	return rm1 + (rm2 - rm1) * (240 - h) / 60;
    }
    else {
	return rm1;
    }
}



TKinokoCanvasTwoColorScale::TKinokoCanvasTwoColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
: TKinokoCanvasColorScale(ImageArea, NumberOfColors)
{
}

TKinokoCanvasTwoColorScale::~TKinokoCanvasTwoColorScale()
{
}

void TKinokoCanvasTwoColorScale::CreateColorTable(void)
{
    float StartR = 0.1, StartG = 0.1, StartB = 1.0;
    float EndR = 1.0, EndG = 0, EndB = 0;
    
    float WidthR = EndR - StartR;
    float WidthG = EndG - StartG;
    float WidthB = EndB - StartB;

    for (int i = 0; i < _NumberOfColors; i++) {
	float Value = (float) i / (_NumberOfColors - 1);
	float r = Value * WidthR + StartR;
	float g = Value * WidthG + StartG;
	float b = Value * WidthB + StartB;

	int Color = _ImageArea->AllocateColorRgb(r, g, b);
	_ColorTable[i] = Color;
    }
}



TKinokoCanvasWhiteBackgroundRainbowColorScale::TKinokoCanvasWhiteBackgroundRainbowColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
: TKinokoCanvasRainbowColorScale(ImageArea, NumberOfColors)
{
    _BackgroundColor = _ImageArea->AllocateColorRgb(1.0, 1.0, 1.0);
}

TKinokoCanvasWhiteBackgroundRainbowColorScale::~TKinokoCanvasWhiteBackgroundRainbowColorScale()
{
}

int TKinokoCanvasWhiteBackgroundRainbowColorScale::ColorIndexOf(float Value)
{
    if (Value <= 0) {
	return _BackgroundColor;
    }
    else {
	return TKinokoCanvasRainbowColorScale::ColorIndexOf(Value);
    }
}



TKinokoCanvasBlackBackgroundRainbowColorScale::TKinokoCanvasBlackBackgroundRainbowColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
: TKinokoCanvasRainbowColorScale(ImageArea, NumberOfColors)
{
    _BackgroundColor = _ImageArea->AllocateColorRgb(0, 0, 0);
}

TKinokoCanvasBlackBackgroundRainbowColorScale::~TKinokoCanvasBlackBackgroundRainbowColorScale()
{
}

int TKinokoCanvasBlackBackgroundRainbowColorScale::ColorIndexOf(float Value)
{
    if (Value <= 0) {
	return _BackgroundColor;
    }
    else {
	return TKinokoCanvasRainbowColorScale::ColorIndexOf(Value);
    }
}



TKinokoCanvasGrayBackgroundRainbowColorScale::TKinokoCanvasGrayBackgroundRainbowColorScale(TKinokoCanvasImageArea* ImageArea, int NumberOfColors)
: TKinokoCanvasRainbowColorScale(ImageArea, NumberOfColors)
{
    _BackgroundColor = _ImageArea->AllocateColorRgb(0.5, 0.5, 0.5);
}

TKinokoCanvasGrayBackgroundRainbowColorScale::~TKinokoCanvasGrayBackgroundRainbowColorScale()
{
}

int TKinokoCanvasGrayBackgroundRainbowColorScale::ColorIndexOf(float Value)
{
    if (Value <= 0) {
	return _BackgroundColor;
    }
    else {
	return TKinokoCanvasRainbowColorScale::ColorIndexOf(Value);
    }
}
