/* KinokoCanvasDataScale.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 9 January 2002. */


#include <iostream>
#include <algorithm>
#include <cmath>
#include <climits>
#include "KinokoCanvasDataScale.hh"

using namespace std;


TKinokoCanvasDataScale::TKinokoCanvasDataScale(float Min, float Max)
{
    _Min = min(Min, Max);
    _Max = max(Min, Max);

    _Width = _Max - _Min;
    _FirstDivision = _Min;
    _DivisionStep = -1;
    _NumberOfDivisions = -1;

    _UsedMin = _Min;
    _UsedMax = _Max;
}

TKinokoCanvasDataScale::~TKinokoCanvasDataScale()
{
}

float TKinokoCanvasDataScale::Min(void) const
{
    return _Min;
}

float TKinokoCanvasDataScale::Max(void) const
{
    return _Max;
}

float TKinokoCanvasDataScale::Width(void) const
{
    return _Width;
}

TKinokoCanvasDataScale& TKinokoCanvasDataScale::Rescale(float Min, float Max)
{
    _Min = min(Min, Max);
    _Max = max(Min, Max);
    _Width = Max - Min;

    // recalculate steps only when the range has been changed //
    if (
	(fabs(_Min - _UsedMin) > _Width * 1.0e-5) ||
	(fabs(_Max - _UsedMax) > _Width * 1.0e-5)
    ){
	_FirstDivision = _Min;
	_DivisionStep = -1;
	_NumberOfDivisions = -1;
	_UsedMin = _Min;
	_UsedMax = _Max;
    }

    return *this;
}

int TKinokoCanvasDataScale::NumberOfDivisions(void)
{
    if (_NumberOfDivisions < 0) {
	pair<float, float> StepData = DecideSteps(_Min, _Max);
	_FirstDivision = StepData.first;
	_DivisionStep = StepData.second;

	_NumberOfDivisions = (int) ((_Max - _FirstDivision) / _DivisionStep) + 1;
    }

    return _NumberOfDivisions;
}

float TKinokoCanvasDataScale::DivisionValueOf(int DivisionIndex)
{
    if (_NumberOfDivisions < 0) {
	NumberOfDivisions();
    }

    return _FirstDivision + DivisionIndex * _DivisionStep;
}

pair<float, float> TKinokoCanvasDataScale::DecideSteps(float Min, float Max, int Mag)
{
    pair<float, float> Step;  // pair of first-step-value and step-width
    float Width = Max - Min;

    // too deep recursive calls. give up.
    if (abs(Mag) > 10) {
	Step.first = Min;
	Step.second = Width / 4;
	return Step;
    }

    // order-normalization to make the width fit in range (1, 20]
    if (Width > 20.1) {
	Step = DecideSteps(Min / 10, Max / 10, Mag - 1);
	Step.first *= 10;
	Step.second *= 10;
	return Step;
    }
    else if (Width < 1.001) {
	Step = DecideSteps(Min * 10, Max * 10, Mag + 1);
	Step.first /= 10;
	Step.second /= 10;
	return Step;
    }

    // factor-normalization to make the width fit in range (2, 10]
    else if (Width > 10.1) {
	Step = DecideSteps(Min / 2, Max / 2, Mag);
	Step.first *= 2;
	Step.second *= 2;
	return Step;
    }
    else if (Width < 2.001) {
	Step = DecideSteps(Min * 2, Max * 2, Mag);
	Step.first /= 2;
	Step.second /= 2;
	return Step;
    }

    float IMin;
    if ((Min < -LONG_MAX) || (Min > LONG_MAX)) {
	IMin = Min;
    }
    else {
	IMin = ((long) Min < Min) ? ((long) Min + 1) : ((long) Min);
    }
    Step.first = IMin;

    float IWidth = Max - IMin;
    if (IWidth <= 5) {
	Step.second = 1;
    }
    else {
	Step.second = 2;
    }

    return Step;
}



TKinokoCanvasLogDataScale::TKinokoCanvasLogDataScale(float Min, float Max)
: TKinokoCanvasDataScale(Min, Max)
{
    _IsUsingNormalScale = false;
    
    //...  to be removed
    if ((Min <= 0) || (Max <= Min)) {
	cerr << "WARNING: TKinokoCanvasLogDataScale::TKinokoCavansLogDataScale(): ";
	cerr << "invalid log scale range: ";
	cerr << "(" << Min << ", " << Max << ")" << endl;
    }
    if (_Min <= 0) {
	_Min = 0.5;
    }
    if (_Max <= _Min) {
	_Min = _Max / 100.0;
    }
}

TKinokoCanvasLogDataScale::~TKinokoCanvasLogDataScale()
{
}

TKinokoCanvasDataScale& TKinokoCanvasLogDataScale::Rescale(float Min, float Max)
{
    //...  to be removed
    if ((Min <= 0) || (Max <= Min)) {
	cerr << "WARNING: TKinokoCanvasLogDataScale::Rescale(): ";
	cerr << "invalid log scale range: ";
	cerr << "(" << Min << ", " << Max << ")" << endl;
    }
    if (Min <= 0) {
	Min = 0.5;
    }
    if (Max <= Min) {
	Min = Max / 10.0;
    }

    return TKinokoCanvasDataScale::Rescale(Min, Max);
}

int TKinokoCanvasLogDataScale::NumberOfDivisions(void)
{
    if (_NumberOfDivisions > 0) {
	return _NumberOfDivisions;
    }

    float LogMin = log10(_Min);
    float LogMax = log10(_Max);

    int ILogMin = (int) (((int) LogMin < LogMin) ? (LogMin + 1) : LogMin);
    float LogWidth = LogMax - ILogMin;
    
    if (LogWidth > 1) {
	int LogDivisionStep = (int) ((LogWidth - 1) / 4) + 1;
	_NumberOfDivisions = (int) (LogWidth / LogDivisionStep) + 1;
	_DivisionStep = pow(10.0, (double) LogDivisionStep);
	_FirstDivision = pow(10.0, (double) ILogMin);
	_IsUsingNormalScale = false;
    }
    else {
	_NumberOfDivisions = -1;
	_NumberOfDivisions = TKinokoCanvasDataScale::NumberOfDivisions();
	_IsUsingNormalScale = true;
    }

    return _NumberOfDivisions;
}

float TKinokoCanvasLogDataScale::DivisionValueOf(int DivisionIndex)
{
    if (_NumberOfDivisions < 0) {
	NumberOfDivisions();
    }

    if (_IsUsingNormalScale) {
	return TKinokoCanvasDataScale::DivisionValueOf(DivisionIndex);
    }

    return _FirstDivision * pow(_DivisionStep, DivisionIndex);
}
