/* KinokoCanvasDrawableObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 9 January 2002. */


#include <strstream>
#include <algorithm>
#include <cmath>
#include "KinokoCanvasObject.hh"
#include "KinokoCanvasDrawableObject.hh"

using namespace std;


#define clip(x, lower, upper) (min(max((lower), (x)), (upper)))


TKinokoCanvasDrawableObject::TKinokoCanvasDrawableObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea)
: TKinokoCanvasObject(Canvas, ImageArea)
{
    _ColorIndex = _ColorIndexList[Color_Foreground];
    _LineWidth = 1;
    _LineStyleIndex = 0;
    _FontIndex = _FontIndexList[Font_Normal];

    _TextAdjustment = "tl";

    SetCoordinate(0, 0, 1, 1);
}

TKinokoCanvasDrawableObject::~TKinokoCanvasDrawableObject()
{
}

int TKinokoCanvasDrawableObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "set") {
	string Name, Value;
	InputStream >> Name >> ws;
	if (! Name.empty()) {
	    getline(InputStream, Value, '\n');
	    Result = ProcessSetCommand(Name, Value);
	}
    }
    else if (Command == "clear") {
	Result = ProcessClearCommand(InputStream);
    }
    else if (Command == "savecontext") {
	Result = ProcessSaveContextCommand(InputStream);
    }
    else if (Command == "restorecontext") {
	Result = ProcessRestoreContextCommand(InputStream);
    }
    else if (Command == "drawtext") {
	Result = ProcessDrawTextCommand(InputStream);
    }
    else if (Command == "drawline") {
	Result = ProcessDrawLineCommand(InputStream);
    }
    else if (Command == "drawrect") {
	Result = ProcessDrawRectCommand(InputStream);
    }
    else if (Command == "drawrectfill") {
	Result = ProcessDrawRectFillCommand(InputStream);
    }
    else if (Command == "drawcircle") {
	Result = ProcessDrawCircleCommand(InputStream);
    }
    else if (Command == "drawcirclefill") {
	Result = ProcessDrawCircleFillCommand(InputStream);
    }
    else if (Command == "drawimage") {
	Result = ProcessDrawImageCommand(InputStream);
    }
    else if (Command == "setfont") {
	Result = ProcessSetFontCommand(InputStream);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "color") {
	Result = SetColor(Value);
    }
    else if (Name == "linewidth") {
	Result = SetLineWidth(Value);
    }
    else if (Name == "linestyle") {
	Result = SetLineStyle(Value);
    }
    else if (Name == "textadjustment") {
	_TextAdjustment = Value;
    }
    else {
	Result = 0;
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessClearCommand(istream& InputStream)
{
    Clear();
    return 1;
}

int TKinokoCanvasDrawableObject::ProcessSaveContextCommand(istream& InputStream)
{
    return SaveContext();
}

int TKinokoCanvasDrawableObject::ProcessRestoreContextCommand(istream& InputStream)
{
    return RestoreContext();
}

int TKinokoCanvasDrawableObject::ProcessSetFontCommand(istream& InputStream)
{
    int Result = 0;

    string FontName;
    if (InputStream >> FontName) {
	int FontSize;
	if (! (InputStream >> FontSize)) {
	    FontSize = 0;
	}

	Result = SetFont(FontName, FontSize);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawTextCommand(istream& InputStream)
{
    int Result = 0;

    float x, y;
    if (InputStream >> x >> y) {
	Result = 1;

	if ((x < _XMin) || (x > _XMax) || (y < _YMin) || (y > _YMax)) {
	    return Result;
	}

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	
	string Text;
	InputStream >> ws;
	if (getline(InputStream, Text, '\n')) {
	    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
	    int OldFontIndex = _ImageArea->SetFont(_FontIndex);

	    _ImageArea->DrawText(cx, cy, Text, _TextAdjustment);

	    _ImageArea->SetColor(OldColorIndex);
	    _ImageArea->SetFont(OldFontIndex);
	}
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawLineCommand(istream& InputStream)
{
    int Result = 0;

    float x0, y0, x1, y1;
    if (InputStream >> x0 >> y0 >> x1 >> y1) {
	Result = 1;

	float cx0 = CanvasXOf(clip(x0, _XMin, _XMax));
	float cy0 = CanvasYOf(clip(y0, _YMin, _YMax));
	float cx1 = CanvasXOf(clip(x1, _XMin, _XMax));
	float cy1 = CanvasYOf(clip(y1, _YMin, _YMax));
	
	int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
	int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
	int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

	_ImageArea->DrawLine(cx0, cy0, cx1, cy1);

	_ImageArea->SetColor(OldColorIndex);
	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetLineStyle(OldLineStyleIndex);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawRectCommand(istream& InputStream)
{
    int Result = 0;

    float x0, y0, x1, y1;
    if (InputStream >> x0 >> y0 >> x1 >> y1) {
	Result = 1;

	float cx0 = CanvasXOf(clip(x0, _XMin, _XMax));
	float cy0 = CanvasYOf(clip(y0, _YMin, _YMax));
	float cx1 = CanvasXOf(clip(x1, _XMin, _XMax));
	float cy1 = CanvasYOf(clip(y1, _YMin, _YMax));
	
	int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
	int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
	int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

	_ImageArea->DrawRect(cx0, cy0, cx1, cy1);

	_ImageArea->SetColor(OldColorIndex);
	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetLineStyle(OldLineStyleIndex);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawRectFillCommand(istream& InputStream)
{
    int Result = 0;

    float x0, y0, x1, y1;
    if (InputStream >> x0 >> y0 >> x1 >> y1) {
	Result = 1;

	float cx0 = CanvasXOf(clip(x0, _XMin, _XMax));
	float cy0 = CanvasYOf(clip(y0, _YMin, _YMax));
	float cx1 = CanvasXOf(clip(x1, _XMin, _XMax));
	float cy1 = CanvasYOf(clip(y1, _YMin, _YMax));
	
	int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
	int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
	int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

	_ImageArea->DrawRectFill(cx0, cy0, cx1, cy1);

	_ImageArea->SetColor(OldColorIndex);
	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetLineStyle(OldLineStyleIndex);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawCircleCommand(istream& InputStream)
{
    int Result = 0;

    float x, y, r;
    if (InputStream >> x >> y >> r) {
	Result = 1;

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	float rx = fabs(CanvasXOf(x + r) - CanvasXOf(x));
	float ry = fabs(CanvasYOf(y + r) - CanvasYOf(y));
	
	int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
	int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
	int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

	_ImageArea->DrawEllipse(cx, cy, rx, ry);

	_ImageArea->SetColor(OldColorIndex);
	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetLineStyle(OldLineStyleIndex);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawCircleFillCommand(istream& InputStream)
{
    int Result = 0;

    float x, y, r;
    if (InputStream >> x >> y >> r) {
	Result = 1;

	float cx = CanvasXOf(x);
	float cy = CanvasYOf(y);
	float rx = fabs(CanvasXOf(x + r) - CanvasXOf(x));
	float ry = fabs(CanvasYOf(y + r) - CanvasYOf(y));
	
	int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
	int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
	int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

	_ImageArea->DrawEllipseFill(cx, cy, rx, ry);

	_ImageArea->SetColor(OldColorIndex);
	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetLineStyle(OldLineStyleIndex);
    }

    return Result;
}

int TKinokoCanvasDrawableObject::ProcessDrawImageCommand(std::istream& InputStream)
{
    int Result = 0;

    float x, y;
    string FileName;
    if (InputStream >> x >> y >> FileName) {
	TKinokoCanvasImage* Image = _ImageArea->LoadXpmImage(FileName);
	if (Image == 0) {
	    return Result;
	}
	Result = 1;

	float cx0 = CanvasXOf(x);
	float cy0 = CanvasYOf(y);
	float cx1 = cx0 + Image->Width();
	float cy1 = cy0 + Image->Height();

	float cxmax = max(CanvasXOf(_XMin), CanvasXOf(_XMax));
	float cxmin = min(CanvasXOf(_XMin), CanvasXOf(_XMax));
	float cymax = max(CanvasYOf(_YMin), CanvasYOf(_YMax));
	float cymin = min(CanvasYOf(_YMin), CanvasYOf(_YMax));

	if ((cx0 < cxmin) || (cx1 > cxmax) || (cy0 < cymin) || (cy1 > cymax)) {
	    return Result;
	}

	_ImageArea->DrawImage(cx0, cy0, Image);
    }

    return Result;
}

float TKinokoCanvasDrawableObject::CanvasXOf(float x)
{
    return (x - _XMin) / _XWidth * _Width + _OffsetX;
}

float TKinokoCanvasDrawableObject::CanvasYOf(float y)
{
    return (y - _YMin) / _YWidth * _Height + _OffsetY;
}

int TKinokoCanvasDrawableObject::SetCoordinate(float XMin, float YMin, float XWidth, float YWidth)
{
    _XMin = XMin;
    _YMin = YMin;
    _XWidth = XWidth;
    _YWidth = YWidth;

    _XMax = XMin + XWidth;
    _YMax = YMin + YWidth;

    return 1;
}

int TKinokoCanvasDrawableObject::SetColor(const string& ColorName)
{
    int ColorIndex = _ImageArea->AllocateColor(ColorName);

    if (ColorIndex >= 0) {
	_ColorIndex = ColorIndex;
	return 1;
    }
    else {
	return 0;
    }
}

int TKinokoCanvasDrawableObject::SetFont(const string& FontName, int FontSize)
{
    int FontIndex;
    if (FontName == "default") {
	FontIndex = _FontIndexList[Font_Normal];
    }
    else {
	FontIndex = _ImageArea->LoadFont(FontName, FontSize);
    }

    if (FontIndex >= 0) {
	_FontIndex = FontIndex;
    }
    else {
	return 0;
    }

    int OldFontIndex = _ImageArea->SetFont(FontIndex);
    _LineHeight = (int) (1.2 * _ImageArea->TextHeightOf("ABCDEfghijklpq"));
    _ImageArea->SetFont(OldFontIndex);

    return 1;
}

int TKinokoCanvasDrawableObject::SetLineWidth(const string& LineWidthString)
{
    int Value;
    if (istrstream(LineWidthString.c_str()) >> Value) {
	_LineWidth = Value;
    }
    else {
	return 0;
    }

    return 1;
}

int TKinokoCanvasDrawableObject::SetLineStyle(const string& LineStyleString)
{
    _LineStyleIndex = _ImageArea->CreateLineStyle(LineStyleString);

    return 1;
}

int TKinokoCanvasDrawableObject::SaveContext(void)
{
    _ColorIndexStack.push_back(_ColorIndex);
    _LineWidthStack.push_back(_LineWidth);
    _LineStyleIndexStack.push_back(_LineStyleIndex);
    _FontIndexStack.push_back(_FontIndex);
    _TextAdjustmentStack.push_back(_TextAdjustment);

    return 1;
}

int TKinokoCanvasDrawableObject::RestoreContext(void)
{
    if (_ColorIndexStack.empty()) {
	return 0;
    }

    _ColorIndex = _ColorIndexStack.back();
    _LineWidth = _LineWidthStack.back();
    _LineStyleIndex = _LineStyleIndexStack.back();
    _FontIndex = _FontIndexStack.back();
    _TextAdjustment = _TextAdjustmentStack.back();

    _ColorIndexStack.pop_back();
    _LineWidthStack.pop_back();
    _LineStyleIndexStack.pop_back();
    _FontIndexStack.pop_back();
    _TextAdjustmentStack.pop_back();

    return 1;
}
