/* KinokoCanvasFramedObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 9 January 2002. */


#include <strstream>
#include <algorithm>
#include <cmath>
#include "KinokoCanvasDataScale.hh"
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasDrawableObject.hh"
#include "KinokoCanvasFramedObject.hh"

using namespace std;


TKinokoCanvasFramedObject::TKinokoCanvasFramedObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea)
: TKinokoCanvasDrawableObject(Canvas, ImageArea)
{
    _TopMargin = 4 * _ImageArea->TextHeightOf("TITLE");
    _BottomMargin = _ImageArea->TextHeightOf("0");
    _RightMargin = _ImageArea->TextWidthOf("00");
    _LeftMargin = _ImageArea->TextWidthOf("00");

    _FrameOffsetX = 0;
    _FrameOffsetY = 0;
    _FrameWidth = 0;
    _FrameHeight = 0;

    _IsFrameValid = false;
}

TKinokoCanvasFramedObject::~TKinokoCanvasFramedObject()
{
}

void TKinokoCanvasFramedObject::SetPosition(int OffsetX, int OffsetY, int Width, int Height)
{
    TKinokoCanvasObject::SetPosition(OffsetX, OffsetY, Width, Height);

    _FrameOffsetX = _OffsetX + _LeftMargin;
    _FrameOffsetY = _OffsetY + _TopMargin;
    _FrameWidth = _Width - (_LeftMargin + _RightMargin);
    _FrameHeight = _Height - (_TopMargin + _BottomMargin);

    _IsFrameValid = ((_FrameWidth > 0) && (_FrameHeight > 0));
}

void TKinokoCanvasFramedObject::Clear(void)
{
    TKinokoCanvasObject::Clear();
}

float TKinokoCanvasFramedObject::CanvasXOf(float x)
{
    return (x - _XMin) / _XWidth * _FrameWidth + _FrameOffsetX;
}

float TKinokoCanvasFramedObject::CanvasYOf(float y)
{
    return (y - _YMin) / _YWidth * _FrameHeight + _FrameOffsetY;
}

int TKinokoCanvasFramedObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "frame") {
	Result = ProcessFrameCommand(InputStream);
	DrawFrame();
    }
    else if (Command == "drawframe") {
	Result = ProcessDrawFrameCommand(InputStream);
    }
    else {
	Result = TKinokoCanvasDrawableObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvasFramedObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "title") {
	Result = SetTitle(Value);
    }
    else {
	Result = TKinokoCanvasDrawableObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

int TKinokoCanvasFramedObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = 0;

    float XMin, XMax, YMin, YMax;
    if (InputStream >> XMin >> XMax >> YMin >> YMax) {
	if ((XMin < XMax) && (YMin < YMax)) {
	    Result = SetCoordinate(XMin, YMin, XMax - XMin, YMax - YMin);
	}
    }

    return Result;
}

int TKinokoCanvasFramedObject::ProcessDrawFrameCommand(istream& InputStream)
{
    return DrawFrame();
}

int TKinokoCanvasFramedObject::SetTitle(const string& Title)
{
    _Title = Title;
    return 1;
}

int TKinokoCanvasFramedObject::DrawFrame(void)
{
    if (_Title.empty()) {
	return 1;
    }

    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Title]);

    _ImageArea->DrawText(
	_FrameOffsetX + _FrameWidth / 2, _OffsetY + (int) (_TopMargin * 0.7), 
	_Title, "bc"
    );

    _ImageArea->SetFont(OldFontIndex);

    return 1;
}



TKinokoCanvasColorScaledObject::TKinokoCanvasColorScaledObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea, TKinokoCanvasColorScale* ColorScale)
: TKinokoCanvasFramedObject(Canvas, ImageArea)
{
    _ColorScale = ColorScale;

    _ZScaleBoxWidth = 8;
    _ZScaleWidth = _ImageArea->TextWidthOf("000000") + _ZScaleBoxWidth;
    _RightMargin += _ZScaleWidth;

    _IsZScaleLog = false;
    SetZCoordinate(0, 1);
    _ZScale = new TKinokoCanvasDataScale();
}

TKinokoCanvasColorScaledObject::~TKinokoCanvasColorScaledObject()
{
    delete _ColorScale;
}

int TKinokoCanvasColorScaledObject::SetZCoordinate(float ZMin, float ZWidth)
{
    _ZMinOrg = ZMin;
    _ZWidthOrg = ZWidth;

    if (_IsZScaleLog) {
	if (ZMin <= 0) {
	    float ZMax = ZMin + ZWidth;
	    float ZMaxForLog = (ZMax <= 0) ? 1.0 : ZMax;
	    float ZMinForLog = min(ZMaxForLog / 100.0, 0.5);
	    ZMin = ZMinForLog;
	    ZWidth = ZMaxForLog - ZMinForLog;
	}

	_LogZMin = log10(ZMin);
	_LogZMax = log10(ZMin + ZWidth);
	_LogZWidth = _LogZMax - _LogZMin;
    }

    _ZMin = ZMin;
    _ZWidth = ZWidth;
    _ZMax = ZMin + ZWidth;

    return 1;
}

int TKinokoCanvasColorScaledObject::SetZScaleLinear(void)
{
    if (_IsZScaleLog) {
	_IsZScaleLog = false;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
	delete _ZScale;
	_ZScale = new TKinokoCanvasDataScale(_ZMin, _ZMax);
    }

    return 1;
}

int TKinokoCanvasColorScaledObject::SetZScaleLog(void)
{
    if (! _IsZScaleLog) {
	_IsZScaleLog = true;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
	delete _ZScale;
	_ZScale = new TKinokoCanvasLogDataScale(_ZMin, _ZMax);
    }

    return 1;
}

int TKinokoCanvasColorScaledObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = TKinokoCanvasFramedObject::ProcessFrameCommand(InputStream);
    if (Result == 0) {
	return Result;
    }

    Result = 0;

    float ZMin, ZMax;
    if (InputStream >> ZMin >> ZMax) {
	if (ZMax > ZMin) {
	    Result = SetZCoordinate(ZMin, ZMax - ZMin);
	}
    }

    return Result;
}

int TKinokoCanvasColorScaledObject::DrawFrame(void)
{
    TKinokoCanvasFramedObject::DrawFrame();
    if (! _IsFrameValid) {
	return 1;
    }

    int x1 = _OffsetX + _Width - _RightMargin + _ZScaleWidth;
    int y1 = _OffsetY + _Height - _BottomMargin;
    int x0 = x1 - _ZScaleBoxWidth;
    int y0 = _OffsetY + _TopMargin;
    int ScaleHeight = y1 - y0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth =  _ImageArea->SetLineWidth(1);
    int OldLineStyle =  _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    _ImageArea->DrawRect(x0, y0, x1, y1);

    int NumberOfColors = 64;
    float ColorStep = 1.0 / NumberOfColors;
    for (float z = 0; z < 1.0; z += ColorStep) {
	_ImageArea->SetColor(_ColorScale->ColorIndexOf(z));
	_ImageArea->DrawRectFill(
	    x0, y1 - (int) (ScaleHeight * (z + ColorStep)),
	    x1, y1 - (int) (ScaleHeight * z)
	);
    }

    _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);

    _ZScale->Rescale(_ZMin, _ZMax);
    int NumberOfZScaleDivisions = _ZScale->NumberOfDivisions();
    for (int i = 0; i < NumberOfZScaleDivisions; i++) {
	float zs = _ZScale->DivisionValueOf(i);

	int y;
	if (! _IsZScaleLog) {
	    y = y1 - (int) (ScaleHeight * (zs - _ZMin) / _ZWidth);
	}
	else {
	    y = y1 - (int) (ScaleHeight * (log10(zs) - _LogZMin) / _LogZWidth);
	}

	_ImageArea->DrawLine(x0 - 2, y, x0, y);
	_ImageArea->DrawNumberText(x0 - 3 , y, zs, "cr");
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyle);
    _ImageArea->SetFont(OldFontIndex);

    return 1;
}
