/* KinokoCanvasPlotObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 17 December 2002. */


#include <strstream>
#include <algorithm>
#include <cmath>
#include "KinokoCanvasDataScale.hh"
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasFramedObject.hh"
#include "KinokoCanvasPlotObject.hh"

using namespace std;


#define clip(x, lower, upper) (min(max((lower), (x)), (upper)))


TKinokoCanvasPlotObject::TKinokoCanvasPlotObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea)
: TKinokoCanvasFramedObject(Canvas, ImageArea)
{
    _BottomMargin += 3 * _ImageArea->TextHeightOf("0");
    _RightMargin += _ImageArea->TextWidthOf("0");
    _LeftMargin += _ImageArea->TextWidthOf("00000");

    _Marker = 0;
    _MarkerSize = 3;
    _MarkerFactory = new TKinokoCanvasMarkerFactory();

    _IsXScaleLog = false;
    _IsYScaleLog = false;

    _IsXScaleLabelDisabled = false;
    _IsYScaleLabelDisabled = false;
    _IsYTitleVertical = false;

    SetCoordinate(0, 0, 1, 1);
    _XScale = new TKinokoCanvasDataScale();
    _YScale = new TKinokoCanvasDataScale();
}

TKinokoCanvasPlotObject::~TKinokoCanvasPlotObject()
{
    delete _Marker;
    delete _MarkerFactory;
    delete _YScale;
    delete _XScale;
}

TKinokoShellObject* TKinokoCanvasPlotObject::Clone(void)
{
    return new TKinokoCanvasPlotObject(_Canvas, _ImageArea);
}

vector<string> TKinokoCanvasPlotObject::ActionList(void)
{
    vector<string> ActionList = TKinokoCanvasFramedObject::ActionList();

    ActionList.push_back("setYScaleLinear");
    ActionList.push_back("setYScaleLog");

    return ActionList;
}

int TKinokoCanvasPlotObject::ProcessAction(const string& ActionName)
{
    int Result = 1;
    if (ActionName == "setYScaleLinear") {
	SetYScaleLinear();
    }
    else if (ActionName == "setYScaleLog") {
	SetYScaleLog();
    }
    else {
	Result = TKinokoCanvasFramedObject::ProcessAction(ActionName);
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "plot") {
	Result = ProcessPlotCommand(InputStream, "linespoints");
    }
    else if (Command == "linesplot") {
	Result = ProcessPlotCommand(InputStream, "lines");
    }
    else if (Command == "pointsplot") {
	Result = ProcessPlotCommand(InputStream, "points");
    }
    else if (Command == "errorplot") {
	Result = ProcessErrorPlotCommand(InputStream);
    }
    else if (Command == "minmaxplot") {
	Result = ProcessMinmaxPlotCommand(InputStream);
    }
    else if (Command == "hist") {
	Result = ProcessHistCommand(InputStream);
    }
    else if (Command == "comment") {
	Result = ProcessCommentCommand(InputStream);
    }
    else if (Command == "writexscalelabel") {
	Result = ProcessWriteXScaleLabelCommand(InputStream);
    }
    else if (Command == "writeyscalelabel") {
	Result = ProcessWriteYScaleLabelCommand(InputStream);
    }
    else {
	return TKinokoCanvasFramedObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvasPlotObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "xscale") {
	if (Value == "log") {
	    SetXScaleLog();
	}
	else {
	    SetXScaleLinear();
	}
    }
    else if (Name == "yscale") {
	if (Value == "log") {
	    SetYScaleLog();
	}
	else {
	    SetYScaleLinear();
	}
    }
    else if (Name == "xtitle") {
	_XTitle = Value;
    }
    else if (Name == "ytitle") {
	_YTitle = Value;
    }
    else if (Name == "marker") {
	Result = SetMarker(Value);
    }
    else if (Name == "markersize") {
	Result = SetMarkerSize(Value);
    }
    else if (Name == "xscalelabel") {
	_IsXScaleLabelDisabled = (Value == "disabled");
    }
    else if (Name == "yscalelabel") {
	_IsYScaleLabelDisabled = (Value == "disabled");
    }
    else if (Name == "ytitleorientation") {
	_IsYTitleVertical = (Value == "vertical");
    }
    else {
	Result = TKinokoCanvasFramedObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

float TKinokoCanvasPlotObject::CanvasXOf(float x)
{
    if (! _IsXScaleLog) {
	return (x - _XMin) / _XWidth * _FrameWidth + _FrameOffsetX;
    }
    else {
	return (log10(x) - _LogXMin) / _LogXWidth * _FrameWidth + _FrameOffsetX;
    }
}

float TKinokoCanvasPlotObject::CanvasYOf(float y)
{
    if (! _IsYScaleLog) {
	return (1.0 - (y - _YMin) / _YWidth) * _FrameHeight + _FrameOffsetY;
    }
    else {
	return (1.0 - (log10(y) - _LogYMin) / _LogYWidth) * _FrameHeight + _FrameOffsetY;
    }
}

int TKinokoCanvasPlotObject::SetCoordinate(float XMin, float YMin, float XWidth, float YWidth)
{
    _XMinOrg = XMin;
    _XWidthOrg = XWidth;
    _YMinOrg = YMin;
    _YWidthOrg = YWidth;

    if (_IsXScaleLog) {
	if (XMin <= 0) {
	    float XMax = XMin + XWidth;
	    float XMaxForLog = (XMax <= 0) ? 1.0 : XMax;
	    float XMinForLog = min(XMaxForLog / 100.0, 0.5);
	    XMin = XMinForLog;
	    XWidth = XMaxForLog - XMinForLog;
	}

	_LogXMin = log10(XMin);
	_LogXMax = log10(XMin + XWidth);
	_LogXWidth = _LogXMax - _LogXMin;
    }
    
    if (_IsYScaleLog) {
	if (YMin <= 0) {
	    float YMax = YMin + YWidth;
	    float YMaxForLog = (YMax <= 0) ? 1.0 : YMax;
	    float YMinForLog = min(YMaxForLog / 100.0, 0.5);
	    YMin = YMinForLog;
	    YWidth = YMaxForLog - YMinForLog;
	}

	_LogYMin = log10(YMin);
	_LogYMax = log10(YMin + YWidth);
	_LogYWidth = _LogYMax - _LogYMin;
    }

    TKinokoCanvasFramedObject::SetCoordinate(XMin, YMin, XWidth, YWidth);

    return 1;
}

int TKinokoCanvasPlotObject::SaveContext(void)
{
    int Result = TKinokoCanvasFramedObject::SaveContext();

    if (Result > 0) {
	if (_Marker) {
	    _MarkerStack.push_back(_Marker->Clone());
	}
	else {
	    _MarkerStack.push_back(0);
	}

	_MarkerSizeStack.push_back(_MarkerSize);
    }
    
    return Result;
}

int TKinokoCanvasPlotObject::RestoreContext(void)
{
    int Result = TKinokoCanvasFramedObject::RestoreContext();

    if (Result > 0) {
	if (_Marker) {
	    delete _Marker;
	}

	_Marker = _MarkerStack.back();
	_MarkerSize = _MarkerSizeStack.back();
	_MarkerStack.pop_back();
	_MarkerSizeStack.pop_back();
    }

    return Result;
}

int TKinokoCanvasPlotObject::SetXScaleLinear(void)
{
    if (_IsXScaleLog) {
	_IsXScaleLog = false;
	SetCoordinate(_XMinOrg, _XWidthOrg, _YMinOrg, _YWidthOrg);
	delete _XScale;
	_XScale = new TKinokoCanvasDataScale(_XMin, _XMax);
    }

    return 1;
}

int TKinokoCanvasPlotObject::SetXScaleLog(void)
{
    if (! _IsXScaleLog) {
	_IsXScaleLog = true;
	SetCoordinate(_XMinOrg, _XWidthOrg, _YMinOrg, _YWidthOrg);
	delete _XScale;
	_XScale = new TKinokoCanvasLogDataScale(_XMin, _XMax);
    }

    return 1;
}

int TKinokoCanvasPlotObject::SetYScaleLinear(void)
{
    if (_IsYScaleLog) {
	_IsYScaleLog = false;
	SetCoordinate(_XMinOrg, _XWidthOrg, _YMinOrg, _YWidthOrg);
	delete _YScale;
	_YScale = new TKinokoCanvasDataScale(_YMin, _YMax);
    }

    return 1;
}

int TKinokoCanvasPlotObject::SetYScaleLog(void)
{
    if (! _IsYScaleLog) {
	_IsYScaleLog = true;
	SetCoordinate(_XMinOrg, _XWidthOrg, _YMinOrg, _YWidthOrg);
	delete _YScale;
	_YScale = new TKinokoCanvasLogDataScale(_YMin, _YMax);
    }

    return 1;
}

int TKinokoCanvasPlotObject::ProcessPlotCommand(istream& InputStream, const string& Style)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    bool DrawPoints = false;
    bool DrawLines = false;
    if ((Style == "linespoints") || (Style == "points")) {
	DrawPoints = true;
    }
    if ((Style == "linespoints") || (Style == "lines")) {
	DrawLines = true;
    }

    float x, y;
    float cx, cy;
    vector<pair<float, float> > PointList;
    while (InputStream >> x >> y) {
	//... temporary
	if ((x < _XMin) || (x > _XMax)) {
	    continue;
	}
	y = clip(y, _YMin, _YMax);

	cx = CanvasXOf(x);
	cy = CanvasYOf(y);
	
	if (DrawPoints) {
	    if (_Marker) {
		_Marker->Draw(_ImageArea, cx, cy, _MarkerSize);
	    }
	    else {
		_ImageArea->DrawRect(cx - 2, cy - 2, cx + 2, cy + 2);
	    }
	}
	if (DrawLines) {
	    PointList.push_back(make_pair(cx, cy));
	}
    }

    _ImageArea->SetLineStyle(_LineStyleIndex);

    if (! PointList.empty()) {
	_ImageArea->DrawLines(PointList);
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    Result = 1;

    return Result;
}

int TKinokoCanvasPlotObject::ProcessErrorPlotCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    float x, y, ex, ey;
    float cx, cy, cx0, cy0, cx1, cy1;
    while (InputStream >> x >> y >> ex >> ey) {
	//... temporary
	if ((x < _XMin) || (x > _XMax)) {
	    continue;
	}

	cx = CanvasXOf(clip(x, _XMin, _XMax));
	cx0 = CanvasXOf(clip(x - ex, _XMin, _XMax));
	cx1 = CanvasXOf(clip(x + ex, _XMin, _XMax));

	cy = CanvasYOf(clip(y, _YMin, _YMax));
	cy0 = CanvasYOf(clip(y - ey, _YMin, _YMax));
	cy1 = CanvasYOf(clip(y + ey, _YMin, _YMax));

	if (cx0 != cx1) {
	    _ImageArea->DrawLine(cx0, cy, cx1, cy);
	}
	if (cy0 != cy1) {
	    _ImageArea->DrawLine(cx, cy0, cx, cy1);
	}

	if (_Marker) {
	    _Marker->Draw(_ImageArea, cx, cy, _MarkerSize);
	}
	else if ((cx0 == cx1) || (cy0 == cy1)) {
	    _ImageArea->DrawRect(cx - 2, cy - 2, cx + 2, cy + 2);
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessMinmaxPlotCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();

    float x, y, ymin, ymax, deviation;
    float cx, cy, cymin, cymax, cy0, cy1;
    while (InputStream >> x >> y >> ymin >> ymax >> deviation) {
	//... temporary
	if ((x < _XMin) || (x > _XMax)) {
	    continue;
	}
	y = clip(y, _YMin, _YMax);

	cx = CanvasXOf(x);
	cy = CanvasYOf(y);
	cymin = CanvasYOf(clip(ymin, _YMin, _YMax));
	cymax = CanvasYOf(clip(ymax, _YMin, _YMax));
	cy0 = CanvasYOf(clip(y - deviation, _YMin, _YMax));
	cy1 = CanvasYOf(clip(y + deviation, _YMin, _YMax));

	_ImageArea->DrawRect(cx - 2, cy - 2, cx + 2, cy + 2);
	if (cy0 != cy1) {
	    _ImageArea->DrawLine(cx, cy0, cx, cy1);
	}

	_ImageArea->DrawLine(cx - 2, cymin - 2, cx + 2, cymin + 2);
	_ImageArea->DrawLine(cx - 2, cymin + 2, cx + 2, cymin - 2);
	_ImageArea->DrawLine(cx - 2, cymax - 2, cx + 2, cymax + 2);
	_ImageArea->DrawLine(cx - 2, cymax + 2, cx + 2, cymax - 2);
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessHistCommand(istream& InputStream)
{
    // syntax: hist xmin step y0 y1 y2 y3 ...
    // [xmin, xmin + step) -> y0, [xmin + step, xmin + 2 step) -> y1, ...

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    float xmin, xstep;
    if (! (InputStream >> xmin >> xstep)) {
	return 0;
    }

    float cx0 = CanvasXOf(xmin);
    float cy0 = CanvasYOf(_YMin);
    vector<pair<float, float> > PointList;
    PointList.push_back(make_pair(cx0, cy0));

    float x = xmin, y;
    float cx1, cy1;
    while (InputStream >> y) {
	//... temporary
	x += xstep;
	if ((x < _XMin) || (x > _XMax)) {
	    continue;
	}
	y = clip(y, _YMin, _YMax);

	cx1 = CanvasXOf(x);
	cy1 = CanvasYOf(y);

	PointList.push_back(make_pair(cx0, cy1));
	PointList.push_back(make_pair(cx1, cy1));

	cx0 = cx1;
	cy0 = cy1;
    }

    cy1 = CanvasYOf(_YMin);
    PointList.push_back(make_pair(cx0, cy1));

    int OldColorIndex = _ImageArea->SetColor(_ColorIndex);
    int OldLineWidth = _ImageArea->SetLineWidth(_LineWidth);
    int OldLineStyleIndex = _ImageArea->SetLineStyle(_LineStyleIndex);

    _ImageArea->DrawLines(PointList);

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);

    return Result = 1;
}

int TKinokoCanvasPlotObject::ProcessCommentCommand(istream& InputStream)
{
    static const int SideMargin = 7;
    static const int LineMargin = 5;

    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Normal]);

    float x, y;
    if (InputStream >> x >> y >> ws) {
	int x0 = abs((int) ((x / 100.0) * _FrameWidth)) + _FrameOffsetX;
	int y0 = abs((int) ((y / 100.0) * _FrameHeight)) + _FrameOffsetY;
	
	static char Buffer[256];
	vector<string> TextList;
	vector<int> YOffsetList;
	int MaxWidth = 0, TotalHeight = 0;
	while (InputStream.getline(Buffer, sizeof(Buffer), '\n')) {
	    int Width = _ImageArea->TextWidthOf(Buffer) + 2 * SideMargin;
	    int Height = _ImageArea->TextHeightOf(Buffer) + LineMargin;
	    MaxWidth = (Width > MaxWidth) ? Width : MaxWidth;
	    TotalHeight += Height;

	    TextList.push_back(Buffer);
	    YOffsetList.push_back(TotalHeight);
	}
	TotalHeight += LineMargin;

	if (x < 0) {
	    x0 = x0 - MaxWidth;
	}
	if (y < 0) {
	    y0 = y0 - TotalHeight;
	}

	_ImageArea->SetTextAdjustment("bl");
	_ImageArea->DrawRect(x0, y0, x0 + MaxWidth, y0 + TotalHeight);
	for (unsigned i = 0; i < TextList.size(); i++) {
	    _ImageArea->DrawText(
		x0 + SideMargin, y0 + YOffsetList[i], TextList[i]
	    );
	}

	Result = 1;
    }
    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasPlotObject::ProcessWriteXScaleLabelCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    float x;
    if ((InputStream >> x) && ((x >= _XMin) || (x <= _XMax))) {
	float cx = CanvasXOf(x);
	float y1 = _OffsetY + _Height - _BottomMargin;

	string Label;
	InputStream >> ws;
	if (getline(InputStream, Label)) {
	    _ImageArea->DrawText(cx, y1 + 3, Label, "tc");
	}
	else {
	    _ImageArea->DrawNumberText(cx, y1 + 3, x, "tc");
	}
	_ImageArea->DrawLine(cx, y1 - 5, cx, y1);

	Result = 1;
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasPlotObject::ProcessWriteYScaleLabelCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    int Result = 0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(1);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    float y;
    if ((InputStream >> y) && ((y >= _YMin) || (y <= _YMax))) {
	float cy = CanvasYOf(y);
	float x0 = _OffsetX + _LeftMargin;

	InputStream >> ws;
	string Label;
	if (getline(InputStream, Label)) {
	    _ImageArea->DrawText(x0 - 3, cy, Label, "cr");
	}
	else {
	    _ImageArea->DrawNumberText(x0 - 3, cy, y, "cr");
	}
	_ImageArea->DrawLine(x0, cy, x0 + 5, cy);

	Result = 1;
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return Result;
}

int TKinokoCanvasPlotObject::SetMarker(string MarkerName)
{
    string Parameter;
    size_t NameLength = MarkerName.find_first_of(":");
    if (NameLength != string::npos) {
	Parameter = MarkerName.substr(NameLength + 1, string::npos);
	MarkerName = MarkerName.substr(0, NameLength);
    }

    if (_Marker) {
	delete _Marker;
    }
    _Marker = _MarkerFactory->CreateMarker(MarkerName, Parameter);

    return _Marker ? 1 : 0;
}

int TKinokoCanvasPlotObject::SetMarkerSize(const string& SizeString)
{
    int Value = 0;
    istrstream(SizeString.c_str()) >> Value;

    if (Value > 0) {
	_MarkerSize = Value;
    }
    else {
	return 0;
    }

    return 1;
}

int TKinokoCanvasPlotObject::DrawFrame(void)
{
    TKinokoCanvasFramedObject::DrawFrame();

    if (! _IsFrameValid) {
	return 1;
    }

    int x0 = _OffsetX + _LeftMargin;
    int y0 = _OffsetY + _TopMargin;
    int x1 = _OffsetX + _Width - _RightMargin;
    int y1 = _OffsetY + _Height - _BottomMargin;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth = _ImageArea->SetLineWidth(2);
    int OldLineStyleIndex = _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    _ImageArea->DrawRect(x0, y0, x1, y1);

    _ImageArea->DrawText(x1, (int) (y1 + 1.5 * _LineHeight), _XTitle, "tr");
    if (! _IsYTitleVertical) {
	_ImageArea->DrawText(x0, y0 - _LineHeight, _YTitle, "br");
    }
    else {
	int OldTextOrientation = _ImageArea->SetTextOrientation(90);
	_ImageArea->DrawText((int) (_OffsetX + 0.5 * _LineHeight), y0, _YTitle, "tl");
	_ImageArea->SetTextOrientation(OldTextOrientation);
    }

    _ImageArea->SetLineWidth(1);
    if (! _IsXScaleLabelDisabled) {
	_XScale->Rescale(_XMin, _XMax);

	int NumberOfXScaleDivisions = _XScale->NumberOfDivisions();
	for (int i = 0; i < NumberOfXScaleDivisions; i++) {
	    float xs = _XScale->DivisionValueOf(i);
	    float x = CanvasXOf(xs);
	    _ImageArea->DrawLine(x, y1 - 5, x, y1);
	    _ImageArea->DrawNumberText(x, y1 + 3, xs, "tc");
	}
    }
    if (! _IsYScaleLabelDisabled) {
	_YScale->Rescale(_YMin, _YMax);

	int NumberOfYScaleDivisions = _YScale->NumberOfDivisions();
	for (int i = 0; i < NumberOfYScaleDivisions; i++) {
	    float ys = _YScale->DivisionValueOf(i);
	    float y = CanvasYOf(ys);
	    _ImageArea->DrawLine(x0, y, x0 + 5, y);
	    _ImageArea->DrawNumberText(x0 - 3, y, ys, "cr");
	}
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyleIndex);
    _ImageArea->SetFont(OldFontIndex);

    return 1;
}



TKinokoCanvasMapPlotObject::TKinokoCanvasMapPlotObject(TKinokoCanvas* Canvas, TKinokoCanvasImageArea* ImageArea)
: TKinokoCanvasColorScaledObject(Canvas, ImageArea, new TKinokoCanvasGrayBackgroundRainbowColorScale(ImageArea)), TKinokoCanvasFramedObject(Canvas, ImageArea)
{
}

TKinokoCanvasMapPlotObject::~TKinokoCanvasMapPlotObject()
{
}

TKinokoShellObject* TKinokoCanvasMapPlotObject::Clone(void)
{
    return new TKinokoCanvasMapPlotObject(_Canvas, _ImageArea);
}

float TKinokoCanvasMapPlotObject::CanvasXOf(float x)
{
    return (x - _XMin) / _XWidth * _FrameWidth + _FrameOffsetX;
}

float TKinokoCanvasMapPlotObject::CanvasYOf(float y)
{
    return (1.0 - (y - _YMin) / _YWidth) * _FrameHeight + _FrameOffsetY;
}

int TKinokoCanvasMapPlotObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "plot") {
	Result = ProcessPlotCommand(InputStream);
    }
    else {
	Result = TKinokoCanvasFramedObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvasMapPlotObject::ProcessPlotCommand(istream& InputStream)
{
    if (! _IsFrameValid) {
	return 1;
    }

    float Diameter;
    if (! (InputStream >> Diameter)) {
	return 0;
    }
    Diameter *= _FrameWidth / 100.0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    float x, y, z;
    float cx, cy;
    while (InputStream >> x >> y >> z) {
	if ((x < _XMin) || (x > _XMax) || (y < _YMin) || (y > _YMax)) {
	    continue;
	}
	cx = CanvasXOf(x);
	cy = CanvasYOf(y);

	z = clip(z, _ZMin, _ZMax);

	_ImageArea->SetColor(
	    _ColorScale->ColorIndexOf((z - _ZMin) / _ZWidth)
	);

	_ImageArea->DrawCircleFill(cx, cy, Diameter/2);
    }

    _ImageArea->SetColor(OldColorIndex);

    return 1;
}

int TKinokoCanvasMapPlotObject::DrawFrame(void)
{
    int Result = TKinokoCanvasColorScaledObject::DrawFrame();
    if (Result == 0) {
	return Result;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    _ImageArea->DrawRectFill(
	_FrameOffsetX, _FrameOffsetY, 
	_FrameOffsetX + _FrameWidth, _FrameOffsetY + _FrameHeight
    );
    _ImageArea->SetColor(OldColorIndex);

    return Result;
}
