/* KinokoControl.cc */
/* Created by Enomoto Sanshiro on 1 October 2001. */
/* Last updated by Enomoto Sanshiro on 27 March 2002. */


#include <iostream>
#include <fstream>
#include <strstream>
#include <cstring>
#include <string>
#include <vector>
#include <map>
#include <set>
#include "KinokoShell.hh"
#include "KinokoControlWidget.hh"
#include "KinokoControlPanel.hh"
#include "KinokoControlScript.hh"
#include "KinokoControl.hh"

using namespace std;



TKinokoControl::TKinokoControl(TKinokoShellConnector* ShellConnector)
: TKinokoShell(ShellConnector)
{
    _Script = 0;
    _ShellScript = 0;
}

TKinokoControl::~TKinokoControl()
{
    std::map<string, TKinokoControlWidget*>::iterator WidgetIterator;
    for (
	WidgetIterator = _WidgetTable.begin();
	WidgetIterator != _WidgetTable.end();
	WidgetIterator++
    ){
	delete (*WidgetIterator).second;
    }

    //... for backward compatibility ...//
    std::map<string, TKinokoControlScript*>::iterator ActionScriptIterator;
    for (
	ActionScriptIterator = _ActionScriptTable.begin();
	ActionScriptIterator != _ActionScriptTable.end();
	ActionScriptIterator++
    ){
	delete (*ActionScriptIterator).second;
    }

    delete _Script;
    delete _ShellScript;
}

void TKinokoControl::AddWidget(TKinokoControlWidget* Widget, const string& EnabledStateList)
{
    _WidgetTable[Widget->Name()] = Widget;

    if (! EnabledStateList.empty()) {
	string State;
	set<string> StateSet;
	istrstream StateListStream(EnabledStateList.c_str());
	while (StateListStream >> State) {
	    StateSet.insert(State);
	}

	_EnabledStateList[Widget] = StateSet;
    }
}

void TKinokoControl::AddInputWidget(TKinokoControlWidget* Widget)
{
    _InputWidgetList.push_back(Widget);
    _InputValueCache.push_back("");
}

void TKinokoControl::ReadScript(const std::string& ScriptText)
{
    if (_Script == 0) {
	if (_ShellScript == 0) {
	    _ShellScript = new TKinokoShellScript(
		CreatePopupWindow(), CreateFileSelectDialog()
	    );
	}
	_Script = new TKinokoControlScript(this);
	_Script->Merge(_ShellScript);
    }

    try {
	istrstream ScriptTextStream(ScriptText.c_str());
        _Script->Parse(ScriptTextStream);
	_Script->Execute();
    }
    catch (TScriptException &e) {
	cerr << "ERROR: " << e << endl;  //...
	return;
    }
}

#if 1
//... for backward compatibility ...//
void TKinokoControl::AddActionScript(const std::string& Name, const std::string& ActionScriptText)
{
    if (_ShellScript == 0) {
	_ShellScript = new TKinokoShellScript(
	    CreatePopupWindow(), CreateFileSelectDialog()
	);
    }

    TKinokoControlScript* ActionScript = new TKinokoControlScript(this);
    ActionScript->Merge(_ShellScript);

    try {
	istrstream ScriptTextStream(ActionScriptText.c_str());
        ActionScript->Parse(ScriptTextStream);
    }
    catch (TScriptException &e) {
	delete ActionScript;
	cerr << "ERROR: " << e << endl;  //...
	return;
    }
    
    _ActionScriptTable[Name] = ActionScript;
}
#endif

void TKinokoControl::OnClickWidget(TKinokoControlWidget* Widget)
{
    string Action = Widget->ActionOnClick();
    if (Action.empty()) {
	Invoke(Widget->Name());
    }
    else {
	ExecuteAction(Action);
    }
}

void TKinokoControl::Invoke(const string& MethodName)
{
    static char Message[2048];
    ostrstream MessageStream(Message, sizeof(Message) - 1);

    MessageStream << ".invoke " << MethodName << " ";
    for (unsigned i = 0; i < _InputWidgetList.size(); i++) {
	string Name = _InputWidgetList[i]->Name();
	string Value = ReplaceEscape(_InputWidgetList[i]->Value());

	if (Value != _InputValueCache[i]) {
	    _InputValueCache[i] = Value;
	    MessageStream << Name << "=" << Value << "&";
	}
    }
    MessageStream << ";" << endl << ends;

    _ShellConnector->SendMessage(Message, strlen(Message));
}

int TKinokoControl::ExecuteAction(const string& ActionName)
{
#if 1
    //... for backward compatibility ...//
    TKinokoControlScript* ActionScript = _ActionScriptTable[ActionName];
    if (ActionScript) {
	try {
	    TParaSymbolTable* SymbolTable = ActionScript->GetSymbolTable();
	    ActionScript->GetPackage()->ExecuteBareStatements(SymbolTable);
	    ActionScript->GetPackage()->Execute(SymbolTable);
	}
	catch (TScriptException &e) {
	    cerr << "ERROR: " << ActionName << ": " << e << endl;  //...
	    return 0;
	}
	return 1;
    }
#endif

    try {
	_Script->Execute(ActionName);
    }
    catch (TScriptException &e) {
	cerr << "ERROR: " << e << endl;  //...
	return 0;
    }

    return 1;
}

TKinokoControlWidget* TKinokoControl::LookupWidget(const std::string& WidgetName)
{
    return _WidgetTable[WidgetName];
}

int TKinokoControl::ProcessSystemCommand(const string& Command, std::istream& InputStream)
{
    int Result = 0;
    if (Command == ".open") {
	Result = ProcessOpenCommand(InputStream);
    }
    else if (Command == ".clear") {
	Result = ProcessClearCommand(InputStream);
    }
    else if (Command == ".save") {
	Result = ProcessSaveCommand(InputStream);
    }
    else if (Command == ".load") {
	Result = ProcessLoadCommand(InputStream);
    }
    else if (Command == ".set") {
	Result = ProcessSetCommand(InputStream);
    }
    else if (Command == ".execute") {
	Result = ProcessExecuteCommand(InputStream);
    }
    else if (Command == ".changeState") {
	Result = ProcessChangeStateCommand(InputStream);
    }
    else if (Command == ".openPopup") {
	Result = ProcessOpenPopupCommand(InputStream, false);
    }
    else if (Command == ".openQueryPopup") {
	Result = ProcessOpenPopupCommand(InputStream, true);
    }
    else {
	Result = TKinokoShell::ProcessSystemCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoControl::ProcessOpenCommand(istream& InputStream)
{
    string ScriptFileName;
    if (InputStream >> ScriptFileName) {
	Construct(ScriptFileName);
    }

    return 1;
}

int TKinokoControl::ProcessClearCommand(istream& InputStream)
{
    ClearValues();

    return 1;
}

int TKinokoControl::ProcessSaveCommand(istream& InputStream)
{
    string FileName;
    if (InputStream >> FileName) {
	SaveValuesTo(FileName);
    }

    return 1;
}

int TKinokoControl::ProcessLoadCommand(istream& InputStream)
{
    string FileName;
    if (InputStream >> FileName) {
	LoadValuesFrom(FileName);
    }

    return 1;
}

int TKinokoControl::ProcessSetCommand(istream& InputStream)
{
    string WidgetName;
    if (InputStream >> WidgetName >> ws) {
	char Value[256];
	InputStream.getline(Value, sizeof(Value), ';');
	SetValue(WidgetName, Value);
    }

    return 1;
}

int TKinokoControl::ProcessExecuteCommand(istream& InputStream)
{
    int Result = 0;

    string ActionName;
    if (InputStream >> ActionName) {
	Result = ExecuteAction(ActionName);
    }

    return Result;
}

int TKinokoControl::ProcessChangeStateCommand(istream& InputStream)
{
    string StateName;
    if (InputStream >> StateName) {
	ChangeState(StateName);
    }

    return 1;
}

int TKinokoControl::ProcessOpenPopupCommand(istream& InputStream, bool IsQuery)
{
    TKinokoShellPopupWindow* PopupWindow = CreatePopupWindow();

    string Type;
    if (getline(InputStream >> ws, Type, ',')) {
	PopupWindow->SetType(Type);
    }

    string ActionList;
    if (getline(InputStream >> ws, ActionList, ',')) {
	istrstream ActionListStream(ActionList.c_str());
	string Action;
	while (ActionListStream >> Action) {
	    PopupWindow->AddAction(Action);
	}
    }

    string Message;
    getline(InputStream >> ws, Message, ';');
    PopupWindow->SetMessage(Message.c_str());

    bool IsModal = IsQuery;
    string Selection = PopupWindow->Open(IsModal);

    if (IsQuery) {
	char Reply[256];
	ostrstream ReplyStream(Reply, sizeof(Reply));
	ReplyStream << Selection << ";" << endl << ends;
	_ShellConnector->SendMessage(Reply, strlen(Reply));
    }

    delete PopupWindow;

    return 1;
}

void TKinokoControl::SetValue(const string& WidgetName, const string& Value)
{
    if (_WidgetTable.count(WidgetName)) {
	_WidgetTable[WidgetName]->SetValue(Value);
    }
}

void TKinokoControl::ChangeState(const string& StateName)
{
    map<TKinokoControlWidget*, set<std::string> >::iterator i;
    for (i = _EnabledStateList.begin(); i != _EnabledStateList.end(); i++) {
	TKinokoControlWidget* Widget = (*i).first;

	if ((*i).second.count(StateName) > 0) {
	    Widget->Enable();
	}
	else {
	    Widget->Disable();
	}
    }
}

int TKinokoControl::ClearValues(void)
{
    for (unsigned i = 0; i < _InputWidgetList.size(); i++) {
	_InputWidgetList[i]->SetValue("");
    }

    return 1;
}

int TKinokoControl::SaveValuesTo(const string& FileName)
{
    ofstream File(FileName.c_str());
    if (! File) {
	return 0;
    }

    for (unsigned i = 0; i < _InputWidgetList.size(); i++) {
	File << _InputWidgetList[i]->Name() << " ";
	File << _InputWidgetList[i]->Value() << endl;
    }

    return 1;
}

int TKinokoControl::LoadValuesFrom(const string& FileName)
{
    ifstream File(FileName.c_str());
    if (! File) {
	return 0;
    }

    char Line[1024];
    while (File.getline(Line, sizeof(Line))) {
	string Name, Value;
	istrstream LineStream(Line);
	getline(LineStream >> Name >> ws, Value);

	if (_WidgetTable.count(Name)) {
	    _WidgetTable[Name]->SetValue(Value);
	}
    }

    return 1;
}

string TKinokoControl::ReplaceEscape(const string& Value)
{
    string Result;
    for (unsigned i = 0; i < Value.size(); i++) {
	switch (Value[i]) {
	  case '%':
	    Result += "%p";
	    break;
	  case '&':
	    Result += "%a";
	    break;
          case '=':
	    Result += "%e";
	    break;
	  case ';':
	    Result += "%s";
	    break;
	  default:
	    Result += Value[i];
	}
    }

    return Result;
}
