/* KinokoShellConnector.cc */
/* Created by Enomoto Sanshiro on 3 October 2001. */
/* Last updated by Enomoto Sanshiro on 8 October 2001. */


#include <iostream>
#include <strstream>
#include <string>
#include "MushTimer.hh"
#include "MushNetworkSocket.hh"
#include "KinokoShellConnector.hh"

using namespace std;


static const char LineTerminator = ';';


TKinokoShellConnector::TKinokoShellConnector(void)
{
    _FileDescriptor = 0;

    _BufferSize = 1024 * 1024;
    _Buffer = new char[_BufferSize];

    _DataStart = _Buffer;
    _DataStart[0] = '\0';
    _DataLength = 0;
}

TKinokoShellConnector::~TKinokoShellConnector()
{
    delete[] _Buffer;
} 

int TKinokoShellConnector::FileDescriptor(void)
{
    return _FileDescriptor;
}

int TKinokoShellConnector::ProcessInput(void)
{
    if (_DataStart - _Buffer > _BufferSize / 2) {
	memcpy(_Buffer, _DataStart, _DataLength);
	_DataStart = _Buffer;
    }

    char* NextStart = _DataStart + _DataLength;
    size_t MaxSize = _BufferSize - (NextStart - _Buffer) - 1;

    int ReceivedLength = ReceiveInput(NextStart, MaxSize);
    if (ReceivedLength > 0) {
        _DataLength += ReceivedLength;
	_DataStart[_DataLength] = '\0';
    }

    return ReceivedLength;
}

char* TKinokoShellConnector::NextLine(void)
{
    if (_DataLength <= 0) {
	_Buffer[0] = '\0';
	return 0;
    }

    if (strchr(_DataStart, LineTerminator) == NULL) {
	return 0;
    }

    char* Start = _DataStart;
    char* End = strchr(_DataStart, LineTerminator);
    *End = '\0';
    _DataStart = End + 1;
    _DataLength -= _DataStart - Start;

    while (isspace(*_DataStart) || (*_DataStart == '\0')) {
	if (_DataLength <= 0) {
	    _DataLength = 0;
	    _DataStart = _Buffer;
	    break;
	}

	_DataStart++;
	_DataLength--;
    }

    return Start;
}



TKinokoSocketShellConnector::TKinokoSocketShellConnector(const string& HostName, int PortNumber)
{
    _HostName = HostName;
    _PortNumber = PortNumber;

    _Socket = 0;
}

TKinokoSocketShellConnector::~TKinokoSocketShellConnector()
{
    delete _Socket;
} 

int TKinokoSocketShellConnector::Connect(void)
{
    try {
	_Socket = new TMushClientNetworkSocket(_HostName, _PortNumber);
	_Socket->Open();
    }
    catch (TSystemCallException &e) {
	delete _Socket;

	TMushServerNetworkSocket* ServerSocket = 0;
	try {
	    ServerSocket = new TMushServerNetworkSocket(_PortNumber);
	    ServerSocket->EnableLocalAddressReuse();
	    ServerSocket->Open();
	}
	catch (TSystemCallException &e) {
	    delete ServerSocket;
	    return -1;
	}

	_Socket = ServerSocket;
    }

    _FileDescriptor = _Socket->FileDescriptor();

    return _FileDescriptor;
}

int TKinokoSocketShellConnector::ReceiveInput(void* Buffer, int MaxSize)
{
    return _Socket->Receive(Buffer, MaxSize);
}

int TKinokoSocketShellConnector::SendMessage(void* Message, int Size)
{
    return _Socket->Send(Message, Size);
}



TKinokoClientSocketShellConnector::TKinokoClientSocketShellConnector(const string& HostName, int PortNumber)
: TKinokoSocketShellConnector(HostName, PortNumber)
{
}

TKinokoClientSocketShellConnector::~TKinokoClientSocketShellConnector()
{
}

int TKinokoClientSocketShellConnector::Connect(void)
{
    while (1) {
	try {
	    _Socket = new TMushClientNetworkSocket(_HostName, _PortNumber);
	    _Socket->Open();
	}
	catch (TSystemCallException &e) {
	    TMushRealTimeTimer(1, 0).Suspend();
	    delete _Socket;
	    continue;
	}
	break;
    }

    _FileDescriptor = _Socket->FileDescriptor();

    return _FileDescriptor;
}



TKinokoConsoleShellConnector::TKinokoConsoleShellConnector(void)
{
}

TKinokoConsoleShellConnector::~TKinokoConsoleShellConnector()
{
} 

int TKinokoConsoleShellConnector::Connect(void)
{
    _FileDescriptor = 1;

    return _FileDescriptor;
}

int TKinokoConsoleShellConnector::ReceiveInput(void* Buffer, int MaxSize)
{
    cin.getline((char*) Buffer, (size_t) MaxSize, '\n');
    int ReadSize = cin.gcount();
    ((char*) Buffer)[ReadSize++] = '\n';

    return ReadSize;
}

int TKinokoConsoleShellConnector::SendMessage(void* Message, int Size)
{
    cout.write((char*) Message, Size);

    return Size;
}
