/* KinokoCanvasGtk.cc */
/* Created by Enomoto Sanshiro on 9 July 2000. */
/* Last updated by Enomoto Sanshiro on 1 October 2001. */


#include <gtk/gtk.h>
#include <gdk_imlib.h>
#include "KinokoCanvas.hh"
#include "KinokoCanvasImageAreaGtk.hh"
#include "KinokoCanvasGtk.hh"
#include "KinokoShellFileSelectDialogGtk.hh"
#include "KinokoShellAboutDialogGtk.hh"
#include "KinokoShellPopupWindowGtk.hh"

using namespace std;


static TKinokoCanvasGtk* g_Canvas = 0;
static gint g_InputTag;

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data);
static gint expose_event_cb(GtkWidget* Widget, GdkEventExpose* Event);
static gint button_press_event_cb(GtkWidget* Widget, GdkEventButton* Event);
static void context_menu_cb(GtkWidget* Widget, gpointer Data);
static void input_event_cb(gpointer Data, gint File, GdkInputCondition Condition);
static void menu_save_image_cb(void);
static void menu_quit_cb(void);
static void menu_about_cb(void);


static GtkItemFactoryEntry MenuItemList[] = {
    {"/_File",              (char*) NULL, 0, 0, "<Branch>"},
    {"/File/_Save Image...", (char*) NULL, (GtkItemFactoryCallback) menu_save_image_cb, 0},
    {"/File/",              (char*) NULL, 0, 0, "<Separator>"},
    {"/File/e_Xit",         "<control>X", (GtkItemFactoryCallback) menu_quit_cb,     0},
    {"/_Help",              (char*) NULL, 0, 0, "<Branch>"},
    {"/Help/_About...",     (char*) NULL, (GtkItemFactoryCallback) menu_about_cb, 0},
};
static gint NumberOfMenuItems = sizeof(MenuItemList) / sizeof(MenuItemList[0]);



TKinokoCanvasGtk::TKinokoCanvasGtk(TKinokoShellConnector* ShellConnector, int argc, char** argv, int Width, int Height, int Left, int Top, bool EnableEps)
: TKinokoCanvas(ShellConnector)
{
    _Width = Width;
    _Height = Height;
    _Left = Left;
    _Top = Top;
    _IsEpsEnabled = EnableEps;

    _RootWindow = 0;
    _ImageArea = 0;

    _CurrentContextMenuObject = 0;
    _CurrentContextMenu = 0;

    gtk_init(&argc, &argv);
}

TKinokoCanvasGtk::~TKinokoCanvasGtk()
{
    delete _ImageArea;
}

TKinokoShellPopupWindow* TKinokoCanvasGtk::CreatePopupWindow(void)
{
    return new TKinokoShellPopupWindowGtk();
}

TKinokoShellFileSelectDialog* TKinokoCanvasGtk::CreateFileSelectDialog(void)
{
    return new TKinokoShellFileSelectDialogGtk();
}

void TKinokoCanvasGtk::Start(void)
{
    if (g_Canvas == 0) {
	g_Canvas = this;
	BuildRootWindow();
	
	g_InputTag = gdk_input_add(
	    _ShellConnector->FileDescriptor(),
	    (GdkInputCondition) (GDK_INPUT_READ | GDK_INPUT_EXCEPTION), 
	    input_event_cb, NULL
	);
	
	_ImageArea = new TKinokoCanvasImageAreaGtk(_DrawingAreaWidget);

	Initialize(_ImageArea, _IsEpsEnabled);
    }

    Clear();

    gtk_main();
}

void TKinokoCanvasGtk::Quit(void)
{
    gtk_main_quit();
}

void TKinokoCanvasGtk::BuildRootWindow(void)
{
    _RootWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);

    // Menu and Menubar //
    GtkAccelGroup* AccelGroup = gtk_accel_group_new();
    GtkItemFactory* ItemFactory = gtk_item_factory_new(
	GTK_TYPE_MENU_BAR, "<menu>", AccelGroup
    );
    gtk_item_factory_create_items(
	ItemFactory, NumberOfMenuItems, MenuItemList, NULL
    );
    gtk_accel_group_attach(AccelGroup, GTK_OBJECT(_RootWindow));
    GtkWidget* MenuHandleBox = gtk_handle_box_new();
    GtkWidget* Menu = gtk_item_factory_get_widget(ItemFactory, "<menu>");
    gtk_container_add(GTK_CONTAINER(MenuHandleBox), Menu);

    // Notebook //
#if 0
    GtkWidget* Notebook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(Notebook), GTK_POS_TOP);
#endif
    
    // Image Frame //
    _DrawingAreaWidget = gtk_drawing_area_new();
    gtk_drawing_area_size(
	GTK_DRAWING_AREA(_DrawingAreaWidget), _Width, _Height
    );
#if 0
    GtkWidget* ImageFrame = gtk_frame_new("");
    GtkWidget* ImageFrameLabel = gtk_label_new("Images");
    gtk_notebook_append_page(
	GTK_NOTEBOOK(Notebook), ImageFrame, ImageFrameLabel
    ); 
    gtk_container_add(GTK_CONTAINER(ImageFrame), _DrawingAreaWidget);
#endif

    // Message Frame //
    _TextWidget = gtk_text_new(NULL, NULL);
    gtk_text_set_editable(GTK_TEXT(_TextWidget), FALSE);
#if 0
    GtkWidget* Scrollbar = gtk_vscrollbar_new(GTK_TEXT(_TextWidget)->vadj);
    GtkWidget* MessageFrame = gtk_frame_new("");
    GtkWidget* MessageFrameLabel = gtk_label_new("Messages");
    GtkWidget* MessageHBox = gtk_hbox_new(FALSE, 0);
    gtk_notebook_append_page(
	GTK_NOTEBOOK(Notebook), MessageFrame, MessageFrameLabel
    ); 
    gtk_box_pack_start(GTK_BOX(MessageHBox), _TextWidget, TRUE, TRUE, 0);
    gtk_box_pack_start(GTK_BOX(MessageHBox), Scrollbar, FALSE, FALSE, 0);
    gtk_container_add(GTK_CONTAINER(MessageFrame), MessageHBox);
#endif

    // Packing //
    GtkWidget* VBox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(_RootWindow), VBox);
#if 0
    gtk_box_pack_start(GTK_BOX(VBox), Notebook, TRUE, TRUE, 0);
#else
    gtk_box_pack_start(GTK_BOX(VBox), MenuHandleBox, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(VBox), _DrawingAreaWidget, TRUE, TRUE, 0);
#endif

    // Event Handler Connection //
    gtk_signal_connect(
	GTK_OBJECT(_RootWindow), "delete_event", 
	GTK_SIGNAL_FUNC(delete_event_cb), NULL
    );
    gtk_signal_connect(
	GTK_OBJECT(_DrawingAreaWidget), "expose_event", 
	GTK_SIGNAL_FUNC(expose_event_cb), NULL
    );
    gtk_signal_connect(
	GTK_OBJECT(_DrawingAreaWidget), "button_press_event", 
	GTK_SIGNAL_FUNC(button_press_event_cb), NULL
    );
    gtk_widget_set_events(
	_DrawingAreaWidget, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK
    );

    // Show //
    gtk_widget_show_all(_RootWindow);
    if ((_Top > 0) && (_Left > 0)) {
	gtk_window_reposition(GTK_WINDOW(_RootWindow), _Left, _Top);
    }
}

int TKinokoCanvasGtk::SetTitle(const string& Title)
{
    gtk_window_set_title(GTK_WINDOW(_RootWindow), Title.c_str());
    return 1;
}

int TKinokoCanvasGtk::OpenContextMenu(TKinokoCanvasObject* CanvasObject)
{
    string ObjectName = CanvasObject->Name();
    vector<string> ActionList = CanvasObject->ActionList();
    if (ActionList.empty()) {
	return 0;
    }

    GtkWidget* Menu = gtk_menu_new();
    gtk_menu_set_title(GTK_MENU(Menu), ObjectName.c_str());

    for (unsigned i = 0; i < ActionList.size(); i++) {
	string Label = ActionList[i];
	GtkWidget* MenuItem = gtk_menu_item_new_with_label(Label.c_str());
	gtk_menu_append(GTK_MENU(Menu), MenuItem);
	gtk_widget_show(MenuItem);

	char* ActionName = strdup(Label.c_str());
	_CurrentContextMenuActionNameList.push_back(ActionName);

	gtk_signal_connect(
	    GTK_OBJECT(MenuItem), 
	    "activate", GTK_SIGNAL_FUNC(context_menu_cb), ActionName
	);
    }
    
    _CurrentContextMenuObject = CanvasObject;
    _CurrentContextMenu = Menu;

    gtk_menu_popup(
	GTK_MENU(Menu), 
	NULL, NULL, NULL, NULL, 3, 0
    );

    return 1;
}

int TKinokoCanvasGtk::ProcessContextMenuAction(const char* ActionName)
{
    int Result = 0;

    if (_CurrentContextMenuObject != 0) {
	Result = _CurrentContextMenuObject->ProcessAction(ActionName);
	_CurrentContextMenuObject = 0;
    }

    //... BUG:
    // the following cleanups will not be not performed 
    // if any of menu items are not selected.

    if (_CurrentContextMenu != 0) {
	gtk_widget_unref(_CurrentContextMenu);
	_CurrentContextMenu = 0;
    }
    while (! _CurrentContextMenuActionNameList.empty()) {
	free(_CurrentContextMenuActionNameList.back());
	_CurrentContextMenuActionNameList.pop_back();
    }

    return Result;
}

static void context_menu_cb(GtkWidget* Widget, gpointer Data)
{
    if (g_Canvas) {
	g_Canvas->ProcessContextMenuAction((char*) Data);
    }
}

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data)
{
    bool Result = g_Canvas->TryToQuit();
    return Result ? FALSE : TRUE;
}

static gint expose_event_cb(GtkWidget* Widget, GdkEventExpose* Event)
{
    if (g_Canvas) {
	g_Canvas->Redraw();
    }

    return FALSE;
}

static gint button_press_event_cb(GtkWidget* Widget, GdkEventButton* Event)
{
    if (g_Canvas) {
	g_Canvas->OnClick(Event->button, (int) Event->x, (int) Event->y);
    }

    return TRUE;
}

static void input_event_cb(gpointer Data, gint File, GdkInputCondition Condition)
{
    if (g_Canvas == 0) {
	return;
    }

    int ProcessedLength = 0;
    if (Condition == GDK_INPUT_READ) {
	ProcessedLength = g_Canvas->ProcessInput();
    }
    
    if ((Condition == GDK_INPUT_EXCEPTION) || (ProcessedLength <= 0)) {
	gdk_input_remove(g_InputTag);
    }
}

static void menu_save_image_cb(void)
{
    TKinokoShellFileSelectDialogGtk FileSelectDialog("Save Image");

    string FileName = FileSelectDialog.Open();
    if (! FileName.empty()) {
	g_Canvas->SaveImageTo(FileName);
    }
}

static void menu_quit_cb(void)
{
    g_Canvas->TryToQuit();
}

static void menu_about_cb(void)
{
    TKinokoShellAboutDialogGtk AboutDialog;

    AboutDialog.SetTitle("Kinoko Canvas ver. 1.0");
    AboutDialog.SetCopyright("Copyright 2000-2002, Enomoto Sanshiro");
    AboutDialog.AddAuthor("Enomoto Sanshiro <sanshiro@awa.tohoku.ac.jp>");
    //AboutDialog.AddComment("The application makes you happy.");

    AboutDialog.Open();
}

