#! /usr/bin/wish
# tinykinoko-tcltk.tcl
# Created by Enomoto Sanshiro on 14 June 2000
# Last updated by Enomoto Sanshiro on 7 May 2001


set ProjectFileName {}
set EditorPath "vi"
set EditorOpts {}
set EditorNeedsTerminal 1
set PlotTool "ROOT"
set HistTool "ROOT"

set ReadoutScriptName {}
set DataFileName {}
set NumberOfEvents {}

set PlotScriptName {}
set HistScriptName {}

set PlotMin {}
set PlotMax {}
set HistMin {}
set HistMax {}
set HistNBin {}


font create TitleFont -family helvetica -size 14 -weight bold 

set LogoImageData {\
    R0lGODlhUAAgAPcAAP///8/gz0+MUABZAu/174+2kG+hcD+CQd/q3y94MR9t\
    Ia/LsA9jEV+XYJ/BoL/VwH+sgPXv79W/v+rf3+DPz8Gfn6x/f1kAAIxPT4I/\
    P20fH2MPD8uvr3gvL5dfX7aPj6Fvb/7+/qLCo5u8m5y8nJ68nJ67nJ+7nJ+6\
    nKC6nKC4nKC3nKK3nKK2nKO2nKS0nKS0m6a0m6azm6aym6eym6iwm6qwm6qv\
    m6qum6uum6ysm66sm66rm66qm6+qm7Com7Cnm7Knm7Omm7Okm7Skm7ajm7ai\
    m7eim7egm7igm7qgm7qfm7uem7ycm72dnP3+/RFiEgFXAQNWAgNVAgZUAgZT\
    AgZSAglRAgtPAgtOAgxOAg5KAg9JAhFIAhFGAhJGAhJFAhVEAhdCAhhCARlA\
    ARpAARs+ARw9AR07AR46ASA5ASE4ASI3ASM2ASQ1ASU0ASUyAScwASgwASgv\
    ASstASwtASwsAS0rAS8rAS8oATAmATIlATQkATYjATciATchATggATofATwd\
    ATwcAT0bAT4YAUAZAUIXAUMWAEUUAEYSAEcSAEgRAEgPAEoPAEsOAE0MAE4M\
    AE8KAFEIAFIHAFQFAFUEAFYDAFgEAv7//m2ebWeaZ2iaZ2iZZ2mZZ2qYZ2qW\
    Z2yWZ2yVZ22VZ26TZ2+SZ3CQZ3GQZ3OPZ3SPZ3WMZ3aMZ3aLZ3eLZ3eKZ3iJ\
    Z3qJZ3mIZ3uHZ3yHZ3uGZ3yFZ3yEZ36DZ4CBZ4GBZ4GAZ4N/Z4N+Z4N9Z4R9\
    Z4V7Z4d8Z4d7Z4d6Z4h6Z4l5Z4t3Z4t2Z4t0Z4x1Z490Z490ZpBzZpBxZpFx\
    ZpJvZpNvZpNuZpRtZpVtZpVsZpZrZpdrZploZppoZptpZ5hkY///////////\
    ////////////////////////////////////////////////////////////\
    ////////////////////////////////////////////////////////////\
    /////////////////////yH5BAEAANoALAAAAABQACAAAAj/ALUJHEiwoMGD\
    CBMqXMiwocOHECNKnEixosWLGDNKFDBAo8eKBQwIPNDxo0mICQYU0EbypEtt\
    AxQUFHBAwEAFAxa0fGlSgUhtBhQITXlgIAMGOHliLCBzYAMH2hoMSCAwqcAF\
    A7IqvVhgQIMFAqdqw1pUW9eVAruW3ErRQdamKQUyWBuTYF22FA/gBGt2gEip\
    ULXFraoSb0W3DQbeHVBWW2Czhi8OYDBwpwKqkU1K5esWgtIKFixYBC1aYcy1\
    BS+UHojhQgZtGVTH1mDQNesLrjVc2MBhoOqCrTsQbJ17d8ICnhH+HtjhggeB\
    HETHvoAh9Wttuldra/5B4HKBzZ8P/8xOkLvE7xsuVDAYu/l6369BfB9/wfvq\
    9N1vaxe4YcN50RUYR1AFr8WmTX8E2XYBbQYFWN1vAfKWGoMFfUBdRKp5gNt+\
    FtRnoIXCeZcBB7YdZJxq8s2njYPKUejQbtQhSFCHsNWnTWuludbhfr7VB2MG\
    uhW0o3I2PrShdy7SaCB/NrpmoXi1+XgBCEjOOCWRGFIpkHzVCaRkkSTSpuB1\
    tQl3pZcliqgcmS9q11xpXxLEpW3pHdQalfO9OVCdBuGI4X58xlkebq+RGOJA\
    AYaoYnq9rXjBoQIl+l9BEWojaIIlyrcBgBpSqCKJ/m25G4DBTZTBe3JmAAKB\
    2oDApkAfqP86nG6PaneqQa5qKRAGtGbAowgjkEBCCSaYcAIKKKSgggorsNDC\
    sy68AAMMMcggwww00FDDtjbccAMOOeSgw7g67MBDDz6k+8O6QAQhxLtDEEFE\
    EfQaccQRSCSRhBJKLMHEv00E3IQTAkERhRRTUFGFFVdgkYUWW2zBRRdefAFG\
    GGKMQUYZZpyBRhpqrMFGG268AUcccsxBRx123IFHHnrswQcfffjxByCBCDII\
    IYUYcggiiCSiyCKMNOLII5BEIskkk1BSiSWXXIJJwQcnvHDDD0c8ccUXZ7xx\
    xx+HPHLJJ6e8cssvxzxzzTfnvHPPPwc9dNFHJ710009HPbU2mmxgwkknnnwC\
    SiiiiDIKKaSUUoopp5yCSiqpqLIKK6248gosscgyCy212HLL57jkkosuu/DS\
    iy+//AJMMMIMQ0wxxRhzDDLJKKPMMsw044wzz0ATjTTTUFNNNdZYcw022QQE\
    ADs=\
}


proc OnStartup {} {
    LoadProjectFrom "last.tkprj"
}


proc OnShutdown {} {
    SaveProjectTo "last.tkprj"
}


proc ClearProject {} {
    global ProjectFileName
    global EditorPath EditorOpts EditorNeedsTerminal
    global PlotTool HistTool
    global ReadoutScriptName DataFileName NumberOfEvents
    global PlotScriptName HistScriptName
    global PlotMin PlotMax HistMin HistMax HistNBin

    set ProjectFileName {untitled}
    set EditorPath "vi"
    set EditorOpts {}
    set EditorNeedsTerminal 1
    set PlotTool "ROOT"
    set HistTool "ROOT"

    set ReadoutScriptName {}
    set DataFileName {}
    set NumberOfEvents {}

    set PlotScriptName {}
    set HistScriptName {}

    set PlotMin {}
    set PlotMax {}
    set HistMin {}
    set HistMax {}
    set HistNBin {}
}


proc SaveProjectTo { FileName } {
    global ProjectFileName
    global EditorPath EditorOpts EditorNeedsTerminal
    global PlotTool HistTool
    global ReadoutScriptName DataFileName NumberOfEvents
    global PlotScriptName HistScriptName
    global PlotMin PlotMax HistMin HistMax HistNBin

    WriteMessage "Saving to $FileName..."
    set File [ open $FileName w ]
    
    puts $File "This is a tinyKinoko Ver 0.1.1 project file."
    puts $File $ProjectFileName
    puts $File $EditorPath
    puts $File $EditorOpts
    puts $File $EditorNeedsTerminal
    puts $File $PlotTool
    puts $File $HistTool

    puts $File $ReadoutScriptName
    puts $File $DataFileName
    puts $File $NumberOfEvents

    puts $File $PlotScriptName
    puts $File $HistScriptName

    puts $File $PlotMin
    puts $File $PlotMax
    puts $File $HistMin
    puts $File $HistMax
    puts $File $HistNBin

    close $File

    WriteMessage "Done."
    WriteMessage ""
}


proc LoadProjectFrom { FileName } {
    global ProjectFileName
    global EditorPath EditorOpts EditorNeedsTerminal
    global PlotTool HistTool
    global ReadoutScriptName DataFileName NumberOfEvents
    global PlotScriptName HistScriptName
    global PlotMin PlotMax HistMin HistMax HistNBin

    set Result [ catch {set File [ open $FileName r ] } ]
    if { $Result != 0 } {
	return Result
    }

    gets $File Magic
    if { $Magic != "This is a tinyKinoko Ver 0.1.1 project file." } {
	set Message "Bad project file: $FileName"
	tk_messageBox -type ok -message $Message
	return 1
    }

    gets $File ProjectFileName
    gets $File EditorPath
    gets $File EditorOpts
    gets $File EditorNeedsTerminal
    gets $File PlotTool
    gets $File HistTool

    gets $File ReadoutScriptName
    gets $File DataFileName
    gets $File NumberOfEvents

    gets $File PlotScriptName
    gets $File HistScriptName

    gets $File PlotMin
    gets $File PlotMax
    gets $File HistMin
    gets $File HistMax
    gets $File HistNBin

    close $File
}


proc StartAcquisition { ReadoutScriptName DataFileName NumberOfEvents } {
    if { $ReadoutScriptName == "" } {
	set Message "Script File is not specified."
	tk_messageBox -icon error -type ok -message $Message
	return
    }
    if { $DataFileName == "" } {
	set Message "Data File is not specified."
	tk_messageBox -icon error -type ok -message $Message
	return
    }

#    if {! [string match "/*" $ReadoutScriptName ] } { 
#	set ReadoutScriptName "[pwd]/$ReadoutScriptName"
#    }
#    if {! [string match "/*" $DataFileName ] } { 
#	set DataFileName "[pwd]/$DataFileName"
#    }

    if { ! [ file exists $ReadoutScriptName ] } {
	set Message "file \"$ReadoutScriptName\" was not found."
	tk_messageBox -icon error -type ok -message $Message
	return
    }
    if [ file exists $DataFileName ] {
	set Message  "file \"$DataFileName\" already exists.\noverwrite?"
	set Choice [ tk_messageBox -icon warning -type yesno -message $Message ]
	if { $Choice == "no"} {
	    return
	}
    }

    if { $NumberOfEvents != "" } {
	exec xterm -e tinykinoko -f $ReadoutScriptName $DataFileName $NumberOfEvents &
    } else {
	exec xterm -e tinykinoko -f $ReadoutScriptName $DataFileName &
    }
}


proc StopAcquisition {} {
    if { [ catch { exec killall tinykinoko } ] != 0 } {
	tk_messageBox -type ok -message "Type Ctrl-C on tinykinoko window"
    }
}


proc EditFile { FileName } {
    global EditorPath EditorOpts EditorNeedsTerminal

    if { $EditorNeedsTerminal } {
	if { $EditorOpts != "" } {
	    exec xterm -e $EditorPath $EditorOpts $FileName &
	} else {
	    exec xterm -e $EditorPath $FileName &
	}
    } else {
	if { $EditorOpts != "" } {
	    exec $EditorPath $EditorOpts $FileName &
	} else {
	    exec $EditorPath $FileName &
	}
    }
}

proc CheckScriptFile { FileName } {
    set Message [ exec ktscheck $FileName ]
    if { $Message == "" } {
	set Message "no error was found."
    }

    tk_messageBox -icon info -type ok -message $Message
}

proc GeneratePlotScript { ScriptFileName DataFileName Min Max } {
    global PlotTool

    regsub {\.[A-Za-z0-9]*$} $DataFileName {} DataName
    regsub -all -- {-} $DataName {_} DataName
    
    if { $ScriptFileName == "" } {
	if { $PlotTool == "ROOT" } {
	    set ScriptFileName "${DataName}_plot.C"
	} elseif { $PlotTool == "GNUPLOT" } {
	    set ScriptFileName "${DataName}_plot.gp"
	}	
    }

    if [ file exists $ScriptFileName ] {
	set Message  "file \"$ScriptFileName\" already exists\noverwrite?"
	set Choice [ tk_messageBox -icon warning -type yesno -message $Message ]
	if { $Choice == "no"} {
	    return $ScriptFileName
	}
    }

    if { $Min == "" || $Max == "" } {
	tk_messageBox -icon error -type ok -message "invalid plot range"
	return $ScriptFileName
    }

    exec kdftable $DataFileName > "${DataName}.dat"

    if { $PlotTool == "ROOT" } {
	exec genscript-root-plot $ScriptFileName "${DataName}.dat" $Min $Max
    } elseif { $PlotTool == "GNUPLOT" } {
	exec genscript-gnuplot-plot $ScriptFileName "${DataName}.dat" $Min $Max
    }

    return $ScriptFileName
}

proc DrawPlot { ScriptFileName DataFileName } {
    global PlotTool
    
    if { ! [ file exists $ScriptFileName ]} {
	tk_messageBox -icon error -type ok -message "invalid script file"
	return
    }

    if { $PlotTool == "ROOT" } {
	exec xterm -e root $ScriptFileName - &
    } elseif { $PlotTool == "GNUPLOT" } {
	exec xterm -e gnuplot $ScriptFileName - &
    }
}

proc GenerateHistogramScript { ScriptFileName DataFileName Min Max NBin } {
    global HistTool

    regsub {\.[A-Za-z0-9]*$} $DataFileName {} DataName
    regsub -all -- {-} $DataName {_} DataName

    if { $ScriptFileName == "" } {
	if { $HistTool == "ROOT" } {
	    set ScriptFileName "${DataName}_hist.C"
	} elseif { $HistTool == "GNUPLOT" } {
	    set ScriptFileName "${DataName}_hist.gp"
	}	
    }

    if [ file exists $ScriptFileName ] {
	set Message  "file \"$ScriptFileName\" already exists\noverwrite?"
	set Choice [ tk_messageBox -icon warning -type yesno -message $Message ]
	if { $Choice == "no"} {
	    return $ScriptFileName
	}
    }

    if { $Min == "" || $Max == "" || $NBin == "" } {
	tk_messageBox -icon error -type ok -message "invalid histogram range"
	return $ScriptFileName
    }

    exec kdftable $DataFileName > "${DataName}.dat"
    
    if { $HistTool == "ROOT" } {
	exec genscript-root-hist $ScriptFileName "${DataName}.dat" $Min $Max $NBin
    } elseif { $HistTool == "GNUPLOT" } {
	exec genscript-gnuplot-hist $ScriptFileName "${DataName}.dat"
    }

    return $ScriptFileName
}

proc DrawHistogram { ScriptFileName DataFileName Min Max NBin } {
    global HistTool

    if { ! [ file exists $ScriptFileName ]} {
	tk_messageBox -icon error -type ok -message "invalid script file: $ScriptFileName"
	return
    }

    if { $Min == "" || $Max == "" || $NBin == "" } {
	tk_messageBox -icon error -type ok -message "invalid histogram range"
	return
    }

    regsub {\.[A-Za-z0-9]*$} $DataFileName {} DataName
    exec kdftable $DataFileName > "${DataName}.dat"
    
    if { $HistTool == "ROOT" } {
	exec xterm -e root $ScriptFileName - &
    } elseif { $HistTool == "GNUPLOT" } {
	exec makehist "${DataName}.dat" $Min $Max $NBin > "${DataName}_hist.dat"
	exec xterm -e gnuplot $ScriptFileName - &
    }
}

proc SetPreferences {} {
    global EditorPath EditorOpts EditorNeedsTerminal
    global PlotTool HistTool

    global _EditorPath _EditorOpts _EditorNeedsTerminal
    global _PlotTool _HistTool

    set _EditorPath $EditorPath
    set _EditorOpts $EditorOpts
    set _EditorNeedsTerminal $EditorNeedsTerminal
    set _PlotTool $PlotTool
    set _HistTool $HistTool

    toplevel .dialog
    wm title .dialog "Preferences"

    frame .dialog.editorFrame -borderwidth 2 -relief groove
    label .dialog.editorFrameLabel -text "Editor" -fg "#060"
    label .dialog.editorPathLabel -text "Path"
    entry .dialog.editorPathEntry -textvariable _EditorPath -bg white -width 32
    label .dialog.editorOptsLabel -text "Options"
    entry .dialog.editorOptsEntry -textvariable _EditorOpts -bg white -width 32
    checkbutton .dialog.needsTermButton -text "open in terminal window" -variable _EditorNeedsTerminal
    grid .dialog.editorPathLabel .dialog.editorPathEntry -in .dialog.editorFrame
    grid .dialog.editorOptsLabel .dialog.editorOptsEntry -in .dialog.editorFrame
    grid x .dialog.needsTermButton -in .dialog.editorFrame -sticky w

    frame .dialog.plotFrame -borderwidth 2 -relief groove
    label .dialog.plotFrameLabel -text "Plot" -fg "#060"
    radiobutton .dialog.rt4PlotButton -text "ROOT" -variable _PlotTool -value "ROOT"
    radiobutton .dialog.gp4PlotButton -text "GNUPLOT" -variable _PlotTool -value "GNUPLOT"
    grid .dialog.rt4PlotButton -in .dialog.plotFrame -sticky w
    grid .dialog.gp4PlotButton -in .dialog.plotFrame -sticky w

    frame .dialog.histFrame -borderwidth 2 -relief groove
    label .dialog.histFrameLabel -text "Histogram" -fg "#060"
    radiobutton .dialog.rt4HistButton -text "ROOT" -variable _HistTool -value "ROOT"
    radiobutton .dialog.gp4HistButton -text "GNUPLOT" -variable _HistTool -value "GNUPLOT"
    grid .dialog.rt4HistButton -in .dialog.histFrame -sticky w
    grid .dialog.gp4HistButton -in .dialog.histFrame -sticky w

    frame .dialog.buttonFrame
    button .dialog.okButton -text "Ok" -command { 
	set EditorPath $_EditorPath
	set EditorOpts $_EditorOpts
	set EditorNeedsTerminal $_EditorNeedsTerminal
	set PlotTool $_PlotTool
	set HistTool $_HistTool

	destroy .dialog 
    }
    button .dialog.cancelButton -text "Cancel" -command { destroy .dialog }
    pack .dialog.okButton .dialog.cancelButton  -in .dialog.buttonFrame -side left

    grid .dialog.editorFrameLabel -sticky w -columnspan 2 -pady 5
    grid .dialog.editorFrame -sticky w -columnspan 2 -padx 10
    grid .dialog.plotFrameLabel .dialog.histFrameLabel -sticky w -pady 5
    grid .dialog.plotFrame .dialog.histFrame -sticky w -padx 10
    grid .dialog.buttonFrame -sticky e -columnspan 2 -pady 15 -padx 10
}


proc CreateMenu { Frame } {
    menubutton .fileMenuButton -text "File" -menu .fileMenuButton.fileMenu
    menu .fileMenuButton.fileMenu -tearoff false

    .fileMenuButton.fileMenu add command -label "New Project" -command {
	ClearProject
    } -underline 0

    .fileMenuButton.fileMenu add command -label "Open..." -command {
	set SelectedProjectFileName [ tk_getOpenFile -filetypes {
	    { "tinyKinoko Project" .tkprj }
	    { "All Files" * }
	}] 
	if { $SelectedProjectFileName != "" } {
	    global ProjectFileName
	    set ProjectFileName $SelectedProjectFileName
	    LoadProjectFrom $ProjectFileName
	}
    } -underline 0

    .fileMenuButton.fileMenu add separator

    .fileMenuButton.fileMenu add command -label "Save" -command {
	global ProjectFileName
	SaveProjectTo $ProjectFileName
    } -accelerator "C-s" -underline 0
    bind . <Control-s> {
	global ProjectFileName
	SaveProjectTo $ProjectFileName
    }

    .fileMenuButton.fileMenu add command -label "Save As..." -command {
	set SelectedProjectFileName [ tk_getSaveFile -filetypes {
	    { "tinyKinoko Project" .tkprj }
	    { "All Files" * }
	}] 
	if { $SelectedProjectFileName != "" } {
	    global ProjectFileName
	    set ProjectFileName $SelectedProjectFileName
	    SaveProjectTo $ProjectFileName
	}
    } -underline 5
    .fileMenuButton.fileMenu add separator
    .fileMenuButton.fileMenu add command -label "eXit" -command {
	OnShutdown
	exit
    } -accelerator "C-x" -underline 1
    bind . <Control-x> {
	OnShutdown
	exit
    }

    menubutton .optMenuButton -text "Options" -menu .optMenuButton.optMenu
    menu .optMenuButton.optMenu -tearoff false

    .optMenuButton.optMenu add command -label "Preferences..." -command {
	SetPreferences
    } -underline 0

    menubutton .helpMenuButton -text "Help" -menu .helpMenuButton.helpMenu
    menu .helpMenuButton.helpMenu -tearoff false

    .helpMenuButton.helpMenu add command -label "About..." -command {
	set ProductName "tinyKinoko Ver. 0.1.0";
	set Copyright "Created by Enomoto Sanshiro on 7 May 2001."
	tk_messageBox -type ok -message "$ProductName\n$Copyright";
    } -underline 0

    pack .fileMenuButton .optMenuButton -in $Frame -side left
    pack .helpMenuButton -in $Frame -side right
}


proc CreateMessageFrame { ParentFrame ListBox } {
    global LogoImageData

    frame .logoFrame
    label .logoSpacer -text ""
    pack .logoSpacer .logoFrame -fill x -side top -in $ParentFrame

    set LogoImage [ image create photo -data $LogoImageData -height 40 -width 90 ]

    label .logoLabel -image $LogoImage
    pack  .logoLabel -side right -in .logoFrame

    $ListBox config -height 8 -width 100
    $ListBox config -yscrollcommand { .vbar set }
    $ListBox config -xscrollcommand { .hbar set }
    scrollbar .vbar -orient vertical -command "$ListBox yview"
    scrollbar .hbar -orient horizontal -command "$ListBox xview"
    
    pack $ListBox -side left -in $ParentFrame
    pack .vbar -side left -fill y -in $ParentFrame
}


proc WriteMessage { Message } {
    .messageListBox insert end $Message
}

proc CreateConfigFrame { ParentFrame } {
    global ReadoutScriptName DataFileName NumberOfEvents

    frame .configFrame
    label .configLabel -text "Configuration" -fg "#060" -font TitleFont
    pack .configLabel .configFrame -in $ParentFrame -fill x -pady 10
    
    label .scriptFileLabel -text "Readout Script (.kts)"
    entry .scriptFileEntry -textvariable ReadoutScriptName -width 20 -bg white
    button .scriptFileSelectButton -text "select..." -command { 
	set SelectedReadoutScriptName [tk_getOpenFile -filetypes {
	    { ".kts Scripts" .kts }
	    { "All Files" * }
	}] 
	if { $SelectedReadoutScriptName != "" } {
	    set ReadoutScriptName $SelectedReadoutScriptName
	    if { [ file dirname $ReadoutScriptName ] == [ pwd ] } {
		set ReadoutScriptName [ file tail $ReadoutScriptName ]
	    }
	}
    }
    grid configure .scriptFileLabel .scriptFileEntry .scriptFileSelectButton -in .configFrame
    
    label .dataFileLabel -text "Data File (.kdf)"
    entry .dataFileEntry -textvariable DataFileName -width 20 -bg white
    button .dataFileSelectButton -text "select..." -command {
	set SelectedDataFileName [tk_getOpenFile -filetypes {
	    { ".kdf Files" .kdf }
	    { "All Files" * }
	}] 
	if { $SelectedDataFileName != "" } {
	    set DataFileName $SelectedDataFileName
	    if { [ file dirname $DataFileName ] == [ pwd ] } {
		set DataFileName [ file tail $DataFileName ]
	    }
	}
    }
    grid configure .dataFileLabel .dataFileEntry .dataFileSelectButton -in .configFrame
    
    label .numEventLabel -text "# of events (optional)"
    entry .numEventEntry -textvariable NumberOfEvents -width 20 -bg white
    grid configure .numEventLabel .numEventEntry -in .configFrame
    
    frame .buttonFrame
    pack .buttonFrame -fill x -in $ParentFrame -pady 8

    button .editScriptButton -text "Edit Script..." -command { 
	if { $ReadoutScriptName == "" } {
	    set Message "Script File is not specified."
	    tk_messageBox -icon error -type ok -message $Message
	} else {
	    EditFile $ReadoutScriptName
	}
    }
    button .checkScriptButton -text "Check Syntax..." -command { 
	if { $ReadoutScriptName == "" } {
	    set Message "Script File is not specified."
	    tk_messageBox -icon error -type ok -message $Message
	} else {
	    CheckScriptFile $ReadoutScriptName
	}
    }

    pack .checkScriptButton .editScriptButton -side right -in .buttonFrame
}


proc CreateAcquisitionFrame { ParentFrame } {
    global ReadoutScriptName DataFileName NumberOfEvents

    frame .acquisitionFrame

    label .acquisitionLabel -text "Acquisition" -fg "#060" -font TitleFont
    pack .acquisitionLabel .acquisitionFrame -in $ParentFrame -side top -fill x -pady 10

    button .startButton -text "Start" -command {
	StartAcquisition $ReadoutScriptName $DataFileName $NumberOfEvents
    }
    button .stopButton -text "Stop" -command {
	StopAcquisition
    }

    pack .stopButton .startButton -side right -in .acquisitionFrame
}


proc CreateDumpFrame { ParentFrame } {
    global ReadoutScriptName DataFileName NumberOfEvents

    frame .dumpFrame

    label .dumpLabel -text "Dump" -fg "#060" -font TitleFont
    pack .dumpLabel .dumpFrame -in $ParentFrame -side top -fill x -pady 10
    
    button .dumpButton -text "Dump..." -command {
	exec kdfdump $DataFileName - > /tmp/tinykinoko-tcl.tmp
	EditFile /tmp/tinykinoko-tcl.tmp
    }

    button .summaryButton -text "Summary..." -command {
	exec kdfprofile $DataFileName > /tmp/tinykinoko-tcl.tmp
	EditFile /tmp/tinykinoko-tcl.tmp
    }

    pack .summaryButton -side right -in .dumpFrame
    pack .dumpButton -side right -in .dumpFrame
}


proc CreatePlotFrame { ParentFrame } {
    global ReadoutScriptName DataFileName NumberOfEvents
    global PlotScriptName
    global PlotMin PlotMax

    frame .plotScriptFrame
    frame .plotRangeFrame
    frame .plotButtonFrame

    label .plotLabel -text "Plot" -fg "#060" -font TitleFont
    pack .plotLabel .plotScriptFrame .plotRangeFrame .plotButtonFrame -in $ParentFrame -side top -fill x -pady 10
    
    label .plotScriptLabel -text "Script File"
    entry .plotScriptEntry -textvariable PlotScriptName -width 20 -bg white
    button .plotScriptButton -text "select..." -command {
	set SelectedPlotScriptName [ tk_getOpenFile ] 
	if { $SelectedPlotScriptName != "" } {
	    set PlotScriptName $SelectedPlotScriptName
	    if { [ file dirname $PlotScriptName ] == [ pwd ] } {
		set PlotScriptName [ file tail $PlotScriptName ]
	    }
	}
    }

    pack .plotScriptLabel .plotScriptEntry .plotScriptButton -side left -in .plotScriptFrame

    label .plotMinLabel -text "Min"
    entry .plotMinEntry -textvariable PlotMin -width 8 -bg white -justify right
    label .plotMaxLabel -text "  Max"
    entry .plotMaxEntry -textvariable PlotMax -width 8 -bg white -justify right

    pack .plotMinLabel .plotMinEntry .plotMaxLabel .plotMaxEntry -side left -in .plotRangeFrame

    button .plotGenScriptButton -text "Generate Script" -command {
	if { $PlotMin == "" } {
	    set PlotMin 0
	}
	if { $PlotMax == "" } {
	    set PlotMax $NumberOfEvents
	}
	set PlotScriptName [ GeneratePlotScript $PlotScriptName $DataFileName $PlotMin $PlotMax ]
    }
    button .plotEditScriptButton -text "Edit Script..." -command {
	EditFile $PlotScriptName
    }
    button .plotDrawButton -text "Draw" -command {
	DrawPlot $PlotScriptName $DataFileName
    }

    pack .plotDrawButton .plotEditScriptButton .plotGenScriptButton -side right -in .plotButtonFrame
}


proc CreateHistFrame { ParentFrame } {
    global ReadoutScriptName DataFileName NumberOfEvents
    global HistScriptName
    global HistMin HistMax HistNBin

    label .histLabel -text "Histogram" -fg "#060" -font TitleFont
    
    frame .histScriptFrame
    frame .histRangeFrame
    frame .histButtonFrame
    pack .histLabel .histScriptFrame .histRangeFrame .histButtonFrame -in $ParentFrame -side top -fill x -pady 10

    label .histScriptLabel -text "Script File"
    entry .histScriptEntry -textvariable HistScriptName -width 20 -bg white
    button .histScriptButton -text "select..." -command {
	set SelectedHistScriptName [ tk_getOpenFile ] 
	if { $SelectedHistScriptName != "" } {
	    set HistScriptName $SelectedHistScriptName
	    if { [ file dirname $HistScriptName ] == [ pwd ] } {
		set HistScriptName [ file tail $HistScriptName ]
	    }
	}
    }

    pack .histScriptLabel .histScriptEntry .histScriptButton -side left -in .histScriptFrame

    label .histMinLabel -text "Min"
    entry .histMinEntry -textvariable HistMin -width 8 -bg white -justify right
    label .histMaxLabel -text "  Max"
    entry .histMaxEntry -textvariable HistMax -width 8 -bg white -justify right
    label .histNBinLabel -text "  Bins"
    entry .histNBinEntry -textvariable HistNBin -width 8 -bg white -justify right

    pack .histMinLabel .histMinEntry .histMaxLabel .histMaxEntry .histNBinLabel .histNBinEntry -side left -in .histRangeFrame

    button .histGenScriptButton -text "Generate Script" -command {
	set HistScriptName [ GenerateHistogramScript $HistScriptName $DataFileName $HistMin $HistMax $HistNBin ]
    }
    button .histEditScriptButton -text "Edit Script..." -command {
	EditFile $HistScriptName
    }
    button .histDrawButton -text "Draw" -command {
	DrawHistogram $HistScriptName $DataFileName $HistMin $HistMax $HistNBin
    }

    pack .histDrawButton .histEditScriptButton .histGenScriptButton -side right -in .histButtonFrame
}



OnStartup

set MenuFrame [ frame .memuFrame  -borderwidth 2 -relief raised ]
set MainFrame [ frame .mainFrame ]
set LeftFrame [ frame .leftFrame ]
set RightFrame [ frame .rightFrame ]
set MessageFrame [ frame .messageFrame ]
set MessageListBox [ listbox .messageListBox -bg white ]

pack $MenuFrame -fill x
pack $MainFrame -fill x
pack $MessageFrame
pack $LeftFrame -side left -padx 15 -in $MainFrame
pack $RightFrame -side left -padx 15 -in $MainFrame

CreateMenu $MenuFrame
CreateConfigFrame $LeftFrame
CreateAcquisitionFrame $LeftFrame
CreateDumpFrame $LeftFrame
CreatePlotFrame $RightFrame
CreateHistFrame $RightFrame
CreateMessageFrame $MessageFrame $MessageListBox

WriteMessage "Welcome to the Kinoko world."
WriteMessage ""
